/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.impl;

import com.terracotta.management.security.DuckTypedSecretProviderBackend;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.terracotta.management.keychain.FileStoreKeyChain;
import com.terracotta.management.keychain.KeyName;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.KeychainInitializationException;
import com.terracotta.management.security.ObfuscatingSecretProviderBackend;
import com.terracotta.management.security.SecretProvider;
import com.terracotta.management.security.SecretProviderBackEnd;

import java.net.MalformedURLException;
import java.net.URL;

/**
 * @author brandony
 * @author Ludovic Orban
 */
public final class SecretFileStoreKeyChainAccessor implements KeyChainAccessor {
  private static final Logger LOG = LoggerFactory.getLogger(SecretFileStoreKeyChainAccessor.class);

  /**
   * A constant for the default keychain file location.
   */
  private static final String DFLT_KEYCHAIN_FILE_LOCATION = "file://" + System
      .getProperty("user.home") + "/.tc/mgmt/keychain";

  /**
   * A constant for the jvm keychain file location property.
   */
  private static final String JVM_KEYCHAIN_LOCATION_PROP = "com.tc.management.keychain.file";

  /*
   * Those are the two standard Terracotta system properties used by the L1 to get access to a keychain.
   */
  private static final String TERRACOTTA_KEYCHAIN_LOCATION_PROP = "com.tc.security.keychain.url";
  private static final String TERRACOTTA_CUSTOM_SECRET_PROVIDER_PROP = "com.terracotta." +
                                                                       "express.SecretProvider";

  private FileStoreKeyChain keyChain;

  public SecretFileStoreKeyChainAccessor() throws KeychainInitializationException {
    URL kcLoc;
    SecretProviderBackEnd secretProviderBackend;

    String keychainUrl = System.getProperty(TERRACOTTA_KEYCHAIN_LOCATION_PROP);
    String secretProviderClassname = System.getProperty(TERRACOTTA_CUSTOM_SECRET_PROVIDER_PROP);
    if (keychainUrl != null && secretProviderClassname != null) {
      try {
        Class clazz;
        try {
          clazz = Class.forName(secretProviderClassname, true, Thread.currentThread().getContextClassLoader());
        } catch (ClassNotFoundException e) {
          clazz = Class.forName(secretProviderClassname, true, getClass().getClassLoader());
        }
        Object secretProvider = clazz.newInstance();

        secretProviderBackend = new DuckTypedSecretProviderBackend(secretProvider);
      } catch (Exception e) {
        throw new KeychainInitializationException("Failed to initialize custom SecretProvider.", e);
      }
    } else {
      keychainUrl = System.getProperty(JVM_KEYCHAIN_LOCATION_PROP) == null ? DFLT_KEYCHAIN_FILE_LOCATION : System
          .getProperty(JVM_KEYCHAIN_LOCATION_PROP);
      secretProviderBackend = new ObfuscatingSecretProviderBackend();
    }

    if (keychainUrl != null) {
      try {
        kcLoc = new URL(keychainUrl);
      } catch (MalformedURLException e) {
        throw new KeychainInitializationException("Failed to instantiate keychain URL.", e);
      }

      SecretProvider.fetchSecret(secretProviderBackend);

      try {
        keyChain = new FileStoreKeyChain(kcLoc);
        keyChain.unlock(SecretProvider.getSecret());
      } catch (RuntimeException e) {
        throw new KeychainInitializationException(e);
      }
    } else {
      throw new KeychainInitializationException("Unable to locate a valid keychain for security context.");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] retrieveSecret(KeyName alias) {
    byte[] secret = keyChain.getPassword(SecretProvider.getSecret(), alias);
    if (secret == null) {
      LOG.info("No secret found in keychain for URL: " + alias.toString());
    }
    return secret;
  }
}
