/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.impl;

import org.apache.shiro.codec.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.resource.exceptions.ExceptionUtils;

import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.json.JSONConfiguration;
import com.sun.jersey.client.urlconnection.HTTPSProperties;
import com.terracotta.management.keychain.URIKeyName;
import com.terracotta.management.security.HMACBuilder;
import com.terracotta.management.security.IACredentials;
import com.terracotta.management.security.IdentityAssertionServiceClient;
import com.terracotta.management.security.InvalidIAInteractionException;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.MaskedUserInfo;
import com.terracotta.management.security.SSLContextFactory;
import com.terracotta.management.security.SecurityServiceDirectory;
import com.terracotta.management.user.UserInfo;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.InvalidKeyException;
import java.security.KeyManagementException;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.Arrays;
import java.util.UUID;

import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.ws.rs.core.Response;

/**
 * @author brandony
 */
public class JerseyIdentityAssertionServiceClient implements IdentityAssertionServiceClient {

  private static final Logger LOG = LoggerFactory.getLogger(JerseyIdentityAssertionServiceClient.class);

  private static final int CONN_TIMEOUT = 5 * 1000; // 5 sec
  private static final int READ_TIMEOUT = 10 * 1000; // 10 sec

  private final Client client;
  private final KeyChainAccessor keyChainAccessor;
  private final SecurityServiceDirectory securityServiceDirectory;

  public JerseyIdentityAssertionServiceClient(KeyChainAccessor keyChainAccessor,
                                              SSLContextFactory sslCtxtFactory,
                                              SecurityServiceDirectory securityServiceDirectory) {
    this.securityServiceDirectory = securityServiceDirectory;

    ClientConfig clientConfig = new DefaultClientConfig();

    if (sslCtxtFactory != null) {
      SSLContext sslCtxt;
      try {
        sslCtxt = sslCtxtFactory.create();
      } catch (NoSuchAlgorithmException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to invalid KeyManagerFactory algorithm.",
            e);
      } catch (IOException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to inability to load keyStore.", e);
      } catch (KeyStoreException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to invalid KeyStore type.", e);
      } catch (CertificateException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to invalid certificates in a KeyStore.", e);
      } catch (UnrecoverableKeyException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to bad key in a KeyStore.", e);
      } catch (KeyManagementException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to one or more invalid keys in a KeyStore.",
            e);
      } catch (URISyntaxException e) {
        throw new RuntimeException(
            "Failure instantiating JerseyIdentityAssertionServiceClient due to bad store location.", e);
      }

      HostnameVerifier hostnameVerifier;
      if (Boolean.getBoolean("tc.ssl.disableHostnameVerifier")) {
        hostnameVerifier = new HostnameVerifier() {
          @Override
          public boolean verify(String hostname, SSLSession session) {
            return true;
          }
        };
      } else {
        //DEV-8842 : returning false every time means :
        // we don't allow the hostname to be different from what is configured in the cert.
        hostnameVerifier = HttpsURLConnection.getDefaultHostnameVerifier();
      }

      clientConfig.getProperties().put(HTTPSProperties.PROPERTY_HTTPS_PROPERTIES, new HTTPSProperties(hostnameVerifier, sslCtxt));
    }

    clientConfig.getFeatures().put(JSONConfiguration.FEATURE_POJO_MAPPING, Boolean.TRUE);

    Integer securitySvcTimeout = securityServiceDirectory.getSecurityServiceTimeout();
    clientConfig.getProperties()
        .put(ClientConfig.PROPERTY_CONNECT_TIMEOUT, securitySvcTimeout == null ? CONN_TIMEOUT : securitySvcTimeout);
    clientConfig.getProperties()
        .put(ClientConfig.PROPERTY_READ_TIMEOUT, securitySvcTimeout == null ? READ_TIMEOUT : securitySvcTimeout);

    client = Client.create(clientConfig);

    this.keyChainAccessor = keyChainAccessor;
  }

  @Override
  public UserInfo retreiveUserDetail(IACredentials credentials) throws InvalidIAInteractionException {
    String clientNonce = credentials.isUsingClientCertAuth() ? null : UUID.randomUUID().toString();
    String reqTicket = credentials.getRequestTicket();
    String sessId = credentials.getIdentityToken();
    String alias = credentials.getRequestAlias();

    URI securitySvcLocation = securityServiceDirectory.getSecurityServiceLocation();
    if (securitySvcLocation == null) {
      throw new InvalidIAInteractionException(
          String.format("No security service location was specified, request ticket '%s', token id '%s'.",
          reqTicket, sessId)
      );
    }
    WebResource r = client.resource(securitySvcLocation);


    ClientResponse response;
    try {
      response = r.header(IACredentials.REQ_TICKET, reqTicket).header(IACredentials.TC_ID_TOKEN, sessId)
          .header(IACredentials.ALIAS, alias).header(IACredentials.CLIENT_NONCE, clientNonce).get(ClientResponse.class);
    } catch (ClientHandlerException che) {
      throw new InvalidIAInteractionException("Communication with IA server failed: " + ExceptionUtils.getRootCause(che).getMessage(), che);
    }

    if (response.getStatus() == Response.Status.OK.getStatusCode()) {
      UserInfo user = response.getEntity(MaskedUserInfo.class);

      if (!credentials.isUsingClientCertAuth()) {
        String signature = response.getHeaders().getFirst(IACredentials.SIGNATURE);

        byte[] reqSig = Base64.decode(signature);

        byte[] calcSig;
        try {
          URIKeyName uriAlias = new URIKeyName(alias);
          byte[] keyMaterial = keyChainAccessor.retrieveSecret(uriAlias);
          if (keyMaterial == null) {
            throw new InvalidIAInteractionException("Missing keychain entry for URL [" + alias + "]");
          }
          calcSig = HMACBuilder.getInstance(keyMaterial).addMessageComponent(reqTicket)
              .addMessageComponent(sessId).addMessageComponent(alias).addMessageComponent(clientNonce)
              .addUserDetail(user).build();
        } catch (NoSuchAlgorithmException e) {
          throw new RuntimeException("BUG Alert! Failed to create signed hash.", e);
        } catch (InvalidKeyException e) {
          throw new RuntimeException("BUG Alert! Failed to create signed hash.", e);
        } catch (URISyntaxException e) {
          throw new RuntimeException(
              "BUG Alert! Unable to determine uri alias for obtaining the key material to sign the hash.", e);
        }

        if (!Arrays.equals(reqSig, calcSig)) {
          throw new InvalidIAInteractionException(String
              .format("Forgery detected from identity assertion service for request ticket '%s', token id '%s'.",
                  reqTicket, sessId));
        }
      }
      return user;
    } else {
      if (LOG.isDebugEnabled()) {
        String extraErrorMessage = "";
        StringBuilder sb = new StringBuilder();
        sb.append(System.getProperty("line.separator"));
        sb.append("HTTP(S) response was:");
        sb.append(System.getProperty("line.separator"));

        BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(response.getEntityInputStream()));
        while (true) {
          try {
            String line = bufferedReader.readLine();
            if (line == null) {
              extraErrorMessage = sb.toString();
              break;
            }
            sb.append(line).append(System.getProperty("line.separator"));
          } catch (IOException e1) {
            LOG.debug("Unable to read response", e1);
            break;
          }
        }
        LOG.debug("Failed to execute IA service request. " + extraErrorMessage);
      }

      throw new InvalidIAInteractionException("Request to identity assertion service failed: IA service failed with HTTP error " + response.getStatus());
    }
  }

}
