/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.shiro.realm;

import com.terracotta.management.security.IACredentials;
import com.terracotta.management.security.IdentityAssertionServiceClient;
import com.terracotta.management.security.InvalidIAInteractionException;
import com.terracotta.management.security.shiro.IdentityAssertionToken;
import com.terracotta.management.user.UserInfo;
import com.terracotta.management.user.UserRole;
import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationInfo;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.authc.SimpleAuthenticationInfo;
import org.apache.shiro.authz.AuthorizationInfo;
import org.apache.shiro.authz.SimpleAuthorizationInfo;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.subject.PrincipalCollection;
import org.apache.shiro.subject.SimplePrincipalCollection;
import org.apache.shiro.util.CollectionUtils;
import org.apache.shiro.util.StringUtils;
import org.terracotta.management.ServiceLocator;

import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Set;

/**
 * This realm implementation acts as a client to a pre-configured security service for identity assertion and federated authorization.
 *
 * @author brandony
 */
public final class TCIdentityAssertionRealm extends AuthorizingRealm {
  private static final String OPERATOR_PERM = "api:read";

  private static final Collection<String> ADMIN_PERMS = Arrays
      .asList(new String[]{"api:update", "api:create", "api:delete"});

  private final IdentityAssertionServiceClient iaClient;

  public TCIdentityAssertionRealm() throws URISyntaxException, MalformedURLException {
    super();

    this.iaClient = ServiceLocator.locate(IdentityAssertionServiceClient.class);
    setAuthenticationTokenClass(IdentityAssertionToken.class);
    setCachingEnabled(false);
    setAuthenticationCachingEnabled(false);
  }

  @Override
  protected void onInit() {
    super.onInit();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected AuthenticationInfo doGetAuthenticationInfo(AuthenticationToken token) throws AuthenticationException {
    if (token == null) {
      return null;
    }

    IdentityAssertionToken iaToken = (IdentityAssertionToken)token;

    IACredentials credentials = (IACredentials) iaToken.getCredentials();

    if (!StringUtils.hasText(credentials.getIdentityToken())) return null;

    UserInfo user;
    try {
      user = iaClient.retreiveUserDetail(credentials);
    } catch (InvalidIAInteractionException e) {
      throw new AuthenticationException(e);
    }

    iaToken.setPrincipal(user.getUsername());

    List<Object> principals = CollectionUtils.asList(user.getUsername(), user.getRoles());
    PrincipalCollection principalCollection = new SimplePrincipalCollection(principals, getName());
    return new SimpleAuthenticationInfo(principalCollection, credentials);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected AuthorizationInfo doGetAuthorizationInfo(PrincipalCollection principals) {
    // retrieve user information
    SimplePrincipalCollection principalCollection = (SimplePrincipalCollection) principals;
    List<Object> listPrincipals = principalCollection.asList();
    Set<UserRole> roles = (Set<UserRole>) listPrincipals.get(1);

    SimpleAuthorizationInfo sai = new SimpleAuthorizationInfo();

    for (UserRole r : roles) {
      sai.addRole(r.toString());
      if (r == UserRole.ADMIN) {
        sai.addStringPermissions(ADMIN_PERMS);
      } else if (r == UserRole.OPERATOR) {
        sai.addStringPermission(OPERATOR_PERM);
      }
    }

    return sai;
  }
}
