/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.impl;

import com.terracotta.management.keychain.FileStoreKeyChain;
import com.terracotta.management.keychain.KeyName;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.KeychainInitializationException;
import com.terracotta.management.security.ObfuscatingSecretProviderBackend;
import com.terracotta.management.security.SecretProvider;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.net.MalformedURLException;
import java.net.URL;

/**
 * @author brandony
 */
public final class ObfuscatedSecretFileStoreKeyChainAccessor implements KeyChainAccessor {
  private static final Logger LOG = LoggerFactory.getLogger(ObfuscatedSecretFileStoreKeyChainAccessor.class);

  /**
   * A constant for the default keychain file location.
   */
  private static final String DFLT_KEYCHAIN_FILE_LOCATION = "file://" + System
      .getProperty("user.home") + "/.tc/mgmt/keychain";

  /**
   * A constant for the jvm keychain file location property.
   */
  private static final String JVM_KEYCHAIN_LOCATION_PROP = "com.tc.management.keychain.file";

  private FileStoreKeyChain keyChain;

  public ObfuscatedSecretFileStoreKeyChainAccessor() throws KeychainInitializationException {
    String keychainPath = System.getProperty(JVM_KEYCHAIN_LOCATION_PROP) == null ? DFLT_KEYCHAIN_FILE_LOCATION : System
        .getProperty(JVM_KEYCHAIN_LOCATION_PROP);
    if (keychainPath != null) {
      URL kcLoc;
      try {
        kcLoc = new URL(keychainPath);
      } catch (MalformedURLException e) {
        throw new KeychainInitializationException("Failed to instantiate keychain URL.", e);
      }

      SecretProvider.fetchSecret(new ObfuscatingSecretProviderBackend());

      try {
        keyChain = new FileStoreKeyChain(kcLoc);
        keyChain.unlock(SecretProvider.getSecret());
      } catch (RuntimeException e) {
        throw new KeychainInitializationException(e);
      }
    } else {
      throw new KeychainInitializationException("Unable to locate a valid keychain for security context.");
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public byte[] retrieveSecret(KeyName alias) {
    byte[] secret = keyChain.getPassword(SecretProvider.getSecret(), alias);
    if (secret == null) {
      LOG.info("No secret found in keychain for URL: " + alias.toString());
    }
    return secret;
  }
}
