/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security;

import com.terracotta.management.user.UserInfo;
import com.terracotta.management.user.UserRole;
import org.apache.shiro.codec.Base64;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * @author brandony
 */
public final class HMACBuilder {
  private static final String DFLT_ALG = "HmacSHA1";

  private static final byte FILLER_BYTE = 0;

  private final byte[] keyMaterial;

  private final String algorithm;

  private List<String> messageComponents;

  private UserInfo user;

  private HMACBuilder(final byte[] keyMaterial,
                      final String algorithm) {
    if (keyMaterial == null) throw new IllegalArgumentException("keyMaterial == null");
    this.keyMaterial = keyMaterial;

    if (algorithm == null) throw new IllegalArgumentException("algorithm == null");
    this.algorithm = algorithm;
  }

  public static HMACBuilder getInstance(final byte[] keyMaterial) {
    return new HMACBuilder(keyMaterial, DFLT_ALG);
  }

  public static HMACBuilder getInstance(final byte[] keyMaterial,
                                        final String algorithm) {
    return new HMACBuilder(keyMaterial, algorithm);
  }

  public HMACBuilder addMessageComponent(String messageComponent) {
    if (messageComponents == null) {
      messageComponents = new ArrayList<String>();
    }

    messageComponents.add(messageComponent);
    return this;
  }

  public HMACBuilder addUserDetail(UserInfo user) {
    this.user = user;
    return this;
  }

  public byte[] build() throws NoSuchAlgorithmException, InvalidKeyException {
    SecretKeySpec secretKey = new SecretKeySpec(keyMaterial, algorithm);
    Mac mac = Mac.getInstance(algorithm);
    mac.init(secretKey);

    if (messageComponents == null) throw new IllegalStateException("No message components available to hash.");

    StringBuilder message = new StringBuilder();
    for (String mc : messageComponents) {
      message.append(mc);
      message.append(mc.length());
    }

    if (user != null) {
      message.append(user.getUsername());
      message.append(user.getUsername().length());

      SortedSet<UserRole> sortedRoles = new TreeSet<UserRole>(user.getRoles());

      for (UserRole r : sortedRoles) {
        message.append(r.toString());
        message.append(r.toString().length());
      }
    }

    byte[] hmac = mac.doFinal(message.toString().getBytes());

    Arrays.fill(keyMaterial, FILLER_BYTE);
    return hmac;
  }

  public String buildEncoded() throws NoSuchAlgorithmException, InvalidKeyException {
    return Base64.encodeToString(build());
  }
}
