function tmc(options, $container) {
  var that = this;
  
  this.options = options;
  this.dashboard = new Dashboard($container.find('#dashboardArea'));
  this.DEFAULT_DROPDOWN_CONFIG = {
    dataTextField: "name",
    dataValueField: "name",
    open: function(e) {
      tmc.suspend();
    },
    close: function(e) {
      tmc.resume();
    },
    animation: false
  };
  this.DEFAULT_DROPDOWN_DATASOURCE_CONFIG = {
    data: [],
    schema: {
      model: {
        id: "name",
        fields: {name: {type: "string", editable: false}}
      }
    }
  };
  this.connectionGroupSelectorDS = new kendo.data.DataSource(
    $.extend({}, this.DEFAULT_DROPDOWN_DATASOURCE_CONFIG, {
      sort: {field: "name", dir: "asc"}
    }));
  this.$connectionGroupSelector = $container.find("#connectionGroupSelector").kendoDropDownList(
    $.extend({}, this.DEFAULT_DROPDOWN_CONFIG, {
      dataSource: this.connectionGroupSelectorDS,
      change: $.proxy(this.handleConnectionGroupSelection, this)
    }));
  this.kConnectionGroupSelector = this.$connectionGroupSelector.data("kendoDropDownList");
  this.$clusterStatus = $container.find('#clusterStatus').tooltip({placement: 'bottom'});
  
  this.$contentArea = $container.find("#contentArea");

  this.templates = $container.find("#templates").detach();
  this.groupPages = {};
  this.connections = {};
  this.clusters = {}; // groupId -> topologyResponse
  this.cacheManagerTopologies = {}; /* groupId -> cacheManagerNames -> nodeNames -> cacheManagerAttributes -> cachesNames */
  this.CACHE_MANAGER_ATTRIBUTES = [
    'Enabled',
    'Status',
    'CacheNames',
    'MaxBytesLocalDiskAsString',
    'MaxBytesLocalHeap',
    'MaxBytesLocalDisk',
    'MaxBytesLocalOffHeapAsString',
    'MaxBytesLocalOffHeap',
    'MaxBytesLocalHeapAsString'
  ];
  
  this.operatorEventTypeMap = {
    INFO:     0,
    WARN:     1,
    DEBUG:    2,
    ERROR:    3,
    CRITICAL: 4
  };
    
  if (options.needMarketingManager) {
    this.marketingManager = new MarketingManager(this);
  }
  this.aboutWindow = new AboutWindow(this);

  this.timerInterval = 5000;
  this.historySampleCount = 30;
  
  this.xhrPool = [];

  $.ajaxSetup({
    beforeSend: function(jqXHR, settings) {
      if (settings.cancelable == undefined || settings.cancelable === true) {
        that.xhrPool.push(jqXHR);
      }
    },
    complete: function(jqXHR) {
      var index = that.xhrPool.indexOf(jqXHR);
      if (index > -1) {
        that.xhrPool.splice(index, 1);
      }
    }
  });

  this.THIN_FORMAT = new ThinFormat();
  this.DEFAULT_FLOAT_PLACES = 1;
  this.BINARY_2_DECIMAL_RATIO = 1000/1024;
  
  var groupAndConnectionLoader = function() {
    that.loadGroups().done(function() {
      that.loadConnections().always(function() {
        that.setPollTimer(10);
      });
    });
  };
  this.loadUserProfile().done(function(xml) {
    if (xml != null) {
      that.handleUserProfile($(xml));
    }
  }).always(function() {
    groupAndConnectionLoader.call();
  });

  this.loadAuthenticationEnabled().done(function(data) {
    if (data != null) {
      that.authenticationEnabled = data;
    } 
  });
  
  $container.bind("resize", function(e) {
    e.stopPropagation();
    that.assignRemainingHeight(that.$contentArea);    
  });
  $container.trigger("resize");

  if (typeof(this.idleTimeoutMinutes) == 'number') {
    $.idleTimer(this.idleTimeoutMinutes * 60 * 1000);
    $(document).bind("idle.idleTimer", function() {
      window.location = that.options.contextPath + "/logout";
    });
  }

  $(document).bind('resourceStateChanged', $.proxy(this.handleResourceStateChanged, this));

  this._handleResponseComplete = $.proxy(this.handleResponseComplete, this);
  this._poll = $.proxy(this.poll, this);
  
  this.DEFAULT_POLL_MILLIS = 3000;

  this.CHART_DEFAULTS = {
    title: {
      visible: false
    },
    categoryAxis: {
      type: "Category",
      field: "time",
      labels: { format: "h:mm:ss", step: 1, visible: true },
      minorGridLines: { visible: false },
      minorTicks: { visible: false },
      majorGridLines: { visible: false },
      majorTicks: { visible: true }
    },
    valueAxis: { 
      labels: {step: 2},
      min: 0,
      minorGridLines: { visible: false },
      minorTicks: { visible: false },
      majorGridLines: { visible: false },
      majorTicks: { visible: true },
      plotBands: [{ from: 0, to: Number.MAX_VALUE, color: "#c2d5e1"}]
    },
    series:[{
      type: "line",
      field: "value",
      width: 2,
      markers: { visible: false },
      missingValues: "interpolate"
    }],
    chartArea: {
      background: "",
      height: 240,
      width: 400,
      margin: 0
    },
    plotArea: {
      margin: 0
    },
    transitions: false,
    legend: { position: "bottom", visible: false },
    axisDefaults : {
      labels : {
        font: "10px Arial,Helvetica,sans-serif",
        format: "{0:n0}" // whole numbers; zero places
      }
    }
  };
}

tmc.prototype = {
  handleResourceStateChanged: function(e) {
    // alert("Server '" + e.serverName + "' is now in '" + e.resourceState + "' mode.");
  },
   
  getTemplate: function(id) {
    return this.templates.find(id);
  },
  
  getPreferenceManager: function() {
    if (this.preferenceManager == null) {
      this.preferenceManager = new PreferenceManager();
    }
    return this.preferenceManager;
  },
  
  newConnection: function() {
    this.getPreferenceManager().newConnection();
  },
  
  managePreferences: function() {
    this.getPreferenceManager().show();
  },

  manageConnections: function() {
    this.getPreferenceManager().manageConnections();
  },

  showConnectionGroup: function(groupName) {
    this.getPreferenceManager().showConnectionGroup(groupName);
  },
  
  deleteConnectionGroup: function(groupName) {
    this.getPreferenceManager().deleteConnectionGroup(groupName);
  },

  showAboutWindow: function() {
    this.aboutWindow.show();
  },
  
  loadUserProfile: function() {
    var that = this;

    return $.ajax({
      type: "GET",
      url: "api/userprofiles/" + this.options.user,
      dataType: "xml"
    }).error(function(jqXHR, textStatus, errorThrown) {
      that.userProfile = {
        statisticsPollingIntervalMillis: that.DEFAULT_POLL_MILLIS,
        operatorEventLevel: 'WARN',
        hiddenConnectionGroups: []
      };
    });
  },

  loadAuthenticationEnabled: function() {
    var that = this;
    
    return $.ajax({
      type: "GET",
      url: "api/config/settings/authentication/",
      dataType: "json"
    }).error(function(jqXHR, textStatus, errorThrown) {
      var msg = "Failed to load authentication status";
      that.handleError(msg, jqXHR, textStatus, errorThrown);
    });
  },
  
  handleUserProfile: function($xml) {
    var $userProfile = $xml.find('userProfile'),
      pollMillis = $userProfile.attr('statisticsPollingIntervalMillis'),
      chartedStats = $userProfile.attr('ehcacheChartedStatistics'),
      gridStats = $userProfile.attr('ehcacheGridStatistics'),
      hiddenGroups = $userProfile.attr('hiddenConnectionGroups'),
      operatorEventLevel = $userProfile.attr('operatorEventLevel');

    this.userProfile = {
      statisticsPollingIntervalMillis: pollMillis != null ? pollMillis : this.DEFAULT_POLL_MILLIS,
      ehcacheChartedStatistics: (chartedStats != null) ? chartedStats.split(' ') : null,
      ehcacheGridStatistics: (gridStats != null) ? gridStats.split(' ') : null,
      hiddenConnectionGroups: (hiddenGroups != null && hiddenGroups != "") ? hiddenGroups.split(',') : [],
      operatorEventLevel: (operatorEventLevel != "undefined") ? operatorEventLevel : 'WARN'
    };
    this.timerInterval = this.userProfile.statisticsPollingIntervalMillis;
  },
  
  updateUserProfile: function(theUserProfile) {
    var that = this,
      input = '<?xml version="1.0" encoding="UTF-8" standalone="yes"?>\n'
        + '<userProfile ehcacheChartedStatistics="' + theUserProfile.ehcacheChartedStatistics.join(' ') + '"\n'
        + '  ehcacheGridStatistics="' + theUserProfile.ehcacheGridStatistics.join(' ') + '"\n'
        + '  id="' + this.options.user + '"\n'
        + '  hiddenConnectionGroups="' + theUserProfile.hiddenConnectionGroups.join() + '"\n'
        + '  operatorEventLevel="' + theUserProfile.operatorEventLevel + '"\n'
        + '  statisticsPollingIntervalMillis="' + theUserProfile.statisticsPollingIntervalMillis + '">\n'
        + '</userProfile>';
    
    return $.ajax({
      type: "PUT",
      url: "api/userprofiles/" + this.options.user,
      success: function() {
        this.userProfile = theUserProfile;
        $(document).trigger("tmc.userProfileChanged");
      },
      context: this,
      processData: false,
      data: input,
      contentType: "text/xml"
    }).error(function(jqXHR, textStatus, errorThrown) {
      var msg = "Failed to update user profile";
      that.handleError(msg, jqXHR, textStatus, errorThrown);
    });
  },
  
  updateAuthentication: function(authenticationEnabled) {
    return $.ajax({
      type: "PUT",
      url: "api/config/settings/authentication/",
      success: function(data) {
        setTimeout(function() {
          window.location.reload(1);
        }, 2000);
      },
      processData: false,
      data: JSON.stringify(authenticationEnabled),
      contentType: "application/json"
    }).error(function(jqXHR, textStatus, errorThrown) {
      var msg = "Failed to update authentication enablement";
      that.handleError(msg, jqXHR, textStatus, errorThrown);
    });
  },

  loadGroups: function() {
    var that = this;
    
    return $.ajax({
      type: "GET",
      url: "api/config/groups",
      success: function(data) {
        $.each(data, function(index, value) {
          if (that.connections[value] == null) {
            that.connections[value] = {};
          }
        });
      },
      dataType: "json"
    }).error(function(jqXHR, textStatus, errorThrown) {
      var msg = "Failed to load groups";
      that.handleError(msg, jqXHR, textStatus, errorThrown);
    });
  },

  loadConnections: function() {
    var that = this;
    
    return $.ajax({
      type: "GET",
      url: "api/config/agents",
      success: function(data) {
        $.alphaSort(data, "name");
        $.each(data, function(index, value) {
          that.registerAgent(value);
        });
        if ($.isEmptyObject(that.connections)) {
          that.manageConnections();
        }
      },
      dataType: "json"
    }).error(function(jqXHR, textStatus, errorThrown) {
      var msg = "Failed to load connections";
      that.handleError(msg, jqXHR, textStatus, errorThrown);
    });
  },
  
  handleAgentsResponse: function(data) {
    var that = this,
      requests = [this.loadClusterTopologies(data), this.loadCacheManagerTopologies(data)];
    
    $.when.apply(null, requests).then(function(a, b) {
      that.handleClusterTopologies(a.length > 0 ? a[0] : []);
      that.handleCacheManagerTopologies(b.length > 0 ? b[0] : []);
    }).always(function() {
      that.completeAgentsResponse(data);
    });
  },

  completeAgentsResponse: function(data) {
    var that = this,
      groups = [],
      activeConnections = {},
      clusterPages = {}, // group -> groupPage
      clusterNodes = {}, // group -> agentInfo[]
      kConnectionGroupSelector = this.kConnectionGroupSelector,
      connectionGroupSelectorDS = this.connectionGroupSelectorDS,
      dataItem = kConnectionGroupSelector.dataItem();
    
    $.alphaSort(data, "agentId");
    
    that.dashboard.update(data);
  
    $.each(data, function(index, value) {
      var idElems = tmc.parseId(value.agentId),
        nodeName = idElems.nodeName,
        group = idElems.groupName,
        suffix = idElems.suffix;

      if($.inArray(group,tmc.userProfile.hiddenConnectionGroups) != -1) {
        //skip this hidden group/cluster
        return true;
      }

      if (suffix == 'embedded') {
        groups.push(group);

        if (connectionGroupSelectorDS.get(group) == null) {
          that.groupPages[group] = new GroupPage(group);
          if (value.agencyOf == 'TSA') {
            that.groupPages[group].setClusterRedundantName(nodeName);
          }
          connectionGroupSelectorDS.add({name: group});
          if (dataItem == null) {
            that.$connectionGroupSelector.closest('div#top').show();
            kConnectionGroupSelector.select(0);
            that.handleConnectionGroupSelection();
          }
        }
        if (value.agencyOf == 'TSA') {
          clusterPages[group] = that.groupPages[group];
        }
      } else {
        var nodeList = clusterNodes[group];
        if (nodeList == null) {
          nodeList = clusterNodes[group] = [];
        }
        nodeList.push(suffix.replace('/', ':'));
      }
    });
    
    $.each(clusterPages, function(group, page) {
      var nodeList = clusterNodes[group] || [],
        existingNodes = page.getClusterNodes(),
        removedNodes = [];
      
      nodeList.sort();
      $.each(nodeList, function(i, node) {
        if ($.inArray(node, existingNodes) == -1) {
          page.addClusterNode(node);
        }
      });
      $.each(existingNodes, function(i, node) {
        if ($.inArray(node, nodeList) == -1) {
          removedNodes.push(node);
        }
      });
      if (removedNodes.length > 0) {
        page.removeClusterNodes(removedNodes);
      }
    });

    var selectedIndex = kConnectionGroupSelector.select(),
      wasSelected = false;
    
    for (var i = connectionGroupSelectorDS.total() - 1; i >= 0; i--) {
      var model = connectionGroupSelectorDS.at(i);

      if ($.inArray(model.name, groups) == -1) {
        connectionGroupSelectorDS.remove(model);
        that.removePage(model.name);
        if (i == selectedIndex) {
          wasSelected = true;
        }
      }
    }
    if (wasSelected) {
      if (connectionGroupSelectorDS.total() > 0) {
        kConnectionGroupSelector.select(Math.min(selectedIndex, connectionGroupSelectorDS.total() - 1));
      } else {
        kConnectionGroupSelector.wrapper.find('.k-input').text("");
      }
      that.handleConnectionGroupSelection();
    } else {
      this.refreshClusterStatus();
    }
    
    if ((dataItem = kConnectionGroupSelector.dataItem()) != null) {
      var currentPage = that.$connectionGroupSelector.data("currentPage"),
        page = that.groupPages[dataItem.name];
        
      if (page != currentPage) {
        that.$connectionGroupSelector.data("currentPage", currentPage = page);
        if (currentPage != null) {
          currentPage.show();
        }
      }
      
      if (currentPage != null) {
        currentPage.refresh();
      }
    } else {
      that.$connectionGroupSelector.closest('div#top').hide();
      that.handleResponseComplete();
    }
  },

  removePage: function(groupName) {
    this.groupPages[groupName].cleanup();
    delete this.groupPages[groupName];
  },
  
  handleResponseComplete: function() {
    var currentPage = this.$connectionGroupSelector.data("currentPage"),
      pollIntervalSeconds;
    
    if (currentPage != null) {
      pollIntervalSeconds = currentPage.getPollIntervalSeconds();
    }
    
    if (pollIntervalSeconds != null) {
      this.timerInterval = pollIntervalSeconds * 1000;
    } else {
      this.timerInterval = this.userProfile != null ?
        this.userProfile.statisticsPollingIntervalMillis :
        this.DEFAULT_POLL_MILLIS;
    }
     
    this.setPollTimer(this.timerInterval);
  },

  clearPollTimer: function() {
    if (this.timerHandle) {
      clearTimeout(this.timerHandle);
      this.timerHandle = null;
    }
  },
  
  setPollTimer: function(waitMillis) {
    if (this.timerHandle != null) {
      this.clearPollTimer();
    }
    
    if (waitMillis == null) {
      waitMillis = this.DEFAULT_POLL_MILLIS;
    }
    
    if (this.timerTime) {
      var diff = $.now() - this.timerTime;
      
      if (diff > waitMillis) {
        waitMillis = 0;
      } else {
        waitMillis -= diff;
      }
    }

    this.timerHandle = setTimeout(this._poll, waitMillis);
  },
  
  selectConnectionGroup: function(groupName) {
    this.kConnectionGroupSelector.select(function(dataItem) {
      return dataItem.name == groupName;
    });
    this.handleConnectionGroupSelection();
  },
  
  handleConnectionGroupSelection: function() {
    var dataItem = this.kConnectionGroupSelector.dataItem(),
      currentPage = this.$connectionGroupSelector.data("currentPage"),
      newPage = dataItem != null ? this.groupPages[dataItem.name] : null;
          
    if (currentPage != null) {
      currentPage.unbind("tmc.responseComplete", this._handleResponseComplete);
      this.suspend();
      currentPage.hide();
    }

    if (newPage != null) {
      newPage.show();
      newPage.bind("tmc.responseComplete", this._handleResponseComplete);
      this.resumeNoWait();
    }

    var groupInfo = this.dashboard.selectGroup(dataItem != null ? dataItem.name : null);
    this.updateClusterStatus(groupInfo);

    this.$connectionGroupSelector.data("currentPage", newPage);
  },
  
  updateClusterStatus: function(groupInfo) {
    if (groupInfo != null) {
      var resourceState,
        tipId;
      
      if (groupInfo.details.restrictedCount > 0) {
        resourceState = 'RESTRICTED';
        tipId = '#restrictedStateTooltip';
      } else if (groupInfo.details.throttledCount > 0) {
        resourceState = 'THROTTLED';
        tipId = '#throttledStateTooltip';
      }
    
      if (resourceState != null) {
        if (this.$clusterStatus.data('resourceState') != resourceState) {
          this.$clusterStatus.find('div').text(resourceState);
          this.updateToolTip(this.$clusterStatus, this.getTemplate(tipId).html());
          this.$clusterStatus.data('resourceState', resourceState).show();
        }
        return;
      }
    }
    this.$clusterStatus.hide();
  },
  
  refreshClusterStatus: function() {
    this.updateClusterStatus(this.dashboard.selectedGroupInfo());
  },
  
  showOperatorEvents: function(connectionGroup) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showOperatorEvents();
    }
  },

  showAppDataOverview: function(connectionGroup) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showAppDataOverview();
    }
  },

  showCacheManagerOverview: function(connectionGroup, cacheManagerName, nodeName) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showCacheManagerOverview(cacheManagerName, nodeName);
    }
  },

  showCacheManagerCharts: function(connectionGroup, cacheManagerName, nodeName) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showCacheManagerCharts(cacheManagerName, nodeName);
    }
  },

  showCacheManagerConfig: function(connectionGroup, cacheManagerName, nodeName) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showCacheManagerConfig(cacheManagerName, nodeName);
    }
  },

  showAppDataCharts: function(group) {
    this.selectConnectionGroup(group);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showAppDataCharts();
    }
  },
  
  showAggregateServerStats: function(connectionGroup) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showAggregateServerStats();
    }
  },
  
  showServerDetails: function(connectionGroup, serverName) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showServerDetails(serverName);
    }
  },

  showServerStats: function(connectionGroup, serverName) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showServerStats(serverName);
    }
  },

  showServerConfig: function(connectionGroup, serverName) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showServerConfig(serverName);
    }
  },

  showClientDetails: function(connectionGroup, remoteAddress) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showServerDetails(remoteAddress);
    }
  },

  showClientStats: function(connectionGroup, remoteAddress) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showClientStats(remoteAddress);
    }
  },

  showClientConfig: function(connectionGroup, remoteAddress) {
    this.selectConnectionGroup(connectionGroup);
    var page = this.$connectionGroupSelector.data("currentPage");
    if (page != null) {
      page.showClientConfig(remoteAddress);
    }
  },

  poll: function() {
    var that = this;
    
    this.clearPollTimer();
    
    var jqxhr = $.ajax({
      type: "GET",
      url: "api/agents",
      success: function(data) {
        if (that.showingConnectErrorMessage != null) {
          window.$tell.data('kendoWindow').destroy();
          that.showingConnectErrorMessage = null;
        }
        that.handleAgentsResponse(data);
      },
      context: this,
      dataType: "json"
    }).error(function(jqXHR, textStatus, errorThrown) {
      if (textStatus != 'abort') {
        if (jqXHR.status == 0 && that.showingConnectErrorMessage == null) {
          var now = new Date();
          that.showingConnectErrorMessage = true;
          window.tell("Lost connection to server at '" + now.toLocaleString() + "'");
        }
        that.setPollTimer(that.timerInterval);
      }
    });

    this.timerTime = $.now();
    
    return jqxhr;
  },

  getCacheManagerTopology: function(connectionGroup) {
    return this.cacheManagerTopologies[connectionGroup];
  },
  
  getTopology: function(connectionGroup) {
    return this.clusters[connectionGroup];
  },
  
  getActiveServerNames: function(connectionGroupName) {
    var that = this,
      topology = this.getTopology(connectionGroupName),
      result = [];

    if (topology != null) {
      $.each(topology.serverGroupEntities, function(i, serverGroup) {
        $.each(serverGroup.servers, function(j, server) {
          if (server.attributes.State == 'ACTIVE-COORDINATOR') {
            result.push(server.attributes.Name);
            return false;
          }
        });
      });
    }

    return result;
  },

  getServersForGroup: function(connectionGroupName, serverGroupName) {
    var that = this,
      topology = this.getTopology(connectionGroupName),
      result = [];

    if (topology != null) {
      $.each(topology.serverGroupEntities, function(i, serverGroup) {
        if (serverGroup.name == serverGroupName) {
          $.each(serverGroup.servers, function(j, server) {
            result.push(server.attributes.Name);
          });
          return false;
        }
      });
    }

    return result;
  },
  
  getActiveServerForGroup: function(connectionGroupName, serverGroupName) {
    var that = this,
      topology = this.getTopology(connectionGroupName),
      result = null;

    if (topology != null) {
      $.each(topology.serverGroupEntities, function(i, serverGroup) {
        if (serverGroup.name == serverGroupName) {
          $.each(serverGroup.servers, function(j, server) {
            if (server.attributes.State == 'ACTIVE-COORDINATOR') {
              result = server.attributes.Name;
              return false;
            }
          });
          return false;
        }
      });
    }

    return result;
  },

  getGroupForServer: function(connectionGroupName, serverName) {
    var that = this,
      topology = this.getTopology(connectionGroupName),
      result;

    if (topology != null) {
      $.each(topology.serverGroupEntities, function(i, serverGroup) {
        $.each(serverGroup.servers, function(j, server) {
          if (server.attributes.Name == serverName) {
            result = serverGroup;
            return false;
          }
        });
        if (result != null) {
          return false;
        }
      });
    }

    return result;
  },

  getServerByName: function(connectionGroupName, serverName) {
    var that = this,
      topology = this.getTopology(connectionGroupName),
      result;

    if (topology != null) {
      $.each(topology.serverGroupEntities, function(i, serverGroup) {
        $.each(serverGroup.servers, function(j, server) {
          if (server.attributes.Name == serverName) {
            result = server;
            return false;
          }
        });
        if (result != null) {
          return false;
        }
      });
    }

    return result;
  },

  getClientByID: function(connectionGroup, clientID) {
    var that = this,
      topology = this.getTopology(connectionGroup),
      result;
  
    if (topology != null) {
      $.each(topology.clientEntities, function(i, client) {
        if (client.attributes.ClientID == clientID) {
          result = client;
          return false;
        }
      });
    }
  
    return result;
  },
  
  handleClusterTopologies: function(data) {
    var that = this,
      existingClusters = $.map(this.clusters, function(v, k) {return k;});
    
    $.each(data, function(i, topology) {
      var idElems = tmc.parseId(topology.agentId)
        group = idElems.groupName,
        index = $.inArray(group, existingClusters);
      
      topology.serverGroupEntities.sort(function(a, b) {
        return a.id - b.id;
      });

      $.each(topology.serverGroupEntities, function(j, serverGroupEntity) {
        serverGroupEntity.servers.sort(function(a, b) {
          var aVal = a.attributes.Name,
            bVal = b.attributes.Name;

          if (aVal < bVal) {
            return -1;
          } else if (aVal > bVal) {
            return 1;
          }
          return 0;
        });
        
        $.each(serverGroupEntity.servers, function(k, server) {
          var oldServer = that.getServerByName(group, server.attributes.Name),
            serverResourceState = server.attributes.ResourceState;
          
          if ((oldServer != null && oldServer.attributes.ResourceState != null && oldServer.attributes.ResourceState != serverResourceState)
              || (oldServer == null && serverResourceState != null && serverResourceState != 'NORMAL'))
          {
            $(document).trigger({
              type: 'resourceStateChanged',
              resourceState: server.attributes.ResourceState,
              connectionGroupName: group,
              serverName: server.attributes.Name,
              serverGroupName: serverGroupEntity.name,
              serverGroupID: serverGroupEntity.id
            });
          }
        });
      });
      
      topology.clientEntities.sort(function(a, b) {
        return a.attributes.ClientID - b.attributes.ClientID;
      });

      that.clusters[group] = topology;
      if (index != -1) {
        existingClusters.splice(index, 1);
      }
    });
    
    $.each(existingClusters, function(i, group) {
      if (that.clusters[group] != null) {
        delete that.clusters[group];
      }
    });
  },
  
  loadClusterTopologies: function(agentsData) {
    var clusterIds = [],
      result;
    
    $.each(agentsData, function(i, agentData) {
      if (agentData.agencyOf == 'TSA') {
        clusterIds.push(agentData.agentId.replace(/\x2F/g, "%2F"));
      }
    });
    
    if (clusterIds.length > 0) {
      var ids = ';ids=' + clusterIds.join(',');
  
      result = $.ajax({
        type: 'GET',
        url: this.lastClusterTopologiesURL = this.encodeURI('api/agents' + ids + '/topologies'),
        dataType: 'json'
      });
    } else {
      result = $.Deferred();
      result.resolve([]);
    }
    
    return result;
  },

  handleCacheManagerTopologies: function(data) {
    var that = this,
      existingCacheManagerMap = {}, /* groupName -> cacheManagerNameList */
      cacheManagerNodeListMap = {}, /* groupName -> cacheManagerNames -> nodeList */
      groupNames = [],
      existingGroupList = $.map(this.cacheManagerTopologies, function(v, k) {return k;});
    
    $.each(this.cacheManagerTopologies, function(groupName, cacheManagers) {
      existingCacheManagerMap[groupName] = $.map(cacheManagers, function(v, k) {return k;});
    });
    
    $.each(data, function(i, value) {
      var cacheManagerName = value.name,
        idElems = tmc.parseId(value.agentId),
        suffix = idElems.suffix,
        nodeName = (suffix == 'embedded') ? idElems.nodeName : suffix.replace('/', ':'),
        groupName = idElems.groupName,
        cacheManagers = that.cacheManagerTopologies[groupName],
        cacheManager = cacheManagers != null ? cacheManagers[cacheManagerName] : null,
        nodeMap = cacheManager != null ? cacheManager.nodeMap : null,
        cacheManagerAttrs = nodeMap != null ? nodeMap[nodeName] : null,
        existingCacheManagerNameList = existingCacheManagerMap[groupName] || [],
        existingCacheManagerNameIndex = $.inArray(cacheManagerName, existingCacheManagerNameList);
      
      if (existingCacheManagerNameIndex != -1) {
        existingCacheManagerNameList.splice(existingCacheManagerNameIndex, 1);
      } 
      
      if (cacheManagers == null) {
        cacheManagers = that.cacheManagerTopologies[groupName] = {};
      }

      if (cacheManager == null) {
        cacheManager = cacheManagers[cacheManagerName] = {};
      }

      if (nodeMap == null) {
        nodeMap = cacheManager.nodeMap = {};
      }

      if (cacheManagerAttrs == null) {
        cacheManagerAttrs = nodeMap[nodeName] = value.attributes;
      } else {
        $.extend(cacheManagerAttrs, value.attributes);
      }

      var nodeListMap = cacheManagerNodeListMap[groupName];
      if (nodeListMap == null) {
        cacheManagerNodeListMap[groupName] = nodeListMap = {};
      }
      
      var nodeList = nodeListMap[cacheManagerName];
      if (nodeList == null) {
        nodeListMap[cacheManagerName] = nodeList = [nodeName];
      } else {
        nodeList.push(nodeName);
      }

      if ($.inArray(groupName, groupNames) == -1) {
        groupNames.push(groupName);
      }
    });
    
    $.each(existingGroupList, function(i, groupName) {
      if ($.inArray(groupName, groupNames) == -1) {
        delete that.cacheManagerTopologies[groupName];
      } else {
        var cacheManagers = that.cacheManagerTopologies[groupName]
          nodeListMap = cacheManagerNodeListMap[groupName];
        
        $.each(existingCacheManagerMap[groupName], function(j, cacheManagerNameList) {
          $.each(cacheManagerNameList, function(k, missingCacheManagerName) {
            delete cacheManagers[missingCacheManagerName];
          });
        });
        $.each(cacheManagers, function(cacheManagerName, cacheManager) {
          var nodeList = nodeListMap[cacheManagerName] || [];
          
          $.each(cacheManager.nodeMap || {}, function(nodeName) {
            if ($.inArray(nodeName, nodeList) == -1) {
              delete cacheManager.nodeMap[nodeName];
            }
          });
          
          if ($.isEmptyObject(cacheManager.nodeMap)) {
            delete cacheManagers[cacheManagerName];
          }
        });
      }
    });
  },
  
  loadCacheManagerTopologies: function(agentsData) {
    var agentIds = [],
      result;
    
    $.each(agentsData, function(i, agentData) {
      if (agentData.agencyOf == 'Ehcache') {
        agentIds.push(agentData.agentId.replace(/\x2F/g, "%2F"));
      }
    });

    if (agentIds.length > 0) {
      var ids = ';ids=' + agentIds.join(','),
        url = 'api/agents' + ids + '/cacheManagers?show=' + this.CACHE_MANAGER_ATTRIBUTES.join('&show=');
    
      this.lastCacheManagerTopologiesURL = this.encodeURI(url);
      
      result = $.ajax({
        type: 'GET',
        url: this.lastCacheManagerTopologiesURL,
        dataType: 'json'
      });
    } else {
      result = $.Deferred();
      result.resolve([]);
    }
    
    return result;
  },
  
  getCustomGroups: function() {
    var result = [];
    
    $.each(tmc.connections, function(group, members) {
      result.push(group);
      $.each(members, function(name, member) {
        if (member.type == 'TSA') {
          result.pop();
          return false;
        }
      });
    });
    
    return result;
  },

  isCustomGroup: function(connectionGroup) {
    return this.getCustomGroups().indexOf(connectionGroup) != -1;
  },
  
  getClusterGroups: function() {
    var result = [];
    
    $.each(tmc.connections, function(group, members) {
      result.push(group);
      $.each(members, function(name, member) {
        if (member.type != 'TSA') {
          result.pop();
          return false;
        }
      });
    });
    
    return result;
  },

  isClusterGroup: function(connectionGroup) {
    return this.getClusterGroups().indexOf(connectionGroup) != -1;
  },

  parseId: function(id) {
    var comps = id.split('#'),
      prefix = comps[0],
      suffix = comps[1],
      idElems = prefix.split('_%_%_'),
      groupName = idElems[0],
      nodeName = idElems[1];
    
    return {groupName: groupName, nodeName: nodeName, suffix: suffix};
  },
  
  createId: function(groupName, nodeName) {
    return groupName + '_%_%_' + nodeName;
  },
  
  registerAgent: function(agentAttrs) {
    var agentConfig = new AgentConfig(agentAttrs),
      groupId = agentConfig.getGroupId(),
      name = agentConfig.getName();

    if (this.connections[groupId] == null) {
      this.connections[groupId] = {};
    }
    this.connections[groupId][name] = agentConfig;
    
    return agentConfig;
  },
  
  getAgentLocation: function(connectionGroupName, connectionName) {
    var connectionGroup = this.connections[connectionGroupName];
    if (connectionGroup != null) {
      var connection = connectionGroup[connectionName];
      if (connection != null) {
        return connection.getAgentLocation();
      }
    }
    return null;
  },
  
  clearContentArea: function() {
    this.$contentArea.children().hide();
  },

  suspend: function() {
    this.clearPollTimer();
    $(this.xhrPool).each(function(idx, jqXHR) {
      if (jqXHR != null) {
        jqXHR.abort();
      }
    });
    this.xhrPool.length = 0
  },
  
  resume: function() {
    var millis = 10;
    
    if (this.timerTime != null) {
      var elapsedTime = $.now() - this.timerTime;
      if (elapsedTime < this.timerInterval) {
        millis = this.timerInterval - elapsedTime;
      }
    }

    this.setPollTimer(millis);
  },
  
  resumeNoWait: function() {
    this.setPollTimer(10);
  },

  reset: function() {
    this.suspend();
    this.resume();
  },
  
  handleError: function(msg, jqXHR, textStatus, errorThrown) {
    switch (jqXHR.status) {
      case 12029:
        msg += "; a connection to the server could not be created";
        break;
      case 400:
        msg += "; " + textStatus;
        break;
      case 409:
        msg += "; the name you specified is already in use, please choose another one";
        alert(msg);
        break;
      default:
        msg += "; " + errorThrown;
    }
    window.status = msg;
    alert(msg);
  },

  setupHelpLink: function(helpLinkElement, helpUrl) {
    helpLinkElement.click(function(e) {
      window.open('tmc-help.html#manage-connections');
      e.preventDefault();
    });
    helpLinkElement.removeClass("k-icon");
    helpLinkElement.addClass("helpLink");
    helpLinkElement.parent().removeClass("k-window-action");
  },

  remainingHeight: function($target) {
    var $parent = $target.parent(),
      result = $parent.height();
    
    $target.siblings("*").filter(":visible").each(function(index, element) {
       result -= $(element).outerHeight(true);
    });

    return result > 0 ? result : 0;
  },
  
  assignRemainingHeight: function($target) {
    var result = this.remainingHeight($target);
    $target.height(result);
    return result;
  },

  resizeHandler: function($target, workerFunc) {
    $target.bind("resize.tmc", function(e) {
      if ($target.is(":visible")) {
        $target.unbind("resize.tmc", arguments.callee);
        
        try {
          workerFunc.call(e);
        } catch (ex) {
          alert(ex.toString());
        }

        $target.bind("resize.tmc", arguments.callee);
      }
    });
  },
  
  labelForState: function(state) {
    switch (state) {
      case 'START-STATE':
        return '<span class="label label-warning">Starting</span>';
      case 'ACTIVE-COORDINATOR':
        return '<span class="label label-success">Active</span>';
      case 'PASSIVE-UNINITIALIZED':
        return '<span class="label">Initializing</span>';
      case 'PASSIVE-STANDBY':
        return '<span class="label label-info">Mirror</span>';
      case 'RECOVERING':
        return '<span class="label label-recovering">Mirror</span>';
      default:
        return '<span class="label label-important">Unreachable</span>';
    }
  },
  
  setDropDownWidth: function(el) {
    var d = el,
      p = d.data("kendoDropDownList").popup.element,
      w = p.css("visibility", "hidden").show().outerWidth();
    
    p.hide().css("visibility", "visible");
    d.closest(".k-widget").width(w);
  },

  formatThin: function(n) {
    return this.THIN_FORMAT.format(n);
  },
  
  formatThinMemory: function(kilobytes) {
    return this.THIN_FORMAT.formatMemory(kilobytes);
  },
  
  formatThinAsMemory: function(n) {
    return this.THIN_FORMAT.formatAsMemory(n);
  },

  formatInt: function(n) {
    return kendo.toString(n, "##,#");
  },
  
  formatIntRightJustified: function(n) {
    return "<div style='text-align:right;'>"
      + this.formatInt(n)
      + "</div>";
  },
  
  formatFloat: function(n) {
    var pattern = "n",
      places = this.DEFAULT_FLOAT_PLACES;
    
    if (arguments[1]) {
      places = arguments[1];
    }
    
    if ($.isNumeric(places)) {
      pattern += places.toFixed();
    }
    
    return kendo.toString(n, pattern);
  },
  
  formatFloatRightJustified: function(n) {
    var places = this.DEFAULT_FLOAT_PLACES;
    
    if (arguments[1]) {
      places = arguments[1];
    }

    return "<div style='text-align:right;'>"
      + this.formatFloat(n, places)
      + "</div>";
  },

  formatPercentage: function(n) {
    return kendo.toString(n, "p");
  },
  
  formatPercentageRightJustified: function(n) {
    return "<div style='text-align:right;'>"
      + this.formatPercentage(n)
      + "</div>";
  },
  
  toMegabytes: function(n) {
    return (n == 0) ? 0 : n/this.THIN_FORMAT.MBYTE;
  },
  
  calculateHitRatio: function(hits, misses) {
    return (hits == 0) ? 0 : hits/(hits+misses);
  },
  
  encodeURI: function(uri) {
    return encodeURI(uri).replace(/#/g, "%23");
  },
  
  updateToolTip: function($elem, newTitle) {
    var title = $elem.attr('data-original-title');
    
    if (title != newTitle) {
      $elem.tooltip('hide')
        .attr('data-original-title', newTitle)
        .tooltip('fixTitle');
    }
  },
  
  /**
   * Converts a decimal memory value to binary kilobytes
   */
//  kilobytes: function(decimalValue) {
//    return (decimalValue * 0.93132257461548)/1000;
//  },

  kilobytes: function(decimalValue) {
    return (decimalValue * this.BINARY_2_DECIMAL_RATIO)/1024;
  }
};
