/*
 * ClusterScopeSelector
 *   $input: jQuery wrapper around HTML input element
 *   options: KendoDropDownList options to override defaults. Must include connectionGroupProvider.
 *     Use $.proxy(this.connectionGroupProvider, this) to ensure proper context upon invocation.
 */

function ClusterScopeSelector($input, options) {
  if ($input == null) {
    throw 'NullPointerException: $input';
  } else if ($input.is('input') == false) {
    throw 'IllegalArgumentException: $input must wrap an HTML input element';
  }
  if (options == null) {
    throw 'NullPointerException: options';
  }
  if (typeof options.connectionGroupProvider !== 'function') {
    throw 'IllegalArgumentException: connectionGroupProvider option must be a function'
        + ' that returns a valid connection group name';
  }
  
  this.options = $.extend({}, tmc.DEFAULT_DROPDOWN_CONFIG, {
    dataSource: new kendo.data.DataSource(
      $.extend(true, {}, tmc.DEFAULT_DROPDOWN_DATASOURCE_CONFIG, {
        schema: {
          model: {
            fields: {
              groupIndex:     {type: "number", editable: false},
              serverState:    {type: "string", editable: false},
              serverAddress:  {type: "string", editable: false}
            }
          }
        }
      })),
    template: kendo.template(tmc.getTemplate("#clusterScopeSelector").html()),
    open: $.proxy(this.handleOpening, this),
    events: ['prePopulate']
  }, options);

  this.SERVER       = 'server';
  this.SERVER_GROUP = 'serverGroup';
  this.ALL_SERVERS  = 'allServers';
  this.CLIENT       = 'client';
  this.ALL_CLIENTS  = 'allClients';
  
   if (this.options.selectorOptions == null) {
     this.options.selectorOptions = {};
   }
   
  this.ALL_SERVERS_ENTRY = {name: this.options.selectorOptions.allServersLabel || "All Servers", type: this.ALL_SERVERS};
  this.ALL_CLIENTS_ENTRY = {name: this.options.selectorOptions.allClientsLabel || "All Clients", type: this.ALL_CLIENTS};
  
  this.testAddDefaultEntries();
  this.options.dataSource.add(this.ALL_SERVERS_ENTRY);
  this.kScopeSelector = $input.kendoDropDownList(this.options).data('kendoDropDownList');
  this.connectionGroupProvider = options.connectionGroupProvider;
}

ClusterScopeSelector.prototype = {
  testAddDefaultEntries: function() {
    var opts = this.options;
    
    if ($.isPlainObject(opts.selectorOptions)) {
      if ($.isArray(opts.selectorOptions.defaultEntries)) {
        $.each(opts.selectorOptions.defaultEntries, function(i, scopeEntry) {
          opts.dataSource.add(scopeEntry);
        });
      }
    }
  },
  
  bind: function(event, handler) {
    this.kScopeSelector.bind(event, handler);
  },
  
  select: function(index) {
    if (index != null) {
      this.kScopeSelector.select(index);
    } else {
      return this.kScopeSelector.select();
    }
  },
  
  selectAggregateServerStats: function() {
    this.kScopeSelector.select(0);
  },
  
  selectServer: function(serverName) {
    var that = this;
    this.kScopeSelector.select(function(dataItem) {
      return dataItem.type == that.SERVER && dataItem.name == serverName;
    });
  },
  
  selectClient: function(remoteAddress) {
    var that = this;
    this.kScopeSelector.select(function(dataItem) {
      return dataItem.type == that.CLIENT && dataItem.name == remoteAddress;
    });
  },

  dataItem: function() {
    return this.kScopeSelector.dataItem();
  },
  
  dataSource: function() {
    return this.kScopeSelector.dataSource;
  },
  
  getTopology: function() {
    return tmc.getTopology(this.connectionGroupProvider());
  },

  handleOpening: function(e) {
    if (e) {
      tmc.suspend(); /* resumed in default close event */
    }

    var that = this,
      ds = this.dataSource(),
      topology = this.getTopology();
      
    ds.clear();
    that.kScopeSelector.trigger('prePopulate', {dataSource: ds});
    this.testAddDefaultEntries();
    ds.add(this.ALL_SERVERS_ENTRY);

    if (topology != null) {
      $.each(topology.serverGroupEntities, function(i, serverGroup) {
        ds.add({
          name: serverGroup.name,
          groupIndex: i,
          type: that.SERVER_GROUP
        });
        
        $.each(serverGroup.servers, function(j, server) {
          ds.add({
            name: server.attributes.Name,
            serverAddress: server.attributes.Address,
            serverState: server.attributes.State,
            type: that.SERVER
          });
        });
      });
      
      if (that.options.selectorOptions.showClients == true) {
        if (topology.clients != null && $.isEmptyObject(topology.clients) == false) {
          ds.add(this.ALL_CLIENTS_ENTRY);
  
          $.each(topology.clients, function(addr, client) {
            ds.add({
              name: addr,
              clientID: client.attributes.ClientID,
              type: that.CLIENT
            });
          });
        }
      }
    }
  },
  
  /*
   * Determine if the ALL_CLIENTS item should be removed;
   * returns: true if it was removed
   */
  testRemoveAllClients: function() {
    var topology = this.getTopology();
    
    if (topology != null && (topology.clients == null || $.isEmptyObject(topology.clients))) {
      var ds = this.dataSource(),
        model = ds.getModel(this.ALL_CLIENTS_ENTRY.name);
      
      if (model != null) {
        ds.remove(model);
        return true;
      }
    }
    
    return false;
  },
  
  handleClientsDetached: function(e) {
    var scopeDS = this.dataSource(),
      wasSelected = false,
      selectedIndex = this.select(),
      index,
      model;

    if (e.nodeList != null) {
      $.each(e.nodeList, function(i, node) {
        if ((model = scopeDS.getModel(node)) != null) {
          index = scopeDS.indexOf(model);
          scopeDS.remove(model);
          if (index == selectedIndex) {
            selectedIndex = Math.min(index, scopeDS.total() - 1);
            wasSelected = true;
          }
        }
      });
    }
    
    if (this.testRemoveAllClients()) {
      selectedIndex = 0;
      wasSelected = true;
    }
    
    if (wasSelected) {
      this.select(selectedIndex);
      this.kScopeSelector.trigger('change', {});
    }
  }
};
