/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.userprofile.resource.services;

import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.user.UserRole;
import com.terracotta.management.userprofile.dao.UserProfileDao;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.exceptions.ResourceRuntimeException;

/**
 * @author brandony
 */
@Path("/userprofiles")
public final class UserProfileResourceService {
  private final UserProfileDao usrProfileDao;

  private final Authorizer authorizer;

  public UserProfileResourceService() {
    this.usrProfileDao = ServiceLocator.locate(UserProfileDao.class);
    this.authorizer = ServiceLocator.locate(Authorizer.class);
  }

  @GET
  @Path("/{userid}")
  @Produces(MediaType.TEXT_XML)
  public String retrieveProfile(@PathParam("userid") String userid) {
    checkAuthorization(userid);
    return usrProfileDao.getById(userid);
  }

  @POST
  @Consumes(MediaType.TEXT_XML)
  public void addUserProfile(String userId, String userProfile) {
    checkAuthorization(userId);

    try {
      usrProfileDao.create(userId, userProfile);
    } catch (DataAccessException e) {
      throw new ResourceRuntimeException("Error adding user profile", e,
          Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
    }
  }

  @PUT
  @Path("/{userid}")
  @Consumes(MediaType.TEXT_XML)
  public void updateUserProfile(@PathParam("userid") String userid, String userProfile) {
    checkAuthorization(userid);

    try {
      usrProfileDao.createOrUpdate(userid, userProfile);
    } catch (DataAccessException e) {
      throw new ResourceRuntimeException("Error updating user profile", e,
          Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
    }
  }

  @DELETE
  @Path("/{userid}")
  public void deleteUserProfile(@PathParam("userid") String userid) {
    checkAuthorization(userid);

    try {
      usrProfileDao.delete(userid);
    } catch (DataAccessException e) {
      throw new ResourceRuntimeException("Error deleting user profile", e,
          Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
    }
  }

  private void checkAuthorization(String userId) {
    if (!authorizer.isUserInRole(UserRole.ADMIN) && !authorizer.getPrincipal().equals(userId)) {
      //TODO: shouldn't that be Response.Status.FORBIDDEN ?
      throw new ResourceRuntimeException("Missing role " + UserRole.ADMIN + " or tried to spoof user " + userId,
          Response.Status.UNAUTHORIZED.getStatusCode());
    }
  }
}
