/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */

package com.terracotta.management.userprofile.dao.impl;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.Scanner;
import java.util.concurrent.ConcurrentHashMap;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.userprofile.dao.UserProfileDao;

/**
 * A DAO for {@link String} objects that uses an XML file as a datastore.
 * 
 * @author brandony
 */
public final class XMLFileUserProfileDao implements UserProfileDao {
  private static final Logger LOG = LoggerFactory.getLogger(XMLFileUserProfileDao.class);

  public static final String TMC_CONFIGURATION_DIRECTORY_PROPERTY = "com.tc.management.config.directory";
  private static final String TMC_CONFIGURATION_DEFAULT_DIRECTORY = System.getProperty("user.home") + "/.tc/mgmt";


  private ConcurrentHashMap<String, String> profilesMap = new ConcurrentHashMap<String, String>();

  private File tmcUserClientPreferencesDirectory;

  public XMLFileUserProfileDao() {
    String tmcConfigurationDirectory = System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY) == null ? TMC_CONFIGURATION_DEFAULT_DIRECTORY
        : System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY);
    tmcUserClientPreferencesDirectory = new File(tmcConfigurationDirectory + "/client/");
    if (!tmcUserClientPreferencesDirectory.exists()) {
        tmcUserClientPreferencesDirectory.mkdirs();
    }
    FilenameFilter filesEndingWithDotXmlFilter = new FilenameFilter() {
      @Override
      public boolean accept(File dir, String name) {
        if (name.endsWith(".xml")) {
          return true;
        }
        return false;
      }
    };
    File[] listFiles = tmcUserClientPreferencesDirectory.listFiles(filesEndingWithDotXmlFilter);
    for (File file : listFiles) {
      loadConfig(file.getName().substring(0, file.getName().indexOf(".xml")));
    }
  }

  private String loadConfig(String username) {
    File configFile = new File(tmcUserClientPreferencesDirectory, username + ".xml");

    String lConfig = null;
    try {
      lConfig = loadConfigFromFile(configFile);
    } catch (IOException e) {
//      try {
//        String templateConfigFileName = System.getProperty("com.tc.management.userprofile.client.template",
//            "templateUserProfileClientPreferences.xml");
//        copyFromTemplateFile(templateConfigFileName, configFile, username);
//        loadConfigFromFile(configFile);
//      } catch (Exception e1) {
        LOG.error("could not create and load user profile client preferences file in " + configFile, e);
      // }
    }

    profilesMap.put(username, lConfig);
    return lConfig;
  }

  // private void copyFromTemplateFile(String templateConfigFileName, File configFile, String username) throws DataAccessException {
  // InputStream src = FileConfigService.class.getClassLoader().getResourceAsStream(templateConfigFileName);
  // String tmpltConfig = loadConfigFromInputStream(src);
  // persistUserProfile(username, tmpltConfig);
  // }

  String loadConfigFromFile(File src) throws IOException {
    if (src.exists()) {
      FileInputStream fis;

      try {
        fis = new FileInputStream(src);
      } catch (FileNotFoundException e) {
        throw new IOException("Failure loading configuration file: " + src.getAbsolutePath(), e);
      }

      return loadConfigFromInputStream(fis);
    }
    return null;
  }

  String loadConfigFromInputStream(InputStream src) {
      return UserProfileSettings.load(src);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getById(String userId) {
    return profilesMap.get(userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void create(String userId, String userProfile) throws DataAccessException {
    String up = profilesMap.putIfAbsent(userId, userProfile);

    if (up != null) {
      throw new DataAccessException(String.format("Failed to create user profile '%s' because it already exists.", userProfile));
    } else {
      persistUserProfile(userId, userProfile);
    }

  }

  private void persistUserProfile(String userId, String userProfile) throws DataAccessException {
    try {
      UserProfileSettings.write(userProfile, new File(tmcUserClientPreferencesDirectory, userId + ".xml"));
    } catch (FileNotFoundException e) {
      throw new DataAccessException(e);
    }
  }

  /**
   * {@inheritDoc}
   * 
   * @throws DataAccessException
   */
  @Override
  public void createOrUpdate(String userId, String userProfile) throws DataAccessException {
    profilesMap.put(userId, userProfile);
    persistUserProfile(userId, userProfile);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void delete(String userId) throws DataAccessException {
    String up = profilesMap.remove(userId);

    if (up == null) {
      throw new DataAccessException(String.format("Failed to delete user profile '%s' because it does not exist.", userId));
    } else {
      deleteUserProfile(userId);
    }

  }


  private void deleteUserProfile(String userId) throws DataAccessException {
    File file = new File(tmcUserClientPreferencesDirectory, userId + ".xml");
    if (!file.delete()) {
      throw new DataAccessException("Could not delete user preference file :" + file.getAbsolutePath());
    }

  }

  /**
   * An inner static class that serves as the container for all user profile detail and provides a mechanism to read and
   * write the underlying XML file.
   */
  private final static class UserProfileSettings {

    private static String load(InputStream is) {
      Scanner s = new Scanner(is, "UTF-8").useDelimiter("\\A");
      return s.hasNext() ? s.next() : "";
    }

    private static void write(String userProfile, File configFile) throws FileNotFoundException {
      PrintWriter out = new PrintWriter(configFile);
      out.print(userProfile);
      out.close();
    }
  }
}
