/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.servlet;

import static com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener.*;

import java.io.IOException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.terracotta.license.LicenseException;
import org.terracotta.management.ServiceLocator;

import com.terracotta.license.LicenseManager;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.security.SecurityContextManager;
import com.terracotta.management.services.SystemConfigService;
import com.terracotta.management.user.UserRole;

/**
 * A {@code Filter} that is aware of TMC setup requirements and redirects requests to certain pages when no valid security context is available or
 * when the TMC is running without a license.
 * 
 * @author brandony
 */
public final class SetupAwareFilter implements Filter {
  private static final String TMC_HOME = "/index.jsp";

  private static final String TMC_AUTHENTICATION = "/setup/authenticationSetup.jsp";

  private static final String TMC_RESTART = "/restart.jsp";

  private static final String TMC_LOGOUT_URL = "/logout.jsp";

  private final SecurityContextManager securityCtxtMgr;

  private boolean restartNeeded;

  static {

    if (HAS_LICENSE == null) {
      try {
        LicenseManager.verifyTMCCapability();
        HAS_LICENSE = true;
        LICENSE_IS_COMMERCIAL_LICENSE = LicenseManager.isCommercialLicense();
      } catch (LicenseException e) {
        HAS_LICENSE = false;
      }
    }
  }

  public SetupAwareFilter() {
    this.securityCtxtMgr = ServiceLocator.locate(SecurityContextManager.class);
  }

  @Override
  public void init(FilterConfig filterConfig) throws ServletException {
    restartNeeded = false;
  }

  @Override
  public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
    HttpServletRequest req = (HttpServletRequest) servletRequest;
    HttpServletResponse resp = (HttpServletResponse) servletResponse;


    if (req.getRequestURI().startsWith(req.getContextPath() + "/images/") || req.getRequestURI().startsWith(req.getContextPath() + "/css/")
        || req.getRequestURI().startsWith(req.getContextPath() + "/javascript/") || req.getRequestURI().startsWith(req.getContextPath() + "/js/")
        || req.getRequestURI().startsWith(req.getContextPath() + "/json/") || req.getRequestURI().startsWith(req.getContextPath() + "/styles/")
        || req.getRequestURI().startsWith(req.getContextPath() + "/tcinfo.html") || req.getRequestURI().startsWith(req.getContextPath() + "/tcAd")

    ) {
      filterChain.doFilter(req, resp);
    } else {
      boolean authenticationEnabled = ServiceLocator.locate(SystemConfigService.class).isAuthenticationEnabled();
      boolean firstRun = ServiceLocator.locate(SystemConfigService.class).isFirstRun();
      boolean isUserConsideredAnAdmin = ServiceLocator.locate(Authorizer.class).isUserInRole(UserRole.ADMIN) || !authenticationEnabled;
      ServletContext servletContext = req.getSession().getServletContext();

      if (req.getRequestURI().startsWith(req.getContextPath() + TMC_RESTART)) {
        restartNeeded = true;
      }
      if (restartNeeded) {
        if (!req.getRequestURI().startsWith(req.getContextPath() + TMC_RESTART)) {
          RequestDispatcher rd = servletContext.getRequestDispatcher(TMC_RESTART);
          rd.forward(req, resp);
        } else {
          filterChain.doFilter(req, resp);
        }
        return;
      }
      if (HAS_LICENSE && firstRun) {
        if (req.getRequestURI().startsWith(req.getContextPath() + TMC_AUTHENTICATION)
            || req.getRequestURI().startsWith(req.getContextPath() + "/setup")) {
          filterChain.doFilter(req, resp);
        } else {
          RequestDispatcher rd = servletContext.getRequestDispatcher(TMC_AUTHENTICATION);
          rd.forward(req, resp);
        }
      } else if (HAS_LICENSE && !firstRun && authenticationEnabled) {
        boolean ctxtValid = securityCtxtMgr.hasValidSecurityContext();
        if (!ctxtValid) {
          if (req.getRequestURI().startsWith(req.getContextPath() + AccountInitializationServlet.TMC_INI_ACCOUNT_SETUP_URL)
              || req.getRequestURI().startsWith(req.getContextPath() + "/setup")) {
            filterChain.doFilter(req, resp);
          } else {
            RequestDispatcher rd = servletContext.getRequestDispatcher(AccountInitializationServlet.TMC_INI_ACCOUNT_SETUP_URL);
            rd.forward(req, resp);
          }
        } else {

          if (!isUserConsideredAnAdmin && req.getRequestURI().startsWith(req.getContextPath() + "/setup")) {
            resp.sendError(HttpServletResponse.SC_UNAUTHORIZED);
          } else {
            filterChain.doFilter(req, resp);
          }

        }
      } else if (HAS_LICENSE && !authenticationEnabled && tmcNeedsToRestart(authenticationEnabled)) {
        if (req.getRequestURI().startsWith(req.getContextPath() + TMC_RESTART)) {
          filterChain.doFilter(req, resp);
        } else {
          RequestDispatcher rd = servletContext.getRequestDispatcher(TMC_RESTART);
          rd.forward(req, resp);
        }
      } else {
        filterChain.doFilter(req, resp);
      }
    }

  }

  @Override
  public void destroy() {
    // no-op
  }

  /**
   * Checks whether or not the tms/tmc is in a valid state If the user changed the authentication status, and the tmc did not restart, this method
   * will return true
   * 
   * @return
   */
  private boolean tmcNeedsToRestart(Boolean authenticationEnabled) {
    if (authenticationEnabled && securityCtxtMgr == null || securityCtxtMgr != null && !authenticationEnabled) {
      return true;
    }
    return false;
  }

}
