/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.servlet;

import com.terracotta.management.services.SystemConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;
import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.services.Utils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * This servlet writes the TMSTrusStore use property in the system settings (settings.ini)
 * If this property is set to true, then if the *stores and the keychain are present, we use them to connect to Https Agents
 *
 * @author Anthony Dahanne
 */
public final class ConfigureTmsTrustStoreServlet extends HttpServlet {

  public static final String TMC_HOME_URL = "/";

  public static final String TMC_RESTART_URL = "/restart.jsp";

  private static final Logger LOG = LoggerFactory.getLogger(ConfigureTmsTrustStoreServlet.class);

  private final SystemConfigService systemConfigService;

  private static final String USE_TMS_TRUSTSTORE = "useTmsTruststore";
  private static final String USE_TMS_TRUSTSTORE_HIDDEN = "useTmsTruststoreHidden";

  public ConfigureTmsTrustStoreServlet() {
    systemConfigService = ServiceLocator.locate(SystemConfigService.class);
  }

  protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {

    String useTmsTrustStoreString = req.getParameter(USE_TMS_TRUSTSTORE);
    String useTmsTrustStoreHiddenField = req.getParameter(USE_TMS_TRUSTSTORE_HIDDEN);
    if (Utils.trimToNull(useTmsTrustStoreHiddenField) == null) {
      LOG.debug("useTmsTrustStore was null");
      // error redirect to home
      resp.sendError(500,"The form was not submitted properly");
    } else {
      boolean tmsTrustStore = useTmsTrustStoreString != null && useTmsTrustStoreString.equals("on");
      if(tmsTrustStore == systemConfigService.isTmsTruststoreUsedForHttpsAgents()) {
        //nothing changed, go back to home
        resp.sendRedirect(req.getContextPath() + TMC_HOME_URL);
      } else {
        //we save the new setting and we ask for a tms restart
        try {
          systemConfigService.setAndSaveTmsTrustoreUsedForHttpsAgents(tmsTrustStore);
          resp.sendRedirect(req.getContextPath() + TMC_RESTART_URL);
        } catch (ServiceExecutionException e) {
          LOG.debug("Could not save configuration",e);
          resp.sendError(500,"the configuration could not be saved");
        }
      }
    }
  }

}
