/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.servlet;

import com.terracotta.management.security.shiro.configuration.ShiroConfigurationGenerator;
import com.terracotta.management.security.shiro.realm.LdapConfigurationChecker;
import com.terracotta.management.security.shiro.realm.LdapConfigurationException;
import com.terracotta.management.services.SystemConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;
import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.services.Utils;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.HashSet;
import java.util.Set;

/**
 * This servlet writes the shiro ldap or ini configuration from the user input OR disable authentication
 *
 * @author Anthony Dahanne
 */
public final class ConfigureAuthServlet extends HttpServlet {

  private static final String NONE = "NONE";

  private static final String INI = "INI";

  private static final String LDAP = "LDAP";

  private static final String AD = "AD";

  public static final String AUTHENTICATION_TYPE = "AUTHENTICATION_TYPE";

  public static final String SOME_FIELDS_ARE_EMPTY = "SOME_FIELDS_ARE_EMPTY";

  public static final String DISPLAY_LDAP_SETUP = "DISPLAY_LDAP_SETUP";

  public static final String DISPLAY_AD_SETUP = "DISPLAY_AD_SETUP";

  public static final String TMC_SETUP_URL = "/setup/accountSetup.jsp";

  public static final String TMC_RESTART_URL = "/restart.jsp";

  private static final String TMC_AUTHENTICATION = "/setup/authenticationSetup.jsp";

  private static final Logger LOG = LoggerFactory.getLogger(ConfigureAuthServlet.class);

  private static final String ERR_MSG = "Some fields are empty, please check you provided the needed information!";

  private static final String OPERATOR_GROUP = "operatorGroup";

  private static final String ADMIN_GROUP = "adminGroup";

  private static final String USER_DN_TEMPLATE = "userDnTemplate";

  private static final String SEARCH_BASE = "searchBase";

  private static final String URL = "url";

  private static final String SYSTEM_USERNAME = "systemUsername";

  private static final String DYNAMIC_GROUP_CONFIGURATION = "dynamicGroupConfiguration";

  private static final String GROUP_ATTRIBUTE_MATCHING = "groupAttributeMatching";

  public static final String GROUP_DN_TEMPLATE = "groupDnTemplate";

  private final SystemConfigService systemConfigService;

  public ConfigureAuthServlet() {
    systemConfigService = ServiceLocator.locate(SystemConfigService.class);
  }

  @Override
  protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
    RequestDispatcher rd = req.getSession().getServletContext().getRequestDispatcher(TMC_AUTHENTICATION);
    rd.forward(req, resp);
  }

  protected void doPost(HttpServletRequest req,
                        HttpServletResponse resp) throws ServletException, IOException {

    String authenticationType = req.getParameter(AUTHENTICATION_TYPE);
    if (Utils.trimToNull(authenticationType) == null) {
      // error redirect to authenticationSetup
      resp.sendRedirect(req.getContextPath() + TMC_SETUP_URL);
    } else if (authenticationType.equals(AD) || authenticationType.equals(LDAP)) {
      String operatorGroupsAsString = Utils.trimToNull(req.getParameter(OPERATOR_GROUP));
      Set<String> operatorGroups = getGroups(operatorGroupsAsString);
      String adminGroupsAsString = Utils.trimToNull(req.getParameter(ADMIN_GROUP));
      Set<String> adminGroups = getGroups(adminGroupsAsString);
      String searchBase = Utils.trimToNull(req.getParameter(SEARCH_BASE));
      String url = Utils.trimToNull(req.getParameter(URL));
      String systemUsername = Utils.trimToNull(req.getParameter(SYSTEM_USERNAME));
      String userDnTemplate = Utils.trimToNull(req.getParameter(USER_DN_TEMPLATE));
      String groupDnTemplate = Utils.trimToNull(req.getParameter(GROUP_DN_TEMPLATE));
      String dynamicGroupConfiguration = Utils.trimToNull(req.getParameter(DYNAMIC_GROUP_CONFIGURATION));
      String groupAttributeMatching = Utils.trimToNull(req.getParameter(GROUP_ATTRIBUTE_MATCHING));

      req.setAttribute(OPERATOR_GROUP, operatorGroupsAsString);
      req.setAttribute(ADMIN_GROUP, adminGroupsAsString);
      req.setAttribute(SEARCH_BASE, searchBase);
      req.setAttribute(URL, url);
      req.setAttribute(SYSTEM_USERNAME, systemUsername);
      req.setAttribute(USER_DN_TEMPLATE, userDnTemplate);
      req.setAttribute(GROUP_DN_TEMPLATE, groupDnTemplate);
      req.setAttribute(DYNAMIC_GROUP_CONFIGURATION, dynamicGroupConfiguration);
      req.setAttribute(GROUP_ATTRIBUTE_MATCHING, groupAttributeMatching);

      if (operatorGroups.isEmpty() || adminGroups.isEmpty() ||  isOneOfThemEmpty(searchBase, url) && (authenticationType.equals(AD) || ((authenticationType.equals(LDAP) && isOneOfThemEmpty(userDnTemplate, groupDnTemplate, groupAttributeMatching))))) {
        wrapErrorIntoRequest(req, resp, ERR_MSG, authenticationType.equals(LDAP));
        return;
      } else {
        req.removeAttribute(SOME_FIELDS_ARE_EMPTY);
      }

      try {
        LdapConfigurationChecker.connectAndCheckConfiguration(url, searchBase, systemUsername, operatorGroups, adminGroups, userDnTemplate, groupDnTemplate, groupAttributeMatching, !Boolean.parseBoolean(dynamicGroupConfiguration));
      } catch (LdapConfigurationException e) {
        String errorMessage = e.getMessage() + ((e.getCause() != null && e.getCause().getMessage() != null) ? "<br />Details : " + e.getCause().getMessage() : "");
        wrapErrorIntoRequest(req, resp, errorMessage, authenticationType.equals(LDAP));
        return;
      }

      if (authenticationType.equals(AD)) {
        try {
          ShiroConfigurationGenerator.writeShiroConfigurationActiveDirectory(operatorGroups, adminGroups, searchBase, url,
                  systemUsername);
          systemConfigService.setAuthenticationEnabled(true);
        } catch (Exception e) {
          LOG.error("Failed to setup active directory configuration!", e);
        }
      } else {
        try {
          ShiroConfigurationGenerator.writeShiroConfigurationLdap(operatorGroups, adminGroups, userDnTemplate,groupDnTemplate, searchBase, url,
                  systemUsername, Boolean.parseBoolean(dynamicGroupConfiguration), groupAttributeMatching);
          systemConfigService.setAuthenticationEnabled(true);
        } catch (Exception e) {
          LOG.error("Failed to setup ldap configuration!", e);
        }
      }

    } else if (authenticationType.equals(INI)) {
      try {
        ShiroConfigurationGenerator.writeShiroConfigurationIniFile();
        systemConfigService.setAuthenticationEnabled(true);
      } catch (Exception e) {
        LOG.error("Failed to setup shiro ini configuration!", e);
      }
    } else if (authenticationType.equals(NONE)) {
      try {
        systemConfigService.setAuthenticationEnabled(false);
      } catch (ServiceExecutionException e) {
        LOG.error("Failed to disable authentication!", e);
      }
    }
    resp.sendRedirect(req.getContextPath() + TMC_RESTART_URL);

  }

  private Set<String> getGroups(String groupsAsString) {
    String[] splitGroups = groupsAsString.split(",");
    Set<String> groups = new HashSet<String>();
    for (String group : splitGroups) {
      String trimmedGroup = Utils.trimToNull(group);
      if(trimmedGroup != null ) {
        groups.add(trimmedGroup);
      }
    }
    return groups;
  }


  private void wrapErrorIntoRequest(HttpServletRequest req, HttpServletResponse resp, String errorMessage, boolean isLdap) throws ServletException, IOException {
    RequestDispatcher rd;
    req.setAttribute(SOME_FIELDS_ARE_EMPTY, errorMessage);
    if(isLdap) {
      req.setAttribute(DISPLAY_LDAP_SETUP, true);
    } else {
      req.setAttribute(DISPLAY_AD_SETUP, true);
    }
    rd = req.getSession().getServletContext().getRequestDispatcher(TMC_AUTHENTICATION);
    rd.forward(req, resp);
  }

  private boolean isOneOfThemEmpty(String... parameters) {
    for (String parameter : parameters) {
      if (parameter == null) {
        return true;
      }
    }
    return false;
  }

}
