/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.services.impl;

import com.sun.jersey.api.client.*;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.json.JSONConfiguration;
import com.terracotta.management.AgentsUtils;
import com.terracotta.management.AggregateCallback;
import com.terracotta.management.AggregateCollectionCallback;
import com.terracotta.management.config.Agent;
import com.terracotta.management.resource.*;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.security.IACredentials;
import com.terracotta.management.services.ConfigService;
import com.terracotta.management.services.JerseyClientFactory;
import com.terracotta.management.services.ResourceServiceClientService;
import net.sf.ehcache.management.resource.CacheEntity;
import net.sf.ehcache.management.resource.CacheManagerEntity;
import net.sf.ehcache.management.resource.CacheStatisticSampleEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;
import org.terracotta.management.embedded.StandaloneServer;
import org.terracotta.management.resource.AgentEntity;
import org.terracotta.management.resource.AgentMetadataEntity;
import org.terracotta.management.resource.Representable;
import org.terracotta.management.resource.VersionedEntity;
import org.terracotta.management.resource.exceptions.ResourceRuntimeException;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.*;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import static com.terracotta.management.services.JerseyClientFactory.*;

/**
 * <p>
 * An implementation of the {@link ResourceServiceClientService} using Jersey Client.
 * </p>
 *
 * @author brandony
 */
public final class JerseyResourceServiceClientService implements ResourceServiceClientService, Observer {
  private static final String HTTPS_SCHEME = "https://";

  private static final String URI_CTXT_SEP = "/";

  private static final Logger LOG = LoggerFactory.getLogger(JerseyResourceServiceClientService.class);

  private final ConfigService configSvc;

  private final JerseyClientFactory clientFactory;

  private final Authorizer authorizer;

  private final ConcurrentHashMap<String, ConcurrentHashMap<String, Client>> clientsByUsernameAndId;

  public JerseyResourceServiceClientService(ConfigService configSvc,
                                            JerseyClientFactory clientFactory,
                                            Authorizer authorizer) {
    this.configSvc = configSvc;

    this.clientFactory = clientFactory;

    this.authorizer = authorizer;

    Collection<Agent> knownAgents = configSvc.getAgents();

    ConcurrentHashMap<String, ConcurrentHashMap<String, Client>> clientsByUsernameAndId = new ConcurrentHashMap<String, ConcurrentHashMap<String, Client>>();
    
    ConcurrentHashMap<String, Client> clientsByIds;
    for (Agent a : knownAgents) {
      if (clientsByUsernameAndId.get(a.getUsername()) == null) {
        clientsByIds = new ConcurrentHashMap<String, Client>();
      } else {
        clientsByIds = clientsByUsernameAndId.get(a.getUsername());
      }
      clientsByIds.put(a.getId(), prepareClient(a));
      clientsByUsernameAndId.put(a.getUsername(), clientsByIds);
    }

    this.clientsByUsernameAndId = clientsByUsernameAndId;
    this.configSvc.registerObserver(this);
  }


  @Override
  public void addClient(Agent a) {
    ConcurrentHashMap<String, Client> clientsByIds;
    a.setUsername(authorizer.getPrincipal());
    if (clientsByUsernameAndId.get(a.getUsername()) == null) {
      clientsByIds = new ConcurrentHashMap<String, Client>();
    } else {
      clientsByIds = clientsByUsernameAndId.get(a.getUsername());
    }
    clientsByIds.put(a.getId(), prepareClient(a));
    clientsByUsernameAndId.put(a.getUsername(), clientsByIds);
  }


  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getKnownAgentIds() {
    checkCurrentUserHasValidConfigAndLoadItOtherwise();
    return Collections.unmodifiableSet(hideProbeAgentFromTheClient(clientsByUsernameAndId.get(authorizer.getPrincipal())));
  }

  private void checkCurrentUserHasValidConfigAndLoadItOtherwise() {
    if (clientsByUsernameAndId.get(authorizer.getPrincipal()) == null) {
      try {
        this.configSvc.saveConfig(authorizer.getPrincipal());
      } catch (ServiceExecutionException e) {
        LOG.error("Impossible to load user defaults settings", e);
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  // TODO: consolidate the common code between the proxyGet methods.
  @Override
  public <RESOURCE extends VersionedEntity> void proxyGet(AggregateCallback<RESOURCE> callback,
                                                          URI request,
                                                          String[] agentIds,
                                                          Class<RESOURCE> entityClass) {
    checkCurrentUserHasValidConfigAndLoadItOtherwise();
    Map<String, Future<RESOURCE>> futures = new HashMap<String, Future<RESOURCE>>(clientsByUsernameAndId.get(authorizer.getPrincipal()).size());

    Collection<String> ids = agentIds == null ? clientsByUsernameAndId.get(authorizer.getPrincipal()).keySet() : Arrays.asList(agentIds);

    for (String id : ids) {
      String[] idParts = id.split("\\#");
      Client client = clientsByUsernameAndId.get(authorizer.getPrincipal()).get(idParts[0]);
      if (client != null) {
        String realAgentId = idParts.length > 1 ? idParts[1] : null;
        URI location = buildURI(client, request, realAgentId);
        AsyncWebResource resource = client.asyncResource(location);

        Future<RESOURCE> asyncCalc;
        try {
          asyncCalc = resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId()).get(entityClass);
        } catch (UniformInterfaceException e) {
          LOG.warn("Failure setting up async computation.", e);
          continue;
        }

        futures.put(id, asyncCalc);
      }
    }

    // Using the future as a latch for processing the request to the embedded resource.
    for (Entry<String, Future<RESOURCE>> entry : futures.entrySet()) {

      try {
        RESOURCE entity = entry.getValue().get();
        String[] idParts = entry.getKey().split("\\#");
        entity.setAgentId(idParts[0] + "#" + entity.getAgentId());

        callback.addResponse(entity);
      } catch (InterruptedException e) {
        LOG.warn("Future interrupted.", e);
      } catch (ExecutionException e) {
        logAgentExecutionException(e);
        callback.addException(e);
      }
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <RESOURCE extends Representable> void proxyGet(AggregateCollectionCallback<RESOURCE> callback,
                                                        URI request,
                                                        String[] agentIds,
                                                        Class<RESOURCE> entityClass) {
    checkCurrentUserHasValidConfigAndLoadItOtherwise();
    ConcurrentHashMap<String, Client> clientsById = clientsByUsernameAndId.get(
            authorizer.getPrincipal());
    if(clientsById == null) {
      return;
    }
    Map<String, Future<Collection<RESOURCE>>> futures = new HashMap<String, Future<Collection<RESOURCE>>>(
        clientsById.size());

    Collection<String> ids;
    if (agentIds == null) {
      ids = hideProbeAgentFromTheClient(clientsById);
    }
    else {
      ids = Arrays.asList(agentIds);
    }

    for (String id : ids) {
      String[] idParts = id.split("\\#");

      Client client = clientsById.get(idParts[0]);
      if (client != null) {
        String realAgentId = idParts.length > 1 ? idParts[1] : null;
        URI location = buildURI(client, request, realAgentId);
        AsyncWebResource resource = client.asyncResource(location);

        Future<Collection<RESOURCE>> asyncCalc;
        try {
          asyncCalc = resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
              .get(buildGenericType(entityClass));
        } catch (UniformInterfaceException e) {
          LOG.warn("Failure setting up async computation.", e);
          continue;
        }

        futures.put(id, asyncCalc);
      }
    }

    String currentAgentId = "";
    // Using the future as a latch for processing the request to the embedded resource.
    for (Entry<String, Future<Collection<RESOURCE>>> entry : futures.entrySet()) {
      try {
        currentAgentId = entry.getKey();
        Collection<RESOURCE> entities = entry.getValue().get();

        for (RESOURCE entity : entities) {
          String[] idParts = entry.getKey().split("\\#");
          entity.setAgentId(idParts[0] + "#" + entity.getAgentId());
        }

        callback.addResponses(entities);
      } catch (InterruptedException e) {
        LOG.warn("Future interrupted.", e);
      } catch (ExecutionException e) {
        if (e.getCause() != null && e.getCause().getCause() != null && e.getCause().getCause() instanceof ConnectException) {
          // the client is down, if we are working with a TSA agent, chances are there may be other agents for the cluster
          Agent agent = configSvc.getAgent(currentAgentId, authorizer.getPrincipal());
          if (agent != null) {
            if (agent.getType() == Agent.TYPE.TSA && agent.getAgentsLocations().size() > 1) {
              String agentLocation = rotateAgentLocationUrls(agent);
              agent.setAgentLocation(agentLocation);
              Client prepareClient = prepareClient(agent);
              clientsById.put(currentAgentId, prepareClient);
              LOG.debug("Using now " + agent.getAgentsLocations().get(0) + " as agent url for " + currentAgentId);
            } else {
              // a standalone connection ? nothing we can do then...
            }
          }
        }
        logAgentExecutionException(e);
        callback.addException(e);
      }
    }

  }

  private Set<String> hideProbeAgentFromTheClient(ConcurrentHashMap<String, Client> clientsById) {
    Set<String> ids = new HashSet<String>(clientsById.keySet());
    ids.remove("probeGroup_%_%_probeClient#embedded");
    ids.remove("probeGroup_%_%_probeClient");
    return ids;
  }

  /**
   * Say you have agent.agentLocation="1,2,3,4", this method returns "2,3,4,1"
   */
  protected String rotateAgentLocationUrls(Agent agent) {
    List<String> agentsLocations = new ArrayList<String>(agent.getAgentsLocations());
    Collections.rotate(agentsLocations, -1);
    String agentLocation = "";
    boolean first = true;
    for (String string : agentsLocations) {
      if (first) {
        agentLocation = string;
        first = false;
      } else {
        agentLocation = agentLocation + "," + string;
      }
    }
    return agentLocation;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void proxyDelete(URI request,
                          String agentId) {
    checkCurrentUserHasValidConfigAndLoadItOtherwise();
    String[] agentParts = agentId.split("\\#");
    Client client = clientsByUsernameAndId.get(authorizer.getPrincipal()).get(agentParts[0]);

    if (client != null) {
      String realAgentId = agentParts.length > 1 ? agentParts[1] : null;
      WebResource resource = client.resource(buildURI(client, request, realAgentId));
      try {
        resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId()).delete();
      } catch (UniformInterfaceException e) {
        // forwarding the error from the agent to the user
        throw new WebApplicationException(e.getCause(),
                Response.status(e.getResponse().getStatus())
                        .entity(e.getResponse().getEntity(String.class))
                        .type(e.getResponse().getHeaders().getFirst("Content-Type"))
                        .build());
      }
    } else {
      throw new ResourceRuntimeException("No such connection: " + agentParts[0], Response.Status.GONE.getStatusCode());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <RESOURCE> void proxyPut(URI request,
                                  RESOURCE entity,
                                  String agentId) {
    checkCurrentUserHasValidConfigAndLoadItOtherwise();
    String[] agentParts = agentId.split("\\#");
    Client client = clientsByUsernameAndId.get(authorizer.getPrincipal()).get(agentParts[0]);

    if (client != null) {
      String realAgentId = agentParts.length > 1 ? agentParts[1] : null;
      WebResource resource = client.resource(buildURI(client, request, realAgentId));
      try {
        resource.type(MediaType.APPLICATION_JSON_TYPE).header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
          .put(entity);
      } catch (UniformInterfaceException e) {
        // forwarding the error from the agent to the user
        throw new WebApplicationException(e.getCause(),
                Response.status(e.getResponse().getStatus())
                        .entity(e.getResponse().getEntity(String.class))
                        .type(e.getResponse().getHeaders().getFirst("Content-Type"))
                        .build());
      }
    } else {
      throw new ResourceRuntimeException("No such connection: " + agentParts[0], Response.Status.GONE.getStatusCode());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <RESOURCE> Response proxyPost(URI request,
                                       RESOURCE entity,
                                       String agentId) {
    checkCurrentUserHasValidConfigAndLoadItOtherwise();
    String[] agentParts = agentId.split("\\#");
    Client client = clientsByUsernameAndId.get(authorizer.getPrincipal()).get(agentParts[0]);

    if (client != null) {
      try {
        String realAgentId = agentParts.length > 1 ? agentParts[1] : null;
        URI location = buildURI(client, request, realAgentId);
        WebResource resource = client.resource(location);
        Object postResult = resource
            .type(MediaType.APPLICATION_JSON_TYPE)
            .header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
            .post(Object.class, entity);
        return Response.created(location).entity(postResult).build();
      } catch (UniformInterfaceException e) {
        // forwarding the error from the agent to the user
        throw new WebApplicationException(e.getCause(),
                Response.status(e.getResponse().getStatus())
                        .entity(e.getResponse().getEntity(String.class))
                        .type(e.getResponse().getHeaders().getFirst("Content-Type"))
                        .build());
      }
    } else {
      throw new ResourceRuntimeException("No such connection: " + agentParts[0], Response.Status.GONE.getStatusCode());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(Observable o, Object id) {
    String agentId = String.class.cast(id);
    Agent agent = configSvc.getAgent(agentId, authorizer.getPrincipal());

    // if the user connect for the first time, we have to initialize his list of connections
    if (clientsByUsernameAndId.get(authorizer.getPrincipal()) == null) {
      ConcurrentHashMap<String, Client> clientsByIds = new ConcurrentHashMap<String, Client>();
      clientsByUsernameAndId.put(authorizer.getPrincipal(), clientsByIds);
    }

    if (agent == null)
      clientsByUsernameAndId.get(authorizer.getPrincipal()).remove(agentId);
    else {
      Client current = clientsByUsernameAndId.get(authorizer.getPrincipal()).get(agentId);

      // if(TMSEnvironmentLoaderListener.HAS_LICENSE) agent.setSecured(true);

      Client newClient = prepareClient(agent);

      URI checkClientURI;

      try {
        checkClientURI = new URI("agents/info");
      } catch (URISyntaxException e) {
        throw new RuntimeException("BUG Alert! Invalid agent info URI.", e);
      }

      URI checkLocation = buildURI(newClient, checkClientURI, null);
      WebResource resource = newClient.resource(checkLocation);

      // boolean available = true;
      Collection<AgentMetadataEntity> ames;
      try{
        ames = (Collection<AgentMetadataEntity>) resource
          .header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
          .get(buildGenericType(AgentMetadataEntity.class));
        if (!ames.isEmpty()) {
          AgentMetadataEntity ame = ames.iterator().next();

          if(agent.update(ame)) {
            newClient = prepareClient(agent);
          }
        }
      } catch (RuntimeException e) {
        // available = false;
      }

      if (current != null) {
        clientsByUsernameAndId.get(authorizer.getPrincipal()).replace(agentId, current, newClient);
      } else {
        clientsByUsernameAndId.get(authorizer.getPrincipal()).putIfAbsent(agentId, newClient);
      }

      // if(!available) {
      // throw new AgentNotAvailableException("REST service not currently available or there is a problem with the configuration.");
      // }
    }
  }

  private void logAgentExecutionException(ExecutionException e) {
    String extraErrorMessage = "";
    if (e.getCause() instanceof UniformInterfaceException) {
      UniformInterfaceException cause = (UniformInterfaceException)e.getCause();

      StringBuilder sb = new StringBuilder();
      sb.append(System.getProperty("line.separator")).append("HTTP(S) response was:");
      sb.append(System.getProperty("line.separator")).append("Status: ")
          .append(cause.getResponse().getStatus());
      sb.append(System.getProperty("line.separator")).append("Content type: ")
          .append(cause.getResponse().getHeaders().getFirst("Content-Type"));
      sb.append(System.getProperty("line.separator")).append("Body:");
      sb.append(System.getProperty("line.separator"));

      BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(cause.getResponse().getEntityInputStream()));
      while (true) {
        try {
          String line = bufferedReader.readLine();
          if (line == null) {
            extraErrorMessage = sb.toString();
            break;
          }
          sb.append(line).append(System.getProperty("line.separator"));
        } catch (IOException ioe) {
          LOG.warn("Unable ro read response", ioe);
          break;
        }
      }

      try {
        cause.getResponse().getEntityInputStream().reset();
      } catch (IOException ioe) {
        LOG.warn("Unable ro reset response", ioe);
      }
    }

    if (LOG.isDebugEnabled()) {
      LOG.warn("Failed to execute request against monitorable entity. " + extraErrorMessage, e);
    } else {
      LOG.warn("Failed to execute request against monitorable entity. " + extraErrorMessage, e.getMessage());
    }
  }

  private static URI buildURI(Client client, URI request, String agentId) {
    URI baseUri = (URI) client.getProperties().get(CLIENT_BASE_URI);

    if (baseUri == null) {
      throw new RuntimeException("Bug Alert! Invalid aggregator configuration: missing client base URI.");
    }

    String requestPath = request.getPath();
    int idx = requestPath.indexOf("/");
    String endPath = idx > -1 ? requestPath.substring(idx + 1, requestPath.length()) : "";

    UriBuilder builder = UriBuilder.fromUri(baseUri).path("agents");
    if (agentId != null) {
      builder = builder.matrixParam("ids", agentId);
    }
    builder.path(endPath);

    String query = request.getQuery();
    if (query != null) {
      builder.replaceQuery(query);
    }

    return builder.build();
  }

  private Client prepareClient(Agent a) {
    ClientConfig clientConfig = new DefaultClientConfig();
    clientConfig.getFeatures().put(JSONConfiguration.FEATURE_POJO_MAPPING, Boolean.TRUE);

    Integer cto = a.getConnectionTimeoutMillis();
    if (cto != null) clientConfig.getProperties().put(ClientConfig.PROPERTY_CONNECT_TIMEOUT, cto);

    Integer rto = a.getReadTimeoutMillis();
    if (rto != null) clientConfig.getProperties().put(ClientConfig.PROPERTY_READ_TIMEOUT, rto);

    URI clientBase;
    try {
      clientBase = extractBaseURI(a);
    } catch (URISyntaxException e) {
      throw new RuntimeException("BUG Alert! Failed to extract valid URI from agent while setting up client. ");
    }

    clientConfig.getProperties().put(CLIENT_BASE_URI, clientBase);

    clientConfig.getProperties()
        .put(CLIENT_CERT_AUTH_ENABLED, clientBase.toString().startsWith(HTTPS_SCHEME) ? a.isClientAuthEnabled() : false);
    clientConfig.getProperties().put(SECURITY_ENABLED, a.isSecured());

    return clientFactory.createClient(clientConfig);
  }

  private static URI extractBaseURI(Agent agent) throws URISyntaxException {
    URL agentURL;
    try {
      if (agent.getType().equals(Agent.TYPE.TSA)) {
        agentURL = new URL(AgentsUtils.checkTsaAgentLocationsAndReturnFirstValidOne(agent));
      } else {
        agentURL = new URL(agent.getAgentLocation());
      }
    } catch (MalformedURLException e) {
      throw new RuntimeException("BUG Alert! Failed to retrieve agent location URL while setting up client.");
    }

    String path = agentURL.getPath();

    if (!path.matches(".*" + StandaloneServer.EMBEDDED_CTXT + "/$")) {
      path += StandaloneServer.EMBEDDED_CTXT + URI_CTXT_SEP;
    } else if (!path.endsWith(URI_CTXT_SEP)) {
      path += URI_CTXT_SEP;
    }

    return new URI(agentURL.getProtocol() + "://" + agentURL.getAuthority() + path);
  }

  private static GenericType buildGenericType(Class<? extends Representable> clazz) {
    if (clazz == AgentEntity.class) return new GenericType<Collection<AgentEntity>>() {
    };
    if (clazz == AgentMetadataEntity.class) return new GenericType<Collection<AgentMetadataEntity>>() {
    };
    if (clazz == CacheManagerEntity.class) return new GenericType<Collection<CacheManagerEntity>>() {
    };
    if (clazz == CacheEntity.class) return new GenericType<Collection<CacheEntity>>() {
    };
    if (clazz == CacheStatisticSampleEntity.class) return new GenericType<Collection<CacheStatisticSampleEntity>>() {
    };
    if (clazz == ProbeEntity.class) return new GenericType<Collection<ProbeEntity>>() {
    };
    if (clazz == TopologyEntity.class) return new GenericType<Collection<TopologyEntity>>() {
    };
    if (clazz == StatisticsEntity.class) return new GenericType<Collection<StatisticsEntity>>() {
    };
    if (clazz == ClientEntity.class) return new GenericType<Collection<ClientEntity>>() {
    };
    if (clazz == ConfigEntity.class) return new GenericType<Collection<ConfigEntity>>() {
    };
    if (clazz == ThreadDumpEntity.class) return new GenericType<Collection<ThreadDumpEntity>>() {
    };
    if (clazz == BackupEntity.class) return new GenericType<Collection<BackupEntity>>() {
    };
    if (clazz == LogEntity.class) return new GenericType<Collection<LogEntity>>() {
    };
    if (clazz == OperatorEventEntity.class) return new GenericType<Collection<OperatorEventEntity>>() {
    };
    else throw new UnsupportedOperationException(
        String.format("Unknown entity class \"%s\" for GenericType construction.", clazz.getCanonicalName()));
  }
}