/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package com.terracotta.management.services.impl;

import com.terracotta.management.config.SystemConfig;
import com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener;
import com.terracotta.management.services.SystemConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Properties;


/**
 * @author Anthony Dahanne
 * 
 * This service is used to interact with the system settings (persisted in settings.ini)
 * 
 */
public class FileSystemConfigService implements SystemConfigService {

  private static final String IS_FIRST_RUN                         = "firstRun";
  private static final String AUTHENTICATION_ENABLED = "authenticationEnabled";
  private static final String USE_TMS_TRUST_STORE_FOR_HTTPS_AGENTS = "useTmsTrustStoreForHttpsAgents";
  private static final Logger LOG = LoggerFactory.getLogger(FileSystemConfigService.class);
  public static final String  TMC_CONFIGURATION_DIRECTORY_PROPERTY = "com.tc.management.config.directory";
  private static final String TMC_CONFIGURATION_DEFAULT_DIRECTORY = System.getProperty("user.home")+System.getProperty("file.separator")+".tc"+System.getProperty("file.separator")+"mgmt"+System.getProperty("file.separator");
  private final File systemConfigFile;
  
  private final SystemConfig  systemConfig                         = new SystemConfig();
  private final Properties                    defaultProps;

  public FileSystemConfigService() {

    systemConfigFile = new File(System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY,TMC_CONFIGURATION_DEFAULT_DIRECTORY) + "settings.ini" );
    defaultProps = new Properties();

    //if the systemConfigFile does not exist, we create it and initialize it
    if(!systemConfigFile.exists()) {
      systemConfigFile.getParentFile().mkdirs();
      try {
        saveSystemConfigFile();
      } catch (IOException e) {
        throw new RuntimeException("Impossible to initialize the system settings persistence file, set to "+ systemConfigFile.getAbsolutePath(), e);
      }
    }
    try {
      loadSystemConfigFromFile();
    } catch (IOException e) {
      throw new RuntimeException("Impossible to interact the system settings persistence file, set to "+ systemConfigFile.getAbsolutePath(), e);
    }
        
  }

  private void loadSystemConfigFromFile() throws IOException {
    FileInputStream in = null;
    try {
      in = new FileInputStream(systemConfigFile);
      defaultProps.load(in);
    } finally {
      if (in != null) {
        in.close();
      }
    }
    systemConfig.setAuthenticationEnabled(Boolean.parseBoolean(defaultProps.getProperty(AUTHENTICATION_ENABLED)));
    systemConfig.setUseTmsTrustore(Boolean.parseBoolean(defaultProps.getProperty(USE_TMS_TRUST_STORE_FOR_HTTPS_AGENTS)));
    systemConfig.setFirstRun(Boolean.parseBoolean(defaultProps.getProperty(IS_FIRST_RUN)));
  }


  @Override
  public synchronized boolean isFirstRun() {
    return systemConfig.isFirstRun();
  }

  @Override
  /**
   * Check whether or not SSL is configured with both a keystore and a truststore
   *
   * @return true if SSL is configured with both a keystore and a truststore
   */
  public boolean storesAndKeychainExist() {

    File tmsKeyStore = new File(System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY, TMC_CONFIGURATION_DEFAULT_DIRECTORY) + "tms-keystore");
    File tmsTrustStore = new File(System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY, TMC_CONFIGURATION_DEFAULT_DIRECTORY) + "tms-truststore");
    File keychain = new File(System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY, TMC_CONFIGURATION_DEFAULT_DIRECTORY) + "keychain");

    return tmsKeyStore.exists() && tmsTrustStore.exists() && keychain.exists();
  }

  @Override
  public String getTmcConfigurationDirectory() {
    return System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY,TMC_CONFIGURATION_DEFAULT_DIRECTORY);
  }


  @Override
  public synchronized boolean isAuthenticationEnabled() {
    return TMSEnvironmentLoaderListener.HAS_LICENSE && systemConfig.isAuthenticationEnabled();
  }

  @Override
  public boolean isTmsTruststoreUsedForHttpsAgents() {
    return systemConfig.isUseTmsTruststore();
  }

  @Override
  public synchronized void setAuthenticationEnabled(boolean authenticationEnabled) throws ServiceExecutionException {
    //if this is the first run, we try to enable the use of the tmstruststore by default
    if(systemConfig.isFirstRun() && storesAndKeychainExist()) {
      systemConfig.setUseTmsTrustore(true);
    }
    systemConfig.setAuthenticationEnabled(authenticationEnabled);
    systemConfig.setFirstRun(false);
    try {
      saveSystemConfigFile();
    } catch (IOException e) {
      LOG.error("Impossible to persist system settings to settings.xml", e);
      throw new ServiceExecutionException(e);
    }
  }

  @Override
  public void setAndSaveTmsTrustoreUsedForHttpsAgents(boolean useTmsTrustore) throws ServiceExecutionException {
    systemConfig.setUseTmsTrustore(useTmsTrustore);
    try {
      saveSystemConfigFile();
    } catch (IOException e) {
      LOG.error("Impossible to persist system settings to settings.xml", e);
      throw new ServiceExecutionException(e);
    }
  }

  private void saveSystemConfigFile() throws IOException {
    defaultProps.put(IS_FIRST_RUN, Boolean.toString(systemConfig.isFirstRun()));
    defaultProps.put(AUTHENTICATION_ENABLED, Boolean.toString(systemConfig.isAuthenticationEnabled()));
    defaultProps.put(USE_TMS_TRUST_STORE_FOR_HTTPS_AGENTS, Boolean.toString(systemConfig.isUseTmsTruststore()));
    FileOutputStream out = new FileOutputStream(systemConfigFile);
    defaultProps.store(out, "TMC system settings");
    out.close();
  }
}
