/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.services.impl;

import com.terracotta.management.config.Agent;
import com.terracotta.management.config.Config;
import com.terracotta.management.services.ConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;

import javax.xml.bind.JAXBException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observer;
import java.util.Set;

/**
 * @author brandony
 */
public final class FileConfigService implements ConfigService {
  private static final Logger LOG = LoggerFactory.getLogger(ConfigService.class);

  private static final String TMC_CONFIGURATION_DIRECTORY_PROPERTY = "com.tc.management.config.directory";
  private static final String TMC_CONFIGURATION_DEFAULT_DIRECTORY  = System.getProperty("user.home") + "/.tc/mgmt";

  private static String TMC_CONFIGURATION_DIRECTORY = System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY) == null ? TMC_CONFIGURATION_DEFAULT_DIRECTORY
      : System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY);

  private static File TMC_USER_SERVER_PREFERENCES_DIRECTORY = new File(TMC_CONFIGURATION_DIRECTORY + "/server/");
  static {
    if (!TMC_USER_SERVER_PREFERENCES_DIRECTORY.exists()) {
      TMC_USER_SERVER_PREFERENCES_DIRECTORY.mkdirs();
    }
  }
  
  private final Map<String, Config> configs = new HashMap<String, Config>();

  private Observer observer;

  public FileConfigService() {
    FilenameFilter filesEndingWithDotXmlFilter = new FilenameFilter() {
      @Override
      public boolean accept(File dir, String name) {
        if (name.endsWith(".xml")) {
          return true;
        }
        return false;
      }
    };
    File[] listFiles = TMC_USER_SERVER_PREFERENCES_DIRECTORY.listFiles(filesEndingWithDotXmlFilter);
    for (File file : listFiles) {
      loadConfig(file.getName().substring(0, file.getName().indexOf(".xml")));
    }
  }
  
  void createConfigFromTemplate(String tmpltConfigFileName, File cfgFile, String username) {
    // build the config from template if it does not exist
    LOG.info("Config file does not exists.  Creating one from template: " + tmpltConfigFileName);
    try {
      copyConfigFileFromClassPathResource(tmpltConfigFileName, cfgFile, username);
    } catch (IOException e) {
      // only log, because we'll try more tactics below.
      LOG.warn("Failed to copy template config.", e);
    }
  }

  @Override
  public Agent getAgent(String agentId, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.getAgent(agentId);
  }

  @Override
  public Collection<Agent> getAgents() {
    List<Agent> agents = new ArrayList<Agent>();
    for (Entry<String, Config> configEntry : configs.entrySet()) {
      Set<String> groups = configEntry.getValue().getGroups();
      for (String group : groups) {
        Collection<Agent> agentsByGroup = configEntry.getValue().getAgentsByGroup(group);
        agents.addAll(agentsByGroup);
      }
    }

    return agents;
  }

  @Override
  public Collection<Agent> getAgentsByGroup(String groupId, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.getAgentsByGroup(groupId);
  }

  private Config loadConfig(String username) {
    File configFile = new File(TMC_USER_SERVER_PREFERENCES_DIRECTORY, username + ".xml");


    if (!configFile.exists()) {
      String templateConfigFileName = System.getProperty("com.tc.management.config.tmplt", "templateConfig.xml");
      createConfigFromTemplate(templateConfigFileName, configFile, username);
    }
    Config lConfig = null;
    try {
      lConfig = loadConfigFromFile(configFile, username);
    } catch (IOException e) {
      LOG.warn("Config file not found.  Creating empty one.");
    }

    // use the loaded config, or fall back to empty config
    Config config = lConfig == null ? new Config(username) : lConfig;
    configs.put(username, config);
    if (this.observer != null) {
      config.addObserver(this.observer);
    }
    // for each agent created, we make sure the jersey clients get prepared
    for (String group : config.getGroups()) {
      for (Agent agent : config.getAgentsByGroup(group)) {
        config.agentsChanged(agent.getId());
      }
    }
    return config;
  }

  @Override
  public Collection<String> getGroups(String username) {
    Config config  =  configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.getGroups();
  }

  @Override
  public boolean addGroup(String groupId, String username) {
    Config config  =  configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.addGroup(groupId);
  }

  @Override
  public boolean deleteGroup(String groupId, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.deleteGroup(groupId);
  }

  @Override
  public boolean renameGroup(String oldGroupId, String newGroupId, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.renameGroup(oldGroupId, newGroupId);
  }

  @Override
  public void deleteAgent(String agentId, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    config.deleteAgent(agentId);
  }

  @Override
  public boolean addAgent(Agent agent, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.addAgent(agent);
  }

  @Override
  public boolean updateAgent(String agentId, Agent agent, String username) {
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    return config.updateAgent(agentId, agent);
  }

  @Override
  public boolean saveConfig(String username) throws ServiceExecutionException {
    boolean success = false;
    Config config = configs.get(username) != null ? configs.get(username) : loadConfig(username);
    File configFile = new File(TMC_USER_SERVER_PREFERENCES_DIRECTORY, username + ".xml");
    try {
      storeConfigInFile(config, configFile, username);
    } catch (IOException e) {
      throw new ServiceExecutionException("Failed to save config file.", e);
    } catch (JAXBException e) {
      throw new ServiceExecutionException("Failed to serialize config.", e);
    }

    return success;
  }

  @Override
  public void registerObserver(Observer observer) {
    this.observer = observer;
    // scan already loaded configs, and register the observer on them
    for (Entry<String, Config> configEntry : configs.entrySet()) {
      configEntry.getValue().addObserver(observer);
    }
  }
  
  void storeConfigInFile(Config cfg, File cfgFile, String username) throws IOException, JAXBException {
    
    boolean existingConfig = cfgFile.exists();
    
    File backFile = new File(cfgFile.getAbsolutePath() + "_tmp" + System.currentTimeMillis());
    if(existingConfig) {
      LOG.debug("Backing up config to: '" + backFile.getAbsolutePath() + "'");
      copyConfigFile(cfgFile, backFile, username);
      if (!cfgFile.delete()) {
        throw new IOException("Unable to delete existing configFile '" + cfgFile + "'");
      }
    }
    else {
      if (!cfgFile.getParentFile().exists() && !cfgFile.getParentFile().mkdirs()) {
        throw new IOException("Unable to create path '" + cfgFile.getParentFile() + "'");
      }
    }

    LOG.info("Writing config to: '" + cfgFile.getAbsolutePath() + "'");
    Config.write(cfg, cfgFile);

    if(existingConfig) {
      LOG.debug("Removing back up config: '" + backFile.getAbsolutePath() + "'");
      if (!backFile.delete()) {
        throw new IOException("Unable to delete backup file '" + backFile + "'");
      }
    }
  }

  Config loadConfigFromFile(File src, String username) throws IOException {
    if (src.exists()) {
      FileInputStream fis;

      try {
        fis = new FileInputStream(src);
      } catch (FileNotFoundException e) {
        throw new IOException("Failure loading configuration file: " + src.getAbsolutePath(), e);
      }
      
      return loadConfigFromInputStream(fis, username);
    }
    return null; 
  }

  Config loadConfigFromInputStream(InputStream src, String username) throws IOException {
    try {
      return Config.load(src, username);
    } catch (JAXBException e) {
      throw new IOException("Failure loading configuration input stream.", e);
    } finally {
      try {
        src.close();
      } catch (IOException e1) {
        LOG.warn("Failed to close config file input stream.", e1);
      }
    }
  }

  private void copyConfigFile(File sourceFile, File destFile, String username) throws IOException {
    if(!sourceFile.exists()) {
      throw new IOException("Source file for copy " + sourceFile.getAbsolutePath() + " does not exist.");
    }
    try {
      Config tmpltConfig = loadConfigFromFile(sourceFile, username);
      storeConfigInFile(tmpltConfig, destFile, username);
    } catch(JAXBException e) {
      throw new IOException("Error serializing template config", e);
    }
  }

  private void copyConfigFileFromClassPathResource(String resourceName, File destFile, String username) throws IOException {
    InputStream src = FileConfigService.class.getClassLoader().getResourceAsStream(resourceName);
    try {
      Config tmpltConfig = loadConfigFromInputStream(src, username);
      storeConfigInFile(tmpltConfig, destFile, username);
    } catch(JAXBException e) {
      throw new IOException("Error serializing template config", e);
    }
  }

}
