/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.web.shiro;

import com.sun.jersey.api.client.filter.ClientFilter;
import com.terracotta.license.LicenseManager;
import com.terracotta.management.UpdateChecker;
import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.resource.services.validator.impl.AggregateEhcacheRequestValidator;
import com.terracotta.management.security.*;
import com.terracotta.management.security.impl.*;
import com.terracotta.management.security.services.SecurityContextSetupService;
import com.terracotta.management.security.services.impl.IniFileSetupService;
import com.terracotta.management.security.shiro.ShiroAuthorizer;
import com.terracotta.management.security.shiro.ShiroSecurityContextManager;
import com.terracotta.management.security.shiro.realm.TCIniRealm;
import com.terracotta.management.security.web.impl.LicensedIdentityAsserter;
import com.terracotta.management.security.web.impl.NoOpIdentityAsserter;
import com.terracotta.management.security.web.jersey.TMSRequestClientFilter;
import com.terracotta.management.security.web.jersey.TMSRequestSecurityClientFilter;
import com.terracotta.management.services.ConfigService;
import com.terracotta.management.services.JerseyClientFactory;
import com.terracotta.management.services.ResourceServiceClientService;
import com.terracotta.management.services.SystemConfigService;
import com.terracotta.management.services.impl.DfltJerseyClientFactory;
import com.terracotta.management.services.impl.FileConfigService;
import com.terracotta.management.services.impl.FileSystemConfigService;
import com.terracotta.management.services.impl.JerseyResourceServiceClientService;
import com.terracotta.management.user.UserInfoFactory;
import com.terracotta.management.user.dao.impl.IniFileUserInfoDao;
import com.terracotta.management.user.impl.DfltUserInfoFactory;
import com.terracotta.management.userprofile.dao.UserProfileDao;
import com.terracotta.management.userprofile.dao.impl.XMLFileUserProfileDao;
import org.apache.shiro.mgt.RealmSecurityManager;
import org.apache.shiro.realm.AuthorizingRealm;
import org.apache.shiro.realm.Realm;
import org.apache.shiro.realm.ldap.JndiLdapRealm;
import org.apache.shiro.web.env.EnvironmentLoaderListener;
import org.apache.shiro.web.env.WebEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.license.LicenseException;
import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.services.validator.RequestValidator;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import java.util.Collection;
import java.util.Timer;

/**
 * @author brandony
 */
public final class TMSEnvironmentLoaderListener extends EnvironmentLoaderListener {
  private static final long EVERY_WEEK = 7 * 24 * 60 * 60 * 1000;
  private static final long DELAY_UPDATE_CHECK = 1000;

  public static final String  TMC_CONFIGURATION_DIRECTORY_PROPERTY = "com.tc.management.config.directory";
  private static final String TMC_CONFIGURATION_DEFAULT_DIRECTORY = System.getProperty("user.home")+System.getProperty("file.separator")+".tc"+System.getProperty("file.separator")+"mgmt"+System.getProperty("file.separator");

  public static Boolean TMS_IS_SECURED = false;

  public static Boolean HAS_LICENSE;

  public static Boolean LICENSE_IS_COMMERCIAL_LICENSE;

  private static final Logger LOG = LoggerFactory.getLogger(TMSEnvironmentLoaderListener.class);
  
  private Timer updateCheckTimer;

  @Override
  public void contextInitialized(ServletContextEvent servletContextEvent) {
    LOG.info("Entering TMSEnvironmentLoaderListener.contextInitialized()");
    checkLicense();
    
    SystemConfigService systemConfigService = new FileSystemConfigService();
    UserInfoFactory usrInfoFactory = new DfltUserInfoFactory();
    UserProfileDao usrProfileDao = new XMLFileUserProfileDao();
    this.loadContextAccordingToLicenseAndAuth(servletContextEvent, systemConfigService, usrInfoFactory,usrProfileDao);
    
    scheduleUpdateCheckIfNeeded();
  }

  void loadContextAccordingToLicenseAndAuth(ServletContextEvent servletContextEvent, SystemConfigService systemConfigService, UserInfoFactory usrInfoFactory, UserProfileDao usrProfileDao) {
    if (HAS_LICENSE && systemConfigService.isAuthenticationEnabled()) {
      TMS_IS_SECURED = true;
      LOG.info("TMS license properly loaded, and authentication is enabled : TMS is secured");
      initEnvironment(servletContextEvent.getServletContext());
      loadLicensedCtxt(systemConfigService, usrInfoFactory, usrProfileDao, servletContextEvent.getServletContext());
    } else {
      LOG.info("TMS is NOT secured  : no license found or no authentication enabled, or either");
      loadUnlicensedCtxt(systemConfigService, usrInfoFactory, usrProfileDao);
    }
  }

  @Override
  public void contextDestroyed(ServletContextEvent servletContextEvent) {
    if (HAS_LICENSE) {
      destroyEnvironment(servletContextEvent.getServletContext());
    }
    ServiceLocator.unload();
    
    if (updateCheckTimer != null) {
      updateCheckTimer.cancel();
    }
  }

  private void checkLicense() {
    if (HAS_LICENSE == null) {
      try {
        LicenseManager.verifyTMCCapability();
        HAS_LICENSE = true;
        LICENSE_IS_COMMERCIAL_LICENSE = LicenseManager.isCommercialLicense();
      } catch (LicenseException e) {
        HAS_LICENSE = false;
      }
    }
  }

  private void loadLicensedCtxt(SystemConfigService fileSystemConfigService,
                                UserInfoFactory usrInfoFactory,
                                UserProfileDao usrProfileDao,
                                ServletContext sc) {
    Authorizer authorizer = new ShiroAuthorizer();

    RequestTicketMonitor monitor = new DfltRequestTicketMonitor();

    KeyChainAccessor keyChainAccessor;
    try {
      keyChainAccessor = new SecretFileStoreKeyChainAccessor();
    } catch (KeychainInitializationException e) {
      throw new RuntimeException(
          "Failure instantiating a licensed TMS (security enabled) due to invalid keychain configuration.", e);
    }

    SSLContextFactory sslCtxtFactory;
    // if ssl is configured with a keystore and truststore, we assume they are stored in the tc mgmt dir
    if (fileSystemConfigService.storesAndKeychainExist() && fileSystemConfigService.isTmsTruststoreUsedForHttpsAgents()) {
      LOG.info("SSL configuration with keystore and truststore found, registering a custom sslContextFactory using them");
      sslCtxtFactory = new TMCStoresSSLContextFactory(keyChainAccessor,
              System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY,TMC_CONFIGURATION_DEFAULT_DIRECTORY) + "tms-keystore",
              System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY,TMC_CONFIGURATION_DEFAULT_DIRECTORY) + "tms-truststore");
    } else {
      LOG.info("The system property useTmsTrustStoreForHttpsAgents found in settings.ini is set to false or " +
              "either one of the tms-truststore, the tms-keystore or the keychain were not found in " +
              System.getProperty(TMC_CONFIGURATION_DIRECTORY_PROPERTY,TMC_CONFIGURATION_DEFAULT_DIRECTORY) +
              ", the TMC will use the default JDK truststore when connecting to HTTPS agents");
      sslCtxtFactory = new DefaultSSLContextFactory();
    }

    ClientFilter clientFilter = new TMSRequestSecurityClientFilter(monitor, keyChainAccessor);
    JerseyClientFactory clientFactory = new DfltJerseyClientFactory(clientFilter, sslCtxtFactory);
    

    ConfigService configSvc = new FileConfigService();
    ResourceServiceClientService rsrcSvcClientSvc = new JerseyResourceServiceClientService(configSvc, clientFactory,
        authorizer);

    
    ServiceLocator locator = new ServiceLocator();
    
    try {
      TCIniRealm realm = (TCIniRealm) initializeRealm(sc);
      ShiroSecurityContextManager secCtxtMgr = new ShiroSecurityContextManager(new IniFileUserInfoDao(), realm);
      SecurityContextSetupService secCtxtSetupSvc = new IniFileSetupService(secCtxtMgr, usrInfoFactory);
      RequestIdentityAsserter idAsserter = new LicensedIdentityAsserter(authorizer, monitor, secCtxtMgr, keyChainAccessor);
      locator.loadService(RequestIdentityAsserter.class, idAsserter);
      locator.loadService(SecurityContextManager.class, secCtxtMgr);
      locator.loadService(RequestIdentityAsserter.class, idAsserter);
      locator.loadService(SecurityContextSetupService.class, secCtxtSetupSvc);
      
    } catch (DataAccessException e) {
      throw new RuntimeException("Failure instantiating TMS context because user info datasource could not be found or initialized.", e);
    } catch (ClassCastException e) {
      LOG.debug("The user has configured an Ldap Realm");
      SecurityContextManager secCtxtMgr = new SecurityContextManager() {
        @Override
        public boolean hasValidSecurityContext() {
          return true;
        }
      };
      locator.loadService(SecurityContextManager.class, secCtxtMgr);
    }



    RequestValidator requestValidator = new AggregateEhcacheRequestValidator(authorizer);
    
    locator.loadService(RequestValidator.class, requestValidator)
                    .loadService(ConfigService.class, configSvc)
                    .loadService(ResourceServiceClientService.class, rsrcSvcClientSvc)
                    .loadService(UserProfileDao.class, usrProfileDao)
                    .loadService(Authorizer.class, authorizer)
                    .loadService(KeyChainAccessor.class, keyChainAccessor)
                    .loadService(SystemConfigService.class, fileSystemConfigService);
    ServiceLocator.load(locator);
  }

  private void loadUnlicensedCtxt(SystemConfigService fileSystemConfigService,
                                  UserInfoFactory usrInfoFactory,
                                  UserProfileDao usrProfileDao) {
    Authorizer authorizer = new NoSecurityContextAuthorizer();

    JerseyClientFactory clientFactory = new DfltJerseyClientFactory(new TMSRequestClientFilter(), null);

    ConfigService configSvc = new FileConfigService();

    ResourceServiceClientService rsrcSvcClientSvc = new JerseyResourceServiceClientService(configSvc, clientFactory,
        authorizer);

    RequestIdentityAsserter idAsserter = new NoOpIdentityAsserter(usrInfoFactory);

    RequestValidator requestValidator = new AggregateEhcacheRequestValidator(authorizer);
    
    ServiceLocator locator = new ServiceLocator().loadService(RequestValidator.class, requestValidator)
        .loadService(ConfigService.class, configSvc)
        .loadService(ResourceServiceClientService.class, rsrcSvcClientSvc)
        .loadService(UserProfileDao.class, usrProfileDao)
        .loadService(Authorizer.class, authorizer)
        .loadService(SystemConfigService.class, fileSystemConfigService)
        .loadService(RequestIdentityAsserter.class, idAsserter);
    ServiceLocator.load(locator);
  }

  private AuthorizingRealm initializeRealm(ServletContext sc) {
    AuthorizingRealm realm = null;

    WebEnvironment env = (WebEnvironment) sc.getAttribute(ENVIRONMENT_ATTRIBUTE_KEY);

    Collection<Realm> realms = ((RealmSecurityManager) env.getWebSecurityManager()).getRealms();

    for (Realm r : realms) {
      if (r instanceof TCIniRealm) {
        realm = (TCIniRealm) r;
      } else if (r instanceof JndiLdapRealm) {
        realm = (JndiLdapRealm) r;
        break;
      }
    }


    if (realm == null) {
      throw new RuntimeException("Failure instantiating TMS context. Failure to find expected security realm.");
    } else {
      return realm;
    }
  }
  

  private void scheduleUpdateCheckIfNeeded() {
    if (!Boolean.getBoolean("com.terracotta.management.skipUpdateCheck")) {
      updateCheckTimer = new Timer(true);
      updateCheckTimer.scheduleAtFixedRate(new UpdateChecker(HAS_LICENSE), DELAY_UPDATE_CHECK, EVERY_WEEK);
    }
  }

}
