/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.web.jersey;

import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientRequest;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.filter.ClientFilter;
import com.terracotta.management.keychain.URIKeyName;
import com.terracotta.management.security.HMACBuilder;
import com.terracotta.management.security.IACredentials;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.RequestTicketMonitor;
import com.terracotta.management.services.JerseyClientFactory;

import javax.ws.rs.core.MultivaluedMap;
import java.net.URI;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;

import static com.terracotta.management.services.JerseyClientFactory.CLIENT_BASE_URI;

/**
 * @author brandony
 */
public final class TMSRequestSecurityClientFilter extends ClientFilter {
  private final RequestTicketMonitor monitor;

  private final KeyChainAccessor accessor;

  public TMSRequestSecurityClientFilter(RequestTicketMonitor monitor,
                                        KeyChainAccessor accessor) {
    this.monitor = monitor;
    this.accessor = accessor;
  }

  @Override
  public ClientResponse handle(ClientRequest cr) throws ClientHandlerException {
    boolean securityEnabled = (Boolean) cr.getProperties().get(JerseyClientFactory.SECURITY_ENABLED);

    if (securityEnabled) {
      Object clientCertAuthEnabledAsObject = cr.getProperties().get(JerseyClientFactory.CLIENT_CERT_AUTH_ENABLED);
      boolean useClientCertAuth = clientCertAuthEnabledAsObject != null ? (Boolean) clientCertAuthEnabledAsObject : false;

      MultivaluedMap<String, Object> headers = cr.getHeaders();
      String ticket = monitor.issueRequestTicket();
      headers.putSingle(IACredentials.REQ_TICKET, ticket);

      if (ticket == null) {
        throw new ClientHandlerException("Failed to obtain request ticket for security.");
      }


      if (!useClientCertAuth) {
        String sessId = (String) headers.get(IACredentials.TC_ID_TOKEN).get(0);
        URIKeyName alias = new URIKeyName((URI) cr.getProperties().get(CLIENT_BASE_URI));
        headers.putSingle(IACredentials.ALIAS, alias);

        try {
          byte[] keyMaterial = accessor.retrieveSecret(alias);
          if (keyMaterial == null) {
            throw new RuntimeException("Missing keychain entry for URL [" + alias + "]");
          }
          headers.putSingle("signature",
              HMACBuilder.getInstance(keyMaterial).addMessageComponent(ticket)
                  .addMessageComponent(sessId).addMessageComponent(alias.getURI().toString()).buildEncoded());
        } catch (NoSuchAlgorithmException e) {
          throw new RuntimeException("BUG Alert! Failed to create signed hash.", e);
        } catch (InvalidKeyException e) {
          throw new RuntimeException("BUG Alert! Failed to create signed hash.", e);
        }
      }
    } else {
      cr.getHeaders().remove(IACredentials.TC_ID_TOKEN);
    }

    return getNext().handle(cr);
  }
}
