/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.security.shiro.web.filter;

import org.apache.shiro.authc.AuthenticationException;
import org.apache.shiro.authc.AuthenticationToken;
import org.apache.shiro.codec.Base64;
import org.apache.shiro.subject.Subject;
import org.apache.shiro.web.filter.authc.FormAuthenticationFilter;
import org.apache.shiro.web.util.WebUtils;

import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * An AuthenticationFilter that expects Basic HTTP auth but falls back to a form authentication
 * when Basic Auth can't be performed.
 *
 * This allows writing command-line scripts that use HTTP basic auth while humans using
 * a web browser will get an authentication page.
 *
 * Currently unused, but that might come in handy some time later.
 *
 * @author Ludovic Orban
 */
public class TCBasicWithFormFallbackAuthenticationFilter extends FormAuthenticationFilter {

  private boolean isBasicAuth;

  /*
   * HTTP Authorization header
   */
  private static final String AUTHORIZATION_HEADER = "Authorization";

  /**
   * Perform basic auth if the related HTTP header is set, otherwise perform form auth.
   *
   * @param request  the incoming <code>ServletRequest</code>
   * @param response the outgoing <code>ServletResponse</code>
   * @return <code>true</code> if the request should continue to be processed; false if the subclass will
   *         handle/render the response directly.
   * @throws Exception if there is an error processing the request.
   */
  @Override
  protected boolean onAccessDenied(ServletRequest request, ServletResponse response) throws Exception {
    if (isBasicLoginAttempt(request, response)) {
      isBasicAuth = true;
      if (!executeLogin(request, response)) {
        HttpServletResponse httpResponse = WebUtils.toHttp(response);
        httpResponse.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
        return false;
      } else {
        return true;
      }
    } else {
      isBasicAuth = false;
      return super.onAccessDenied(request, response);
    }
  }

  private boolean isBasicLoginAttempt(ServletRequest request, ServletResponse response) {
    HttpServletRequest httpRequest = WebUtils.toHttp(request);
    String authzHeader = httpRequest.getHeader(AUTHORIZATION_HEADER);
    return authzHeader != null && authzHeader.toLowerCase().startsWith("basic");
  }

  protected boolean sendChallenge(ServletRequest request, ServletResponse response) {
    HttpServletResponse httpResponse = WebUtils.toHttp(response);
    httpResponse.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
    return false;
  }


  @Override
  protected boolean onLoginSuccess(AuthenticationToken token, Subject subject, ServletRequest request, ServletResponse response) throws Exception {
    if (isBasicAuth) {
      return true;
    } else {
      return super.onLoginSuccess(token, subject, request, response);
    }
  }

  @Override
  protected boolean onLoginFailure(AuthenticationToken token, AuthenticationException e, ServletRequest request, ServletResponse response) {
    if (isBasicAuth) {
      return false;
    } else {
      return super.onLoginFailure(token, e, request, response);
    }
  }

  @Override
  protected AuthenticationToken createToken(ServletRequest request, ServletResponse response) {
    if (isBasicAuth) {
      HttpServletRequest httpRequest = WebUtils.toHttp(request);
      String authorizationHeader = httpRequest.getHeader(AUTHORIZATION_HEADER);
      if (authorizationHeader == null || authorizationHeader.length() == 0) {
        // Create an empty authentication token since there is no
        // Authorization header.
        return createToken("", "", request, response);
      }

      String[] prinCred = getPrincipalsAndCredentials(authorizationHeader, request);
      if (prinCred == null || prinCred.length < 2) {
        // Create an authentication token with an empty password,
        // since one hasn't been provided in the request.
        String username = prinCred == null || prinCred.length == 0 ? "" : prinCred[0];
        return createToken(username, "", request, response);
      }

      String username = prinCred[0];
      String password = prinCred[1];

      return createToken(username, password, request, response);
    } else {
      return super.createToken(request, response);
    }
  }

  private String[] getPrincipalsAndCredentials(String authorizationHeader, ServletRequest request) {
    if (authorizationHeader == null) {
      return null;
    }
    String[] authTokens = authorizationHeader.split(" ");
    if (authTokens == null || authTokens.length < 2) {
      return null;
    }
    String decoded = Base64.decodeToString(authTokens[1]);
    return decoded.split(":", 2);
  }

}
