package com.terracotta.management.security.shiro.realm;

import com.terracotta.management.keychain.URIKeyName;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.KeychainInitializationException;
import com.terracotta.management.security.impl.SecretFileStoreKeyChainAccessor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceLocator;

import java.io.UnsupportedEncodingException;
import java.net.URISyntaxException;

/**
 * 
 * Extends the default ContextFactory to add support for decoding the systempassword from the KeyChain
 * 
 * @author Anthony Dahanne
 */
public class TMCJndiLdapContextFactory extends TCJndiLdapContextFactory {

  private static final Logger log = LoggerFactory.getLogger(ActiveDirectoryRealm.class);

  private KeyChainAccessor keyChainAccessor;

  @Override
  public String getSystemPassword() {
    // if there is no system username defined, we don't need to get a system password !
    if (getSystemUsername() == null) {
      return null;
    }
    if(keyChainAccessor ==  null) {
      try {
        keyChainAccessor = ServiceLocator.locate(KeyChainAccessor.class);
      } catch (IllegalStateException e) {
        //service locator not loaded.
      }
      if(keyChainAccessor == null) {
        // ok ok, the ServiceLocator was not configured with a keychain accessor, no problem, we load it here.
        try {
          keyChainAccessor =  new SecretFileStoreKeyChainAccessor();
        } catch (KeychainInitializationException e) {
          throw new RuntimeException("impossible to initialize the keychain",e);
        }
      }
    }

    String aliasFromSystemUsernameAndUrl = getAliasFromSystemUsernameAndUrl(getUrl(), getSimpleSystemUsername());
    byte[] secret;
    String decodedSystemPassword = null;
    try {
      secret = keyChainAccessor.retrieveSecret(new URIKeyName(aliasFromSystemUsernameAndUrl));
      decodedSystemPassword = new String(secret, "UTF-8");
    } catch (URISyntaxException e) {
      log.debug("Impossible to access the keychain !", e);
    } catch (UnsupportedEncodingException e) {
      log.debug("Impossible to read the systempassword for " + aliasFromSystemUsernameAndUrl, e);
    } catch (NullPointerException e) {
      log.debug("Impossible to read the systempassword for " + aliasFromSystemUsernameAndUrl, e);
    }
    return decodedSystemPassword;
    
  }

  String getAliasFromSystemUsernameAndUrl(String url, String systemUsername) {
    String[] split = url.split("://");
    return split[0] + "://" + systemUsername + "@" + split[1];

  }

  public void setKeyChainAccessor(KeyChainAccessor keyChainAccessor) {
    this.keyChainAccessor = keyChainAccessor;
  }

}
