package com.terracotta.management.security.shiro.realm;

import javax.naming.*;
import javax.naming.directory.*;
import javax.naming.ldap.InitialLdapContext;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLHandshakeException;
import java.net.ConnectException;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Set;

/**
 * @author Anthony Dahanne
 */
public class LdapConfigurationChecker {
  public static void connectAndCheckConfiguration(String ldapUrl, String searchBase, String systemUsername, Set<String> operatorGroups, Set<String> adminGroups, String userDnTemplate, String groupDnTemplate, String groupAttributeMatching, boolean isStaticGroupMatching) throws LdapConfigurationException {
    Hashtable<String, String> env = new Hashtable<String, String>();

    env.put(Context.SECURITY_AUTHENTICATION, "simple");
    env.put(Context.PROVIDER_URL, ldapUrl);
    env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
    env.put(Context.REFERRAL, "follow");
    boolean isActiveDirectory = userDnTemplate == null;
    String simpleSystemUsername = systemUsername;
    if(systemUsername != null) {
      TMCJndiLdapContextFactory tmcJndiLdapContextFactory = new TMCJndiLdapContextFactory();
      tmcJndiLdapContextFactory.setSimpleSystemUsername(simpleSystemUsername);
      if( !isActiveDirectory) {
        systemUsername = getDnSystemUsername(userDnTemplate, systemUsername);
      }
      tmcJndiLdapContextFactory.setSystemUsername(systemUsername);

      tmcJndiLdapContextFactory.setUrl(ldapUrl);
      try {
        env.put(Context.SECURITY_CREDENTIALS,tmcJndiLdapContextFactory.getSystemPassword() );
      } catch (NullPointerException e) {
        throw new LdapConfigurationException("Impossible to retrieve systemUsername password from the keychain : "+tmcJndiLdapContextFactory.getAliasFromSystemUsernameAndUrl(ldapUrl,simpleSystemUsername), e);
      }
      env.put(Context.SECURITY_PRINCIPAL, systemUsername );
    }


    if(ldapUrl.startsWith("ldaps")) {
      env.put("java.naming.ldap.factory.socket","com.terracotta.management.security.impl.CustomTrustStoreSSLSocketFactory");
    }


    try {
      DirContext initialLdapContext = new InitialLdapContext(env, null);
      SearchControls searchCtls = new SearchControls();
      searchCtls.setSearchScope(SearchControls.SUBTREE_SCOPE);

      if(isActiveDirectory) {
        //active directory does not need a groupAttributeMatching to be configured
        String searchFilter = "(&(objectClass=*)(CN={0}))";
        Set<String>  mergedGroups = new HashSet<String>();
        mergedGroups.addAll(operatorGroups);
        mergedGroups.addAll(adminGroups);
        for (String group : mergedGroups) {
          checkGroupExists(searchBase, initialLdapContext, searchCtls, searchFilter, group);
        }
      }
      else {
        //Ldap
        if(isStaticGroupMatching) {
          Set<String>  mergedGroups = new HashSet<String>();
          mergedGroups.addAll(operatorGroups);
          mergedGroups.addAll(adminGroups);
          for (String group : mergedGroups) {
            checkGroupExistsLdap(searchBase, initialLdapContext, searchCtls, groupDnTemplate.substring(0, groupDnTemplate.indexOf("=")) + "=" + group, group);
          }
          // no need to run the verification against all groups; just one is enough to assert all of them have the member field
          if(!verifyStaticGroupAttributeMatchingIsValid(initialLdapContext , groupAttributeMatching, (String) adminGroups.toArray()[0], groupDnTemplate)){
            throw new LdapConfigurationException("Impossible to find the groupMatching attribute : "+ groupAttributeMatching +" in the group named : "+ adminGroups.toArray()[0] + " . Are you sure you want to use staticGroupMatching ?");
          }
        } else {
          //we can run controls only if a username is provided !
          if(systemUsername != null) {
            if(! verifyDynamicGroupAttributeMatchingIsValid(initialLdapContext, searchBase, userDnTemplate, simpleSystemUsername, groupAttributeMatching)){
              throw new LdapConfigurationException("Impossible to find the groupMatching attribute : "+ groupAttributeMatching +" in the user entry named : "+systemUsername + " . Are you sure you want to use dynamicGroupMatching ?");
            }
          }
        }

      }

    } catch (NamingException e) {
      if (e.getRootCause() instanceof UnknownHostException) {
        throw new LdapConfigurationException("The host provided in the Ldap URL is not reachable : " + ldapUrl,e);
      } else if (e.getRootCause() instanceof MalformedURLException) {
        throw new LdapConfigurationException("The provided Ldap URL is not valid : " + ldapUrl,e);
      } else if (e.getRootCause() instanceof ConnectException) {
        throw new LdapConfigurationException("The connection was refused : " + ldapUrl,e);
      } else if (e.getRootCause() instanceof SSLHandshakeException) {
        throw new LdapConfigurationException("You can not connect to the secured Ldap server at  : " + ldapUrl + " because you did not import its certificate to your tms-trustore, or you did not ignore certificate errors using -Dtc.ssl.trustAllCerts=true when launching the TMS.",e);
      } else if (e.getRootCause() instanceof SSLException) {
        throw new LdapConfigurationException("The connection was refused, please check your Ldap server accepts SSL connections : "+ldapUrl,e);
      } else if (e instanceof ServiceUnavailableException) {
        throw new LdapConfigurationException("The host provided in the Ldap URL does not accept non secured (ldap, not ldaps) connections : " + ldapUrl,e);
      } else if (e instanceof AuthenticationException) {
        throw new LdapConfigurationException("Cannot authenticate user (did you add your systemUsername password to the keychain ?), please check your systemUsername credentials : " + systemUsername,e);
      } else if (e instanceof InvalidNameException) {
        throw new LdapConfigurationException("Invalid username : " + systemUsername,e);
      }
      throw new LdapConfigurationException("Your LDAP / Active Directory configuration is not valid, please review your configuration, this information might help you : ",e);
    }

  }

  private static void checkGroupExists(String searchBase, DirContext initialLdapContext, SearchControls searchCtls, String searchFilter, String group) throws NamingException, LdapConfigurationException {
    Object[] searchArguments = new Object[]{ group };
    NamingEnumeration answer = initialLdapContext.search(searchBase, searchFilter, searchArguments, searchCtls);
    if (!answer.hasMoreElements()) {
      throw new LdapConfigurationException("We could not find the specified group in the directory : " + group);
    }
  }

  private static void checkGroupExistsLdap(String searchBase, DirContext initialLdapContext, SearchControls searchCtls, String groupWithPrefix, String group) throws NamingException, LdapConfigurationException {
    NamingEnumeration answer = initialLdapContext.search(searchBase, groupWithPrefix, searchCtls);
    if (!answer.hasMoreElements()) {
      throw new LdapConfigurationException("We could not find the specified group in the directory : " + group);
    }
  }


  public static String getDnSystemUsername(String userDnTemplate, String systemUsername) {
    return userDnTemplate.replace("{0}",systemUsername);
  }

  private static boolean verifyStaticGroupAttributeMatchingIsValid(DirContext ldapContext, String groupAttributeMatching, String adminGroup, String groupDnTemplate) throws NamingException {

      Attributes attributes = ldapContext.getAttributes(groupDnTemplate.replace("{0}",adminGroup));
      NamingEnumeration<? extends Attribute> all = attributes.getAll();
      while (all.hasMore()) {
        Attribute next = all.next();
        if (next.getID().equals(groupAttributeMatching)) {
          return true;
        }
      }
    return false;
  }

    private static boolean verifyDynamicGroupAttributeMatchingIsValid(DirContext ldapContext, String searchBase, String userDnTemplate, String username, String groupAttributeMatching) throws NamingException  {
      SearchControls searchCtls = new SearchControls();
      searchCtls.setSearchScope(SearchControls.SUBTREE_SCOPE);
      NamingEnumeration answer = ldapContext.search(searchBase, getUserDnPrefix(userDnTemplate) + username, searchCtls);

      while (answer.hasMoreElements()) {
        SearchResult sr = (SearchResult) answer.next();

        Attributes attrs = sr.getAttributes();

        if (attrs != null) {
          NamingEnumeration ae = attrs.getAll();
          while (ae.hasMore()) {
            Attribute attr = (Attribute) ae.next();
            if (attr.getID().equalsIgnoreCase(groupAttributeMatching)) {
              return true;
            }
          }
        }
      }
    return false;
    }

  private static String getUserDnPrefix(String userDnTemplate) {
    int index = userDnTemplate.indexOf("{0}");
    return userDnTemplate.substring(0, index);
  }


}
