/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.resource.services.impl;

import com.sun.jersey.api.NotFoundException;
import com.sun.jersey.api.client.ClientHandlerException;
import com.sun.jersey.api.client.ClientResponse;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.terracotta.management.AggregateCollectionCallback;
import com.terracotta.management.config.Agent;
import com.terracotta.management.resource.ErrorEntity;
import com.terracotta.management.resource.ProbeEntity;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.resource.AgentEntity;
import org.terracotta.management.resource.AgentMetadataEntity;
import org.terracotta.management.resource.exceptions.ResourceRuntimeException;
import org.terracotta.management.resource.services.AgentsResourceService;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.RuntimeDelegate;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collection;
import java.util.Iterator;

/**
 * <p>
 * An aggregate implementation of {@link AgentsResourceService}.
 * </p>
 *
 * @author brandony
 */
@Path("/agents")
public final class AggregateAgentsResourceService extends AggregateResourceServiceSupport<AgentEntity>
    implements AgentsResourceService {
  private static final Logger LOG = LoggerFactory.getLogger(AggregateAgentsResourceService.class);

  public AggregateAgentsResourceService() {
    super();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<AgentEntity> getAgents(UriInfo info) {
    LOG.info(String.format("Executing AggregateAgentsResourceService.getAgents: %s", info.getRequestUri()));
    getValidator().validateSafe(info);
    AggregateCollectionCallback<AgentEntity> callback = new AggregateCollectionCallback<AgentEntity>() {/**/
    };

    return doGet(info, callback, AgentEntity.class);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Collection<AgentMetadataEntity> getAgentsMetadata(UriInfo info) {
    LOG.info(String.format("Invoking AggregateAgentsResourceService.getAgentsMetadata: %s", info.getRequestUri()));
    getValidator().validateSafe(info);
    String agentIds = info.getPathSegments().get(0).getMatrixParameters().getFirst(ResourceServiceUtils.AGENT_IDS_KEY);
    if (agentIds != null) {
      throw new ResourceRuntimeException("Cannot use ids matrix parameter on /agents/info", Response.Status.BAD_REQUEST.getStatusCode());
    }

    String path = info.getPath();

    AgentMetadataEntityCollectionCallback callback = new AgentMetadataEntityCollectionCallback(
        getSvcClientSvc().getKnownAgentIds());

    try {
      getSvcClientSvc().proxyGet(callback, new URI(path), null, AgentMetadataEntity.class);
    } catch (URISyntaxException e) {
      throw new RuntimeException("Bug Alert! Bad path URI from jersey.", e);
    }

    return callback.calculateResult();
  }

  @GET
  @Path("/probeUrl/{urlToProbe}")
  @Produces(MediaType.APPLICATION_JSON)
  public ProbeEntity getAgentInfo(@PathParam("urlToProbe") String urlToProbe) throws WebApplicationException {

    Agent agent = getNonSecuredAgent(urlToProbe);
    getSvcClientSvc().addClient(agent);

    AggregateCollectionCallback callback = new AggregateCollectionCallback<AgentMetadataEntity>() {};
    try {
      getSvcClientSvc().proxyGet(callback, new URI("/info"),new String[]{agent.getId()}, AgentMetadataEntity.class);
    } catch (URISyntaxException e) {
      throw new RuntimeException("Bug Alert! Bad path URI from jersey.", e);
    }

    Exception exceptionToShow =  null;
    for (Object e : callback.getExceptions()) {
      String message = ((Exception) e).getMessage();
      exceptionToShow = (Exception) e;
      if(message.contains("401 Unauthorized")) {
        //unauthorized ? let's try with IA
        agent = getSecuredAgent(urlToProbe);
        getSvcClientSvc().addClient(agent);
        callback = new AggregateCollectionCallback<AgentMetadataEntity>() {};
        try {
          getSvcClientSvc().proxyGet(callback, new URI("/info"),new String[]{agent.getId()}, AgentMetadataEntity.class);
        } catch (URISyntaxException uriSyntaxException) {
          throw new RuntimeException("Bug Alert! Bad path URI from jersey.", uriSyntaxException);
        }
        if (callback.calculateResult().isEmpty() && !callback.getExceptions().isEmpty()) {
          Exception exception = (Exception) callback.getExceptions().get(0);
          throw new ResourceRuntimeException("Impossible to connect to the agent",exception, Response.Status.NOT_FOUND.getStatusCode());
        }
      } else {
        // there was an exception, we stop trying
        throw new ResourceRuntimeException("Impossible to connect to the agent", exceptionToShow, Response.Status.NOT_FOUND.getStatusCode());
      }
    }

    Iterator ite = callback.calculateResult().iterator();
    AgentMetadataEntity standaloneOrTsaProbeEntity = (AgentMetadataEntity) ite.next();
    if (callback.calculateResult().size() == 1 && standaloneOrTsaProbeEntity.getAgencyOf().equalsIgnoreCase("ehcache")) {
      //this is a connection to a standalone agent
      ProbeEntity probeEntity = new ProbeEntity(standaloneOrTsaProbeEntity);
      probeEntity.setAgentLocation(urlToProbe);
      return probeEntity;
    } else {
      //this is a connection to a tsa agent
      ProbeEntity probeEntity = null;
      if(standaloneOrTsaProbeEntity.getAgencyOf().equalsIgnoreCase("tsa")) {
        probeEntity = returnTsaEntityOrThrowsException(standaloneOrTsaProbeEntity.getAgentId(), standaloneOrTsaProbeEntity);
      }
      while (ite.hasNext()) {
        standaloneOrTsaProbeEntity = (AgentMetadataEntity) ite.next();
        if(standaloneOrTsaProbeEntity.getAgencyOf().equalsIgnoreCase("tsa")) {
          probeEntity = returnTsaEntityOrThrowsException(standaloneOrTsaProbeEntity.getAgentId(), standaloneOrTsaProbeEntity);
        }
      }
      return probeEntity;
    }
  }

  private ProbeEntity returnTsaEntityOrThrowsException(String agentId, AgentMetadataEntity tsaEntity) {
    // we need to recall the tsa, but on /agents only to the agents locations
    AggregateCollectionCallback callback = new AggregateCollectionCallback<AgentMetadataEntity>() {};
    try {
      getSvcClientSvc().proxyGet(callback, new URI("/"),new String[]{agentId}, AgentEntity.class);
    } catch (URISyntaxException e) {
      throw new RuntimeException("Bug Alert! Bad path URI from jersey.", e);
    }
    Iterator ite2 = callback.calculateResult().iterator();
    if (callback.calculateResult().isEmpty() && !callback.getExceptions().isEmpty()) {
      Exception exception = (Exception) callback.getExceptions().get(0);
      throw new ResourceRuntimeException("Impossible to connect to the TSA agent",exception, Response.Status.NOT_FOUND.getStatusCode());
    }
    if (ite2.hasNext()) {
      AgentEntity tsaAgentEntity = (AgentEntity) ite2.next();
      ProbeEntity probeEntity = new ProbeEntity(tsaEntity);
      probeEntity.setAgentLocation((String) tsaAgentEntity.getRootRepresentables().get("urls"));
      return probeEntity;
    }
    return null;
  }

  private void throwsNotFoundExceptionWithMeaningfulResponse(Exception exception) {
    try {
      UniformInterfaceException uie = (UniformInterfaceException) exception.getCause();
      ClientResponse clientResponse = uie.getResponse();
      ErrorEntity errorEntity = clientResponse.getEntity(ErrorEntity.class);
      errorEntity.setError("The agent is online but returned the following error when we tried to list its info : "+errorEntity.getError());
      Response response =  RuntimeDelegate.getInstance().createResponseBuilder()
              .status(Response.Status.NOT_FOUND)
              .type(MediaType.APPLICATION_JSON_TYPE)
              .entity(errorEntity)
              .build();
      throw new WebApplicationException(response);
    } catch (ClientHandlerException che) {
      //we tried our best to get a meaningful response, but could not find any
      throw new NotFoundException(exception.getMessage());
    } catch (ClassCastException cce) {
      //we tried our best to get a meaningful response, but could not find any
      throw new NotFoundException(exception.getMessage());
    } catch (NullPointerException cce) {
      //we tried our best to get a meaningful response, but could not find any
      throw new NotFoundException(exception.getMessage());
    }
  }

  private Agent getNonSecuredAgent(String urlToProbe) {
    Agent agent =  new Agent();

    agent.setName("probeClient");
    agent.setGroupId("probeGroup");
    agent.setReadTimeoutMillis(60000);
    agent.setConnectionTimeoutMillis(60000);
    agent.setAgentLocation(urlToProbe);
    agent.setSecured(false);
    agent.setClientAuthEnabled(false);
    agent.setEnabled(true);
    agent.setType(Agent.TYPE.EHCACHE);

    return agent;
  }

  private Agent getSecuredAgent(String urlToProbe) {
    Agent agent = getNonSecuredAgent(urlToProbe);
    agent.setSecured(true);
    return agent;
  }

}
