/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package com.terracotta.management.config.resource.services;

import com.terracotta.management.AgentsUtils;
import com.terracotta.management.config.Agent;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.services.ConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;
import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.exceptions.ResourceRuntimeException;
import org.terracotta.management.resource.services.Utils;

import javax.ws.rs.*;
import javax.ws.rs.core.*;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * <p>A resource service for configuration of the aggregator management web service proxy.</p>
 *
 * @author brandony
 */
@Path("/config")
public final class ConfigResourceService {
  private static final Logger LOG = LoggerFactory.getLogger(ConfigResourceService.class);

  @Context
  private UriInfo info;

  private final ConfigService configSvc;

  private final Authorizer    authorizer;

  public ConfigResourceService() {
    this.configSvc = ServiceLocator.locate(ConfigService.class);
    this.authorizer = ServiceLocator.locate(Authorizer.class);
  }

  @POST
  @Deprecated
  /**
   * Save the configuration to file
   * Not needed anymore since we save changes to the file as soon as the connection is added or updated or deleted
   */
  public void save() {
    try {
      configSvc.saveConfig(authorizer.getPrincipal());
    } catch (ServiceExecutionException see) {
      throw new ResourceRuntimeException("Error saving config", see, Response.Status.INTERNAL_SERVER_ERROR.getStatusCode());
    }
  }

  /**
   * Get all the {@link Agent}s of a group.
   *
   * @param groupId the group ID
   * @return a collection of all the agents of the specified group
   */
  @GET
  @Path("/groups/{groupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Collection<Agent> getAgents(@PathParam("groupId") String groupId) {
    return configSvc.getAgentsByGroup(groupId, authorizer.getPrincipal());
  }

  /**
   * <p>Get all agent group names.</p>
   *
   * @return a collection of all agent group names
   */
  @GET
  @Path("/groups")
  @Produces(MediaType.APPLICATION_JSON)
  public Collection<String> getGroups() {
    return configSvc.getGroups(authorizer.getPrincipal());
  }

  @PUT
  @Path("/groups/{groupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Response addGroup(@PathParam("groupId") String groupId) {
    boolean added = configSvc.addGroup(groupId, authorizer.getPrincipal());
    if (added) {
      return Response.status(Response.Status.OK).entity("Group created.").build();
    } else {
      return Response.status(Response.Status.CONFLICT).entity("Group already exists.").build();
    }
  }

  @PUT
  @Path("/groups/{oldGroupId}/{newGroupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Response editGroup(@PathParam("oldGroupId") String oldGroupId,
                           @PathParam("newGroupId") String newGroupId) {
    boolean renamed = configSvc.renameGroup(oldGroupId, newGroupId, authorizer.getPrincipal());
    if (renamed) {
      return Response.status(Response.Status.OK).entity("Group renamed.").build();
    } else {
      return Response.status(Response.Status.CONFLICT)
          .entity("Group source group does not exist or destination group already exists.").build();
    }
  }

  @DELETE
  @Path("/groups/{groupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Response deleteGroup(@PathParam("groupId") String groupId) {
    boolean deleted = configSvc.deleteGroup(groupId, authorizer.getPrincipal());
    if (deleted) {
      return Response.status(Response.Status.OK).entity("Group deleted.").build();
    } else {
      return Response.status(Response.Status.NOT_FOUND).entity("Group does not exist or is not empty.").build();
    }
  }

  /**
   * <p>Get the current {@link Agent} configuration resource objects known by the TMS.</p>
   *
   * @return current config
   */
  @GET
  @Path("/agents")
  @Produces(MediaType.APPLICATION_JSON)
  public Collection<Agent> getAgents() {
    List<Agent> agents = new ArrayList<Agent>();

    Collection<String> groups = configSvc.getGroups(authorizer.getPrincipal());
    for (String groupId : groups) {
      Collection<Agent> agentsByGroup = configSvc.getAgentsByGroup(groupId, authorizer.getPrincipal());
      agents.addAll(agentsByGroup);
    }

    return agents;
  }

  /**
   * @param agent the agent to add
   * @return 400 if the request was not properly formulated, 409 if this agent already exists,
   * 500 if we caught an exception, and 201 if we the agent was created
   */
  @POST
  @Path("/agents")
  @Consumes(MediaType.APPLICATION_JSON)
  public Response addAgent(Agent agent) {
    Response.ResponseBuilder responseBuilder;
    agent.setUsername(authorizer.getPrincipal());

    isValidAgentLocation(agent);

    if (Utils.trimToNull(agent.getName()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing name.");
    } else if (Utils.trimToNull(agent.getGroupId()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing groupId.");
    } else if (agent.getAgentLocation() == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing agentLocation.");
    } else if (!isValidAgentLocation(agent)) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Invalid agentLocation.");
    } else {
      try {
        if (configSvc.addAgent(agent, authorizer.getPrincipal())) {
          configSvc.saveConfig(authorizer.getPrincipal());
          UriBuilder builder = info.getRequestUriBuilder();
          builder.segment(agent.getId());
          responseBuilder = Response.status(Response.Status.CREATED).header("LOCATION", builder.build());
        } else {
          responseBuilder = Response.status(Response.Status.CONFLICT);
        }
      } catch (ServiceExecutionException e) {
        // we could not persist the connection (save the configuration)
        responseBuilder = Response.status(Response.Status.INTERNAL_SERVER_ERROR)
            .entity("The configuration could not be persisted to the configuration file : " + e.getMessage());
      }
    }

    return responseBuilder.build();
  }

  @GET
  @Path("/agents/{agentId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Agent getAgent(@PathParam("agentId") String agentId) {
    return configSvc.getAgent(agentId, authorizer.getPrincipal());
  }

  @DELETE
  @Path("/agents/{agentId}")
  public Response deleteAgent(@PathParam("agentId") String agentId) {
    Response.ResponseBuilder responseBuilder = null;
    configSvc.deleteAgent(agentId, authorizer.getPrincipal());
    try {
      configSvc.saveConfig(authorizer.getPrincipal());
    } catch (ServiceExecutionException e) {
      // we could not persist the connection (save the configuration)
      responseBuilder = Response.status(Response.Status.INTERNAL_SERVER_ERROR)
          .entity("The configuration could not be persisted to the configuration file : " + e.getMessage());
    }
    return responseBuilder != null ? responseBuilder.build() : null;
  }

  @PUT
  @Path("/agents/{agentId}")
  @Consumes(MediaType.APPLICATION_JSON)
  public Response updateAgent(@PathParam("agentId") String agentId,
                              Agent agent) {
    agent.setUsername(authorizer.getPrincipal());
    Response.ResponseBuilder responseBuilder;

    if (Utils.trimToNull(agent.getName()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing name.");
    } else if (Utils.trimToNull(agent.getGroupId()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing groupId.");
    } else if (agent.getAgentLocation() == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing agentLocation.");
    } else if (!isValidAgentLocation(agent)) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Invalid agentLocation.");
    } else {
      try {
        if (configSvc.updateAgent(agentId, agent, authorizer.getPrincipal())) {
          configSvc.saveConfig(authorizer.getPrincipal());
          responseBuilder = Response.ok();
        } else {
          responseBuilder = Response.status(Response.Status.NOT_MODIFIED);
        }
      } catch (ServiceExecutionException e) {
        // we could not persist the connection (save the configuration)
        responseBuilder = Response.status(Response.Status.INTERNAL_SERVER_ERROR)
            .entity("The configuration could not be persisted to the configuration file : " + e.getMessage());
      }
    }

    return responseBuilder.build();
  }
  
  private boolean isValidAgentLocation(Agent agent) {
    boolean valid = true;

    try {
      if (agent.getType().equals(Agent.TYPE.TSA)) {
        AgentsUtils.checkTsaAgentLocationsAndReturnFirstValidOne(agent);
      } else {
        new URL(agent.getAgentLocation());
      }
    } catch (MalformedURLException e) {
      valid = false;
    }

    return valid;
  }
}
