package com.terracotta.management.config;


import com.terracotta.management.config.jaxb.adapters.Cluster;
import com.terracotta.management.config.jaxb.adapters.Group;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
/**
 * <p>An {@code Observable} object encapsulating the TMS configuration.</p>
 *
 * @author brandony
 */
public final class Config extends Observable {
  private final ConcurrentHashMap<String, ConcurrentMap<String, Agent>> agentsByGroupById;
  private final String username;

  public Config(String username) {
    agentsByGroupById = new ConcurrentHashMap<String, ConcurrentMap<String, Agent>>();
    this.username = username;
  }

  public Config(Map<String, Collection<Agent>> monitored, String username) {
    agentsByGroupById = new ConcurrentHashMap<String, ConcurrentMap<String, Agent>>();
    this.username = username;
    for (Map.Entry<String, Collection<Agent>> groupEntry : monitored.entrySet()) {
      addGroup(groupEntry.getKey());
      for (Agent agent : groupEntry.getValue()) {
        agent.setUsername(username);
        addAgent(agent);
      }
    }
  }

  public void agentsChanged(String agentId) {
    setChanged();
    notifyObservers(agentId);
  }

  public Set<String> getGroups() {
    return Collections.unmodifiableSet(agentsByGroupById.keySet());
  }

  public boolean addGroup(String groupId) {
    Object old = agentsByGroupById.putIfAbsent(groupId, new ConcurrentHashMap<String, Agent>());
    return old == null;
  }

  public boolean deleteGroup(String groupId) {
    boolean deleted = false;
    Collection<Agent> agents = getAgentsByGroup(groupId);

    if (agents != null) {
      for (Agent agent : getAgentsByGroup(groupId)) {
        deleteAgent(agent.getId());
      }
      deleted = agentsByGroupById.remove(groupId, Collections.emptyMap());
    }

    return deleted;
  }

  public boolean renameGroup(String oldGroupId,
                             String newGroupId) {
    ConcurrentMap<String, Agent> oldGroup = agentsByGroupById.get(oldGroupId);
    if (oldGroup == null) {
      return false;
    }
    ConcurrentMap<String, Agent> newGroup = new ConcurrentHashMap<String, Agent>();

    for (Map.Entry<String, Agent> entry : oldGroup.entrySet()) {
      Agent oldAgent = entry.getValue();
      Agent newAgent = new Agent(oldAgent);
      newAgent.setGroupId(newGroupId);
      newGroup.put(newAgent.getName(), newAgent);
    }

    ConcurrentMap<String, Agent> current = agentsByGroupById.putIfAbsent(newGroupId, newGroup);
    if (current == null) {
      agentsByGroupById.remove(oldGroupId);
      for (Map.Entry<String, Agent> entry : oldGroup.entrySet()) {
        agentsChanged(entry.getValue().getId());
      }
      for (Map.Entry<String, Agent> entry : newGroup.entrySet()) {
        agentsChanged(entry.getValue().getId());
      }
      return true;
    } else {
      return false;
    }
  }

  public Agent getAgent(String agentId) {
    String name = Agent.nameOfId(agentId);
    String groupId = Agent.groupOfId(agentId);
    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(groupId);
    return agentsMap != null ? agentsMap.get(name) : null;
  }

  public Collection<Agent> getAgentsByGroup(String groupId) {
    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(groupId);
    if (agentsMap == null) {
      return null;
    }
    return agentsMap.values();
  }

  public Agent deleteAgent(String agentId) {
    String oldName = Agent.nameOfId(agentId);
    String oldGroupId = Agent.groupOfId(agentId);

    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(oldGroupId);
    if (agentsMap == null) {
      return null;
    }
    Agent oldValue = agentsMap.remove(oldName);
    if (oldValue == null) {
      return null;
    }

    agentsChanged(agentId);
    return oldValue;
  }

  public boolean addAgent(Agent agent) {
    String name = agent.getName();
    String groupId = agent.getGroupId();

    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(groupId);
    if (agentsMap == null) {
      agentsByGroupById.put(groupId, agentsMap = new ConcurrentHashMap<String, Agent>());
    }
    if (agentsMap.containsKey(name)) {
      return false;
    }

    agentsMap.put(name, agent);
    agentsChanged(agent.getId());
    return true;
  }

  public boolean updateAgent(String agentId,
                             Agent agent) {
    String oldName = Agent.nameOfId(agentId);
    String oldGroupId = Agent.groupOfId(agentId);
    String newName = agent.getName();
    String newGroupId = agent.getGroupId();

    ConcurrentMap<String, Agent> oldAgentGroup = agentsByGroupById.get(oldGroupId);
    if (oldAgentGroup == null) {
      return false;
    }

    ConcurrentMap<String, Agent> newAgentGroup = agentsByGroupById.get(newGroupId);
    if (newAgentGroup == null) {
      agentsByGroupById.put(newGroupId, newAgentGroup = new ConcurrentHashMap<String, Agent>());
    }

    if (oldAgentGroup != newAgentGroup && newAgentGroup.containsKey(newName)) {
      return false;
    }

    Agent oldValue = oldAgentGroup.remove(oldName);
    if (oldValue == null) {
      return false;
    }
//
//    // we clean the old group up
//    if(oldAgentGroup.isEmpty()) {
//      agentsByGroupById.remove(oldGroupId);
//    }

    agentsChanged(agentId);

    newAgentGroup.put(newName, agent);

    agentsChanged(agent.getId());
    return true;
  }

  @Override
  public String toString() {
    return "Config{" +
        "agentsByGroupById=" + agentsByGroupById +
        '}';
  }

  public static Config load(InputStream is, String username) throws JAXBException {
    JAXBContext jaxbContext = JAXBContext.newInstance(com.terracotta.management.config.jaxb.adapters.Config.class);

    Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();
    com.terracotta.management.config.jaxb.adapters.Config ac = com.terracotta.management.config.jaxb.adapters.Config.class.cast(jaxbUnmarshaller
        .unmarshal(is));
    return new Config(ac.getMonitored(), username);
  }

  public static void write(Config config,
                           File configFile) throws JAXBException, IOException {
    File parent = configFile.getParentFile();

    if (parent != null) parent.mkdirs();

    configFile.createNewFile();

    JAXBContext jaxbContext = JAXBContext.newInstance(com.terracotta.management.config.jaxb.adapters.Config.class);
    Marshaller jaxbMarshaller = jaxbContext.createMarshaller();

    // output pretty printed
    jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
    com.terracotta.management.config.jaxb.adapters.Config ac = new com.terracotta.management.config.jaxb.adapters.Config();
    config.addGroupsAndClustersToPersistedConfig(ac);
    jaxbMarshaller.marshal(ac, configFile);
  }

  private void addGroupsAndClustersToPersistedConfig(com.terracotta.management.config.jaxb.adapters.Config ac) {
    List<Group> monitoredGroups = new ArrayList<Group>();
    List<Cluster> monitoredClusters = new ArrayList<Cluster>();
   
    for (String groupId : getGroups()) {
      Collection<Agent> agentsByGroup = getAgentsByGroup(groupId);
      boolean areAgentsPartOfACluster = areAgentsPartOfACluster(agentsByGroup);
      if (areAgentsPartOfACluster) {
        Cluster cluster = new Cluster();
        cluster.setId(groupId);
        cluster.setAgents(new ArrayList<Agent>(agentsByGroup));
        monitoredClusters.add(cluster);
      } else {
        Group group = new Group();
        group.setId(groupId);
        group.setAgents(new ArrayList<Agent>(agentsByGroup));
        monitoredGroups.add(group);
      }
    }
    ac.setMonitoredClusters(monitoredClusters);
    ac.setMonitoredGroups(monitoredGroups);
  }

  Map<String, Collection<Agent>> getMonitored() {
    Map<String, Collection<Agent>> monitored = new HashMap<String, Collection<Agent>>();

    for (String groupId : getGroups()) {
      Collection<Agent> agentsByGroup = getAgentsByGroup(groupId);
      monitored.put(groupId, agentsByGroup);
    }

    return monitored;
  }

  private boolean areAgentsPartOfACluster(Collection<Agent> agentsByGroup) {
    for (Agent agent : agentsByGroup) {
      if (agent.getType() == Agent.TYPE.TSA) {
        return true;
      }
    }
    return false;
  }

  public String getUsername() {
    return this.username;
  }

}
