/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management;

import org.terracotta.management.resource.Representable;

import java.util.*;

/**
 * <p>
 * The default callback implementation for collections of resources.
 * </p>
 * 
 * @author brandony
 */
public abstract class AggregateCollectionCallback<RESOURCE extends Representable> {
  protected Collection<Collection<RESOURCE>> responses = new ArrayList<Collection<RESOURCE>>();

  protected List<Exception> exceptions = new ArrayList<Exception>();

  /**
   * <p>
   * Add a collection of resources to be aggregated.
   * </p>
   * 
   * @param response to aggregate
   */
  public void addResponses(Collection<RESOURCE> response) {
    responses.add(response);
  }

  /**
   * <p>
   * Add an exception.
   * </p>
   *
   * @param exception to aggregate
   */
  public void addException(Exception exception) {
    exceptions.add(exception);
  }


  /**
   * <p>
   * Calculate the collection of aggregate resources to be returned.
   * </p>
   * 
   * @return aggregate result
   */
  public Collection<RESOURCE> calculateResult() {
    Collection<RESOURCE> result = new HashSet<RESOURCE>();

    for (Collection<RESOURCE> r : responses)
      result.addAll(r);

    return result;
  }

  /**
   * <p>
   * Return a copy of the list of exceptions
   * </p>
   *
   * @return aggregate result
   */
  public List<Exception> getExceptions() {
    return Collections.unmodifiableList(exceptions);
  }

}
