/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.userprofile.resource.services;

import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.user.UserRole;
import com.terracotta.management.userprofile.dao.UserProfileDao;
import com.terracotta.management.userprofile.resource.UserProfile;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import org.terracotta.management.ServiceLocator;

/**
 * @author brandony
 */
@Path("/userprofiles")
public final class UserProfileResourceService {
  private final UserProfileDao usrProfileDao;

  private final Authorizer authorizer;

  public UserProfileResourceService() {
    this.usrProfileDao = ServiceLocator.locate(UserProfileDao.class);
    this.authorizer = ServiceLocator.locate(Authorizer.class);
  }

  @GET
  @Path("/{userid}")
  @Produces(MediaType.APPLICATION_JSON)
  public UserProfile retrieveProfile(@PathParam("userid") String userid) {
    checkAuthorization(userid);

    UserProfile up;
    try {
      up = usrProfileDao.getById(userid);
    } catch (DataAccessException e) {
      throw new WebApplicationException(
          Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(e.getMessage()).build());
    }

    if (up == null) {
      throw new WebApplicationException(Response.status(Response.Status.NO_CONTENT).build());
    }

    return up;
  }

  @POST
  @Consumes(MediaType.APPLICATION_JSON)
  public void addUserProfile(UserProfile userProfile) {
    checkAuthorization(userProfile.getId());

    try {
      usrProfileDao.create(userProfile);
      usrProfileDao.flush();
    } catch (DataAccessException e) {
      throw new WebApplicationException(
          Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(e.getMessage()).build());
    }
  }

  @PUT
  @Path("/{userid}")
  @Consumes(MediaType.APPLICATION_JSON)
  public void updateUserProfile(@PathParam("userid") String userid,
                                UserProfile userProfile) {
    checkAuthorization(userid);

    userProfile.setId(userid);

    try {
      usrProfileDao.createOrUpdate(userProfile);
      usrProfileDao.flush();
    } catch (DataAccessException e) {
      throw new WebApplicationException(
          Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(e.getMessage()).build());
    }
  }

  @DELETE
  @Path("/{userid}")
  public void deleteUserProfile(@PathParam("userid") String userid) {
    checkAuthorization(userid);

    try {
      UserProfile up = usrProfileDao.getById(userid);

      if (up == null) {
        throw new WebApplicationException(Response.status(Response.Status.CONFLICT).build());
      }

      usrProfileDao.delete(up);
      usrProfileDao.flush();
    } catch (DataAccessException e) {
      throw new WebApplicationException(
          Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(e.getMessage()).build());
    }
  }

  private void checkAuthorization(String userId) {
    if (!authorizer.isUserInRole(UserRole.ADMIN) && !authorizer.getPrincipal().equals(userId)) {
      throw new WebApplicationException(Response.status(Response.Status.UNAUTHORIZED).build());
    }
  }
}
