/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.security.shiro;

import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.security.SecurityContextManager;
import com.terracotta.management.security.shiro.realm.TCIniRealm;
import com.terracotta.management.user.UserInfo;
import com.terracotta.management.user.UserRole;
import com.terracotta.management.user.dao.DatastoreNotFoundException;
import com.terracotta.management.user.dao.impl.IniFileUserInfoDao;
import com.terracotta.management.user.impl.DfltUserInfo;
import com.terracotta.management.user.services.UserService;
import org.apache.shiro.authc.SimpleAccount;
import org.terracotta.management.ServiceExecutionException;

/**
 * An implementation of {@link SecurityContextManager} that interacts with a {@link com.terracotta.management.user.dao.impl.IniFileUserInfoDao} for realm data management
 * and keeps those changes in sync with the {@link TCIniRealm}. This class also implements the {@link UserService}.
 *
 * @author brandony
 */
public final class ShiroSecurityContextManager implements SecurityContextManager, UserService {
  private final IniFileUserInfoDao iniUserInfoDao;

  private final TCIniRealm tcIniRealm;

  public ShiroSecurityContextManager(IniFileUserInfoDao iniUserInfoDao,
                                     TCIniRealm tcIniRealm) {
    this.iniUserInfoDao = iniUserInfoDao;
    this.tcIniRealm = tcIniRealm;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void addUser(UserInfo user) throws ServiceExecutionException {

    try {
      iniUserInfoDao.create(user);
    } catch (DataAccessException e) {
      throw new ServiceExecutionException(String.format(
          "Unable to create '%s'. Check the configured location of the security ini file and the file permissions for the user that started the application.",
          user));
    }

    tcIniRealm.addAccount(user.getUsername(), user.getPasswordHash(),
        user.getRoles() == null ? null : UserRole.convertRoles(user.getRoles()).toArray(new String[0]));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void updateUser(UserInfo user) throws ServiceExecutionException {
    try {
      iniUserInfoDao.createOrUpdate(user);
    } catch (DataAccessException e) {
      throw new ServiceExecutionException(String.format(
          "Unable to update '%s'. Check the configured location of the security ini file and the file permissions for the user that started the application.",
          user));
    }
    SimpleAccount acct = tcIniRealm.getAccount(user.getUsername());
    acct.setCredentials(user.getPasswordHash());
    acct.setRoles(UserRole.convertRoles(user.getRoles()));
  }

  /**
   * Locking an account on an ini realm is currently not persisted because Shiro ini files lack the semantics to
   * describe this condition.
   * <p/>
   * {@inheritDoc}
   */
  //TODO: Consider extending the ini realm to support some file semantic for account locking.
  @Override
  public synchronized void disableUser(String username) {
    SimpleAccount acct = tcIniRealm.getAccount(username);

    if (acct != null) acct.setLocked(true);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized void deleteUser(UserInfo user) throws ServiceExecutionException {
    try {
      iniUserInfoDao.delete(user);
    } catch (DataAccessException e) {
      throw new ServiceExecutionException(String.format(
          "Unable to delete '%s'. Check the configured location of the security ini file and the file permissions for the user that started the application.",
          user));
    }
    tcIniRealm.removeAccount(user.getUsername());
  }

  /**
   * {@inheritDoc}
   */
  //TODO: Should we check the datasource first and compare? What do we do if they're different?
  @Override
  public synchronized UserInfo getUser(String username) {
    SimpleAccount acct = tcIniRealm.getAccount(username);

    return acct == null ? null : new DfltUserInfo(username, acct.getCredentials().toString(),
        UserRole.convertRoleNames(acct.getRoles()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public synchronized boolean hasValidSecurityContext() {
    boolean valid = true;

    if (tcIniRealm.isEmpty()) valid = false;

    if (valid) {
      try {
        iniUserInfoDao.validate(false);
        valid = iniUserInfoDao.hasUserInfos();
      } catch (DatastoreNotFoundException e) {
        valid = false;
      } catch (DataAccessException e) {
        throw new RuntimeException(e);
      }
    }

    return valid;
  }
}
