/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.servlet;

import com.terracotta.license.LicenseManager;
import com.terracotta.management.security.SecurityContextManager;
import com.terracotta.management.services.ConfigService;

import org.terracotta.license.LicenseException;
import org.terracotta.management.ServiceLocator;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

import static com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener.HAS_LICENSE;
import static com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener.LICENSE_IS_COMMERCIAL_LICENSE;

/**
 * A {@code Filter} that is aware of TMC setup requirements and redirects requests to certain pages when no valid
 * security context is available or when the TMC is running without a license.
 *
 * @author brandony
 */
public final class SetupAwareFilter implements Filter {
  private static final String TMC_HOME = "/index.jsp";
  
  private static final String TMC_AUTHENTICATION = "/setup/authenticationSetup.jsp";
  
  private static final String TMC_RESTART = "/setup/restart.jsp";

  private static final String TMC_LOGOUT_URL = "/logout.jsp";

  private final SecurityContextManager securityCtxtMgr;

  public SetupAwareFilter() {
    this.securityCtxtMgr = ServiceLocator.locate(SecurityContextManager.class);
  }

  @Override
  public void init(FilterConfig filterConfig) throws ServletException {
    //no-op
  }

  @Override
  public void doFilter(ServletRequest servletRequest,
                       ServletResponse servletResponse,
                       FilterChain filterChain) throws IOException, ServletException {
    HttpServletRequest req = (HttpServletRequest) servletRequest;
    HttpServletResponse resp = (HttpServletResponse) servletResponse;

    if (HAS_LICENSE == null) {
      try {
        LicenseManager.verifyTMCCapability();
        HAS_LICENSE = true;
        LICENSE_IS_COMMERCIAL_LICENSE = LicenseManager.isCommercialLicense();
      } catch (LicenseException e) {
        HAS_LICENSE = false;
      }
    }

    Boolean authenticationEnabled = ServiceLocator.locate(ConfigService.class).isAuthenticationEnabled();
    if (HAS_LICENSE && authenticationEnabled == null){  
      if (req.getRequestURI().equals(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL) || req
          .getRequestURI().equals(req.getContextPath() + AccountInitializationServlet.TMC_SETUP_URL) || req
          .getRequestURI().equals(req.getContextPath() + TMC_LOGOUT_URL)|| req
          .getRequestURI().equals(req.getContextPath() + TMC_HOME)) {
        RequestDispatcher rd = req.getSession().getServletContext()
            .getRequestDispatcher(TMC_AUTHENTICATION);
        rd.forward(req, resp);
      } else {
        filterChain.doFilter(req, resp);
      }
    } else if (HAS_LICENSE && authenticationEnabled != null &&  authenticationEnabled ) {
      if(tmcNeedsToRestart(authenticationEnabled)){
        redirectToRestartPage(filterChain, req, resp);
        return;
      }
      boolean ctxtValid = securityCtxtMgr.hasValidSecurityContext();
      if (!ctxtValid && req.getRequestURI()
          .startsWith(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL)|| req
          .getRequestURI().equals(req.getContextPath() + TMC_AUTHENTICATION)) {
        RequestDispatcher rd = req.getSession().getServletContext()
            .getRequestDispatcher(AccountInitializationServlet.TMC_SETUP_URL);
        rd.forward(req, resp);
      } else if (ctxtValid && req.getRequestURI().startsWith(req.getContextPath() + "/setup")) {
        resp.sendRedirect(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL);
      } else {
        filterChain.doFilter(req, resp);
      }
    } else if (HAS_LICENSE  &&  !authenticationEnabled && tmcNeedsToRestart(authenticationEnabled)) {
        redirectToRestartPage(filterChain, req, resp);
    } else {
      if (req.getRequestURI().equals(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL) || req
          .getRequestURI().equals(req.getContextPath() + AccountInitializationServlet.TMC_SETUP_URL) || req
          .getRequestURI().equals(req.getContextPath() + TMC_LOGOUT_URL)|| req
          .getRequestURI().equals(req.getContextPath() + TMC_AUTHENTICATION)) {
        resp.sendRedirect(req.getContextPath() + TMC_HOME);
      } else {
        filterChain.doFilter(req, resp);
      }
    }
  }

  private void redirectToRestartPage(FilterChain filterChain, HttpServletRequest req, HttpServletResponse resp)
      throws ServletException, IOException {
    if (req.getRequestURI().startsWith(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL)
        || req.getRequestURI().startsWith(req.getContextPath() + AccountInitializationServlet.TMC_SETUP_URL)
        || req.getRequestURI().equals(req.getContextPath() + TMC_AUTHENTICATION)
        || req.getRequestURI().equals(req.getContextPath() + TMC_HOME)
        || req.getRequestURI().equals(req.getContextPath() + TMC_LOGOUT_URL)
        ) {
      RequestDispatcher rd = req.getSession().getServletContext()
          .getRequestDispatcher(TMC_RESTART);
      rd.forward(req, resp);
    } else {
      filterChain.doFilter(req, resp);
    }
  }

  @Override
  public void destroy() {
    //no-op
  }
  /**
   * Checks whether or not the tms/tmc is in a valid state
   * If the user changed the authentication status, and the tmc did not restart, this method will return true
   * 
   * @return
   */
  private boolean tmcNeedsToRestart(Boolean authenticationEnabled){
    SecurityContextManager  securityContextManager =  ServiceLocator.locate(SecurityContextManager.class);
    if(authenticationEnabled && securityContextManager ==null || securityContextManager != null && !authenticationEnabled) {
      return true;
    }
    return false;
  }
  
  
}
