/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.security.shiro.realm;

import org.apache.shiro.authc.SimpleAccount;
import org.apache.shiro.config.Ini;
import org.apache.shiro.realm.text.IniRealm;

import java.io.File;

import static com.terracotta.management.security.shiro.ShiroIniFileConstants.DFLT_INI_FILE_LOCATION;
import static com.terracotta.management.security.shiro.ShiroIniFileConstants.JVM_INI_LOCATION_PROP;

/**
 * A Shiro {@code IniRealm} implementation that initializes the realm with detail from an ini file if it can be located.
 * <p/>
 * By default this realm will try and load its data from a ini file location provided via the JVM property
 * {@code com.tc.management.security.ini}. If that JVM property is not set, it will look
 * to {@code $&#123;user.home&#125;/.tc/mgmt/security.ini}.
 *
 * @author brandony
 */
public final class TCIniRealm extends IniRealm {
  public TCIniRealm() {
    this(new File(System.getProperty(JVM_INI_LOCATION_PROP) == null ? DFLT_INI_FILE_LOCATION : System
        .getProperty(JVM_INI_LOCATION_PROP)));
  }

  /**
   * A constructor that allows the ini file location used to load this in-memory realm to be explicitly set.
   *
   * @param iniFile to use to load data into this realm
   */
  public TCIniRealm(File iniFile) {
    super();

    if (iniFile.exists()) {
      String iniFilePath = iniFile.getAbsolutePath();
      Ini ini = Ini.fromResourcePath(iniFilePath);
      setIni(ini);
      setResourcePath(iniFilePath);
    }

    Ini ini = getIni();
    if (ini == null || ini.isEmpty()) {
      SimpleAccount dummy = new SimpleAccount("__tc-dummy",
          "$shiro1$SHA-256$100000$f3WQKo0JpXZp7LWCy+kP4Q==$STWFWj5IYGDMwgpPk8sWoudXPPwqRol/JEI0ZqQm+Js=", getName(),
          null, null);
      dummy.setLocked(true);
      users.put(dummy.getPrincipals().getPrimaryPrincipal().toString(), dummy);
    }
  }

  public boolean removeAccount(String username) {
    return users.remove(username) != null;
  }

  public SimpleAccount getAccount(String username) {
    return users.get(username);
  }

  public boolean isEmpty() {
    return users.isEmpty() || (users.size() == 1 && users.get("__tc-dummy") != null);
  }
}