/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.shiro;

import com.terracotta.management.security.Authorizer;
import com.terracotta.management.user.UserRole;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.subject.Subject;

import java.util.ArrayList;
import java.util.Set;

/**
 * @author brandony
 */
public final class ShiroAuthorizer implements Authorizer {

  @Override
  public boolean isUserInRole(UserRole role) {
    return SecurityUtils.getSubject().hasRole(role.toString());
  }

  @Override
  public boolean isUserInAllRoles(Set<UserRole> roles) {
    return SecurityUtils.getSubject().hasAllRoles(UserRole.convertRoles(roles));
  }

  @Override
  public boolean isUserInAnyRole(Set<UserRole> roles) {
    boolean[] results = SecurityUtils.getSubject().hasRoles(new ArrayList<String>(UserRole.convertRoles(roles)));

    for(boolean r : results) {
      if(r) {
        return true;
      }
    }
    return false;
  }

  @Override
  public String getPrincipal() {
    return SecurityUtils.getSubject().getPrincipal().toString();
  }

  @Override
  public String getPrincipalBySessionId(String sessionId,
                                        boolean isAuthenticated) {
    Subject subject = new Subject.Builder().sessionId(sessionId).buildSubject();
    String result = null;
    
    if(subject != null) {
      Object principal = subject.getPrincipal();
      if (principal != null) {
        result = principal.toString();
        if(isAuthenticated && !subject.isAuthenticated()){
          result = null;
        }
      }
    }
    return result;
  }

  @Override
  public String getSessionId() {
    return SecurityUtils.getSubject().getSession().getId().toString();
  }
}
