package com.terracotta.management.config;

import com.terracotta.management.config.jaxb.adapters.AdaptedConfig;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Observable;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * <p>An {@code Observable} object encapsulating the TMS configuration.</p>
 *
 * @author brandony
 */
public final class Config extends Observable {
  private final ConcurrentHashMap<String, ConcurrentMap<String, Agent>> agentsByGroupById;
  private Boolean authenticationEnabled;

  public Config() {
    agentsByGroupById = new ConcurrentHashMap<String, ConcurrentMap<String, Agent>>();
  }

  public Config(Map<String, Collection<Agent>> monitored, Boolean authenticationEnabled) {
    this.authenticationEnabled = authenticationEnabled;
    agentsByGroupById = new ConcurrentHashMap<String, ConcurrentMap<String, Agent>>();
    for (Map.Entry<String, Collection<Agent>> groupEntry : monitored.entrySet()) {
      addGroup(groupEntry.getKey());
      for (Agent agent : groupEntry.getValue()) {
        addAgent(agent);
      }
    }
  }

  private void agentsChanged(String agentId) {
    setChanged();
    notifyObservers(agentId);
  }

  public Set<String> getGroups() {
    return Collections.unmodifiableSet(agentsByGroupById.keySet());
  }

  public boolean addGroup(String groupId) {
    Object old = agentsByGroupById.putIfAbsent(groupId, new ConcurrentHashMap<String, Agent>());
    return old == null;
  }

  public boolean deleteGroup(String groupId) {
    boolean deleted = false;
    Collection<Agent> agents = getAgentsByGroup(groupId);

    if (agents != null) {
      for (Agent agent : getAgentsByGroup(groupId)) {
        deleteAgent(agent.getId());
      }
      deleted = agentsByGroupById.remove(groupId, Collections.emptyMap());
    }

    return deleted;
  }

  public boolean renameGroup(String oldGroupId,
                             String newGroupId) {
    ConcurrentMap<String, Agent> oldGroup = agentsByGroupById.get(oldGroupId);
    if (oldGroup == null) {
      return false;
    }
    ConcurrentMap<String, Agent> newGroup = new ConcurrentHashMap<String, Agent>();

    for (Map.Entry<String, Agent> entry : oldGroup.entrySet()) {
      Agent oldAgent = entry.getValue();
      Agent newAgent = new Agent(oldAgent);
      newAgent.setGroupId(newGroupId);
      newGroup.put(newAgent.getName(), newAgent);
    }

    ConcurrentMap<String, Agent> current = agentsByGroupById.putIfAbsent(newGroupId, newGroup);
    if (current == null) {
      agentsByGroupById.remove(oldGroupId);
      for (Map.Entry<String, Agent> entry : oldGroup.entrySet()) {
        agentsChanged(entry.getValue().getId());
      }
      for (Map.Entry<String, Agent> entry : newGroup.entrySet()) {
        agentsChanged(entry.getValue().getId());
      }
      return true;
    } else {
      return false;
    }
  }

  public Agent getAgent(String agentId) {
    String name = Agent.nameOfId(agentId);
    String groupId = Agent.groupOfId(agentId);
    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(groupId);
    return agentsMap != null ? agentsMap.get(name) : null;
  }

  public Collection<Agent> getAgentsByGroup(String groupId) {
    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(groupId);
    if (agentsMap == null) {
      return null;
    }
    return agentsMap.values();
  }

  public Agent deleteAgent(String agentId) {
    String oldName = Agent.nameOfId(agentId);
    String oldGroupId = Agent.groupOfId(agentId);

    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(oldGroupId);
    if (agentsMap == null) {
      return null;
    }
    Agent oldValue = agentsMap.remove(oldName);
    if (oldValue == null) {
      return null;
    }

    agentsChanged(agentId);
    return oldValue;
  }

  public boolean addAgent(Agent agent) {
    String name = agent.getName();
    String groupId = agent.getGroupId();

    ConcurrentMap<String, Agent> agentsMap = agentsByGroupById.get(groupId);
    if (agentsMap == null) {
      agentsByGroupById.put(groupId, agentsMap = new ConcurrentHashMap<String, Agent>());
    }
    if (agentsMap.containsKey(name)) {
      return false;
    }

    agentsMap.put(name, agent);
    agentsChanged(agent.getId());
    return true;
  }

  public boolean updateAgent(String agentId,
                             Agent agent) {
    String oldName = Agent.nameOfId(agentId);
    String oldGroupId = Agent.groupOfId(agentId);
    String newName = agent.getName();
    String newGroupId = agent.getGroupId();

    ConcurrentMap<String, Agent> oldAgentGroup = agentsByGroupById.get(oldGroupId);
    if (oldAgentGroup == null) {
      return false;
    }

    ConcurrentMap<String, Agent> newAgentGroup = agentsByGroupById.get(newGroupId);
    if (newAgentGroup == null) {
      return false;
    }

    if (oldAgentGroup != newAgentGroup && newAgentGroup.containsKey(newName)) {
      return false;
    }

    Agent oldValue = oldAgentGroup.remove(oldName);
    if (oldValue == null) {
      return false;
    }
    agentsChanged(agentId);

    newAgentGroup.put(newName, agent);

    agentsChanged(agent.getId());
    return true;
  }

  @Override
  public String toString() {
    return "Config{" +
        "agentsByGroupById=" + agentsByGroupById +
        '}';
  }

  public static Config load(InputStream is) throws JAXBException {
    JAXBContext jaxbContext = JAXBContext.newInstance(AdaptedConfig.class);

    Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();
    AdaptedConfig ac = AdaptedConfig.class.cast(jaxbUnmarshaller.unmarshal(is));
    Boolean authenticationEnabled = ac.isAuthenticationEnabled() == null ? null : ac.isAuthenticationEnabled();
    return new Config(ac.getMonitored(), authenticationEnabled);
  }

  public static void write(Config config,
                           File configFile) throws JAXBException, IOException {
    File parent = configFile.getParentFile();

    if (parent != null) parent.mkdirs();

    configFile.createNewFile();

    JAXBContext jaxbContext = JAXBContext.newInstance(AdaptedConfig.class);
    Marshaller jaxbMarshaller = jaxbContext.createMarshaller();

    // output pretty printed
    jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
    AdaptedConfig ac = new AdaptedConfig();
    ac.setMonitored(config.getMonitored());
    ac.setAuthenticationEnabled(config.isAuthenticationEnabled());
    jaxbMarshaller.marshal(ac, configFile);
  }

  Map<String, Collection<Agent>> getMonitored() {
    Map<String, Collection<Agent>> monitored = new HashMap<String, Collection<Agent>>();

    for (String groupId : getGroups()) {
      Collection<Agent> agentsByGroup = getAgentsByGroup(groupId);
      monitored.put(groupId, agentsByGroup);
    }

    return monitored;
  }
  
  public Boolean isAuthenticationEnabled() {
    return authenticationEnabled;
  }
  
  
  public void setAuthenticationEnabled(Boolean authenticationEnabled) {
    this.authenticationEnabled = authenticationEnabled;
    setChanged();    
  }
}
