package com.terracotta.management.services;

import com.terracotta.management.AggregateCallback;
import com.terracotta.management.AggregateCollectionCallback;
import org.terracotta.management.resource.Representable;
import org.terracotta.management.resource.VersionedEntity;

import javax.ws.rs.core.Response;
import java.net.URI;
import java.util.Set;

/**
 * <p>A service interface for proxy requests to embedded resource services.</p>
 *
 * @author brandony
 */
public interface ResourceServiceClientService {
  /**
   * <p>Submits GET requests to all configured embedded agents management resource service and provides the results to
   * the caller via the {@link com.terracotta.management.AggregateCollectionCallback} implementation submitted.</p>
   *
   * @param callback    - callback aggregation strategy
   * @param request     - the location of this resource
   * @param agentIds    - the identifiers for the agents this request should be submitted to
   * @param entityClass - the class of the resource being requested
   * @param <ENTITY>    - the resource TYPE for which this request it being proxied
   */
  <RESOURCE extends Representable> void proxyGet(AggregateCollectionCallback<RESOURCE> callback,
                                               URI request,
                                               String[] agentIds,
                                               Class<RESOURCE> entityClass);

  <RESOURCE extends VersionedEntity> void proxyGet(AggregateCallback<RESOURCE> callback,
                                               URI request,
                                               String[] agentIds,
                                               Class<RESOURCE> entityClass);

  /**
   * <p>Submits a DELETE request to an embedded agents management resource service.</p>
   *
   * @param request - the location of this resource
   * @param agentId - the identifier for the agent this request should be submitted to
   */
  void proxyDelete(URI request,
                   String agentId);

  /**
   * <p>Submits a PUT request to an embedded agents management resource service.</p>
   *
   * @param request  - the location of this resource
   * @param entity   - the resource to be created
   * @param agentId  - the identifier for the agent this request should be submitted to
   * @param <ENTITY> - the resource TYPE for which this request it being proxied
   */
  <ENTITY> void proxyPut(URI request,
                         ENTITY entity,
                         String agentId);

  /**
   * <p>Submits a POST request to an embedded agents management resource service.</p>
   *
   * @param request  - the location of this resource
   * @param entity   - the resource to be created or updated
   * @param agentId  - the identifier for the agent this request should be submitted to
   * @param <ENTITY> - the resource TYPE for which this request it being proxied
   * @return a response including a header entry, with key <em>LOCATION</em>, providing the resource location if it is created.
   */
  <ENTITY> Response proxyPost(URI request,
                              ENTITY entity,
                              String agentId);

  /**
   * <p>An accessor providing a {@code Set} of all known agent identifiers to this proxy service.</p>
   *
   * @return known agent ids
   */
  Set<String> getKnownAgentIds();
}