/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */

package com.terracotta.management.userprofile.resource;


import javax.xml.bind.annotation.XmlAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * A resource representing the user profile state for a Subject known by the system.
 *
 * @author brandony
 */
public final class UserProfile {
  /**
   * The default display interval for charted statistics.
   */
  private static final Long DFLT_DISPLAY_INTERVAL_MILLIS = TimeUnit.SECONDS.toMillis(1);

  /**
   * The default Ehcache charted statistics names.
   */
  private static final List<String> DFLT_EHCACHE_CHARTED_STATISTICS = new ArrayList<String>(
    Arrays.asList(
      "CacheHitRate",
      "CacheMissRate",
      "CachePutRate",
      "CacheAverageGetTime"
  ));
  
  /**
   * The default Ehcache grid statistics names.
   */
  private static final List<String> DFLT_EHCACHE_GRID_STATISTICS = new ArrayList<String>(
    Arrays.asList(
      "CacheHitRate", 
      "CacheMissRate", 
      "Size", 
      "LocalHeapSize", 
      "LocalHeapSizeInBytes", 
      "LocalHeapSize", 
      "LocalHeapSizeInBytes", 
      "CacheAverageGetTime"
  ));

  private String id;

  private Long intervalMillis;

  private List<String> ehcacheChartedStatistics;
  
  private List<String> ehcacheGridStatistics;
  
  public UserProfile() {
  }

  public UserProfile(String id,
                     Long intervalMillis,
                     List<String> ehcacheChartedStatistics,
                     List<String> ehcacheGridStatistics) {
    this.id = id;
    this.intervalMillis = intervalMillis;
    this.ehcacheChartedStatistics = ehcacheChartedStatistics;
    this.ehcacheGridStatistics = ehcacheGridStatistics;
  }

  /**
   * {@inheritDoc}
   */
  @XmlAttribute
  public String getId() {
    return id;
  }

  public void setId(String id) {
    this.id = id;
  }

  /**
   * {@inheritDoc}
   */
  @XmlAttribute
  public Long getStatisticsPollingIntervalMillis() {
    return intervalMillis == null ? DFLT_DISPLAY_INTERVAL_MILLIS : intervalMillis;
  }

  /**
   * {@inheritDoc}
   */
  public void setStatisticsPollingIntervalMillis(Long intervalMillis) {
    this.intervalMillis = intervalMillis;
  }

  /**
   * {@inheritDoc}
   */
  @XmlAttribute
  public List<String> getEhcacheChartedStatistics() {
    return ehcacheChartedStatistics != null ? new ArrayList<String>(ehcacheChartedStatistics) : DFLT_EHCACHE_CHARTED_STATISTICS;
  }

  /**
   * {@inheritDoc}
   */
  public void setEhcacheChartedStatistics(List<String> ehcacheChartedStatistics) {
    this.ehcacheChartedStatistics = ehcacheChartedStatistics;
  }

  /**
   * {@inheritDoc}
   */
  @XmlAttribute
  public List<String> getEhcacheGridStatistics() {
    return ehcacheGridStatistics != null ? new ArrayList<String>(ehcacheGridStatistics) : DFLT_EHCACHE_GRID_STATISTICS;
  }

  /**
   * {@inheritDoc}
   */
  public void setEhcacheGridStatistics(List<String> ehcacheGridStatistics) {
    this.ehcacheGridStatistics = ehcacheGridStatistics;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    UserProfile that = (UserProfile) o;

    if (ehcacheChartedStatistics != null ? !ehcacheChartedStatistics
        .equals(that.ehcacheChartedStatistics) : that.ehcacheChartedStatistics != null) return false;
    if (ehcacheGridStatistics != null ? !ehcacheGridStatistics
        .equals(that.ehcacheGridStatistics) : that.ehcacheGridStatistics != null) return false;
    if (id != null ? !id.equals(that.id) : that.id != null) return false;
    if (intervalMillis != null ? !intervalMillis.equals(that.intervalMillis) : that.intervalMillis != null)
      return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = id != null ? id.hashCode() : 0;
    result = 31 * result + (intervalMillis != null ? intervalMillis.hashCode() : 0);
    result = 31 * result + (ehcacheChartedStatistics != null ? ehcacheChartedStatistics.hashCode() : 0);
    result = 31 * result + (ehcacheGridStatistics != null ? ehcacheGridStatistics.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "UserProfile{" +
        "id='" + id + '\'' +
        ", intervalMillis=" + intervalMillis +
        ", ehcacheChartedStatistics=" + ehcacheChartedStatistics +
        ", ehcacheGridStatistics=" + ehcacheGridStatistics +
        '}';
  }
}
