/*
 * All content copyright Terracotta, Inc., unless otherwise indicated. All rights reserved.
 */
package com.terracotta.license;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.license.AbstractLicenseResolverFactory;
import org.terracotta.license.License;
import org.terracotta.license.LicenseException;

import java.util.Date;

import static org.terracotta.license.LicenseConstants.CAPABILITY_TMC;
import static org.terracotta.license.LicenseConstants.LICENSE_KEY_FILENAME;
import static org.terracotta.license.LicenseConstants.LICENSE_TYPE_COMMERCIAL;
import static org.terracotta.license.LicenseConstants.PRODUCTKEY_PATH_PROPERTY;

public class LicenseManager {
  private static final Logger LOG = LoggerFactory.getLogger(LicenseManager.class);
  public static final String EXIT_MESSAGE = "TERRACOTTA IS EXITING. Contact your Terracotta sales representative to "
      + "learn how to enable licensed usage of this feature. For more information, "
      + "visit Terracotta support at http://www.terracotta.org.";
  public static final String EXPIRY_WARNING = "Your license key is valid until %s. "
      + "You have %s remaining until the expiration date. "
      + "When the expiration date is reached TERRACOTTA WILL CEASE FUNCTIONING.";
  public static final String EXPIRED_ERROR = "Your product key expired on %s. " + EXIT_MESSAGE;
  private static volatile boolean initialized;

  // lazily-init, don't use directly
  // use getLicense() instead
  private static License license;
  private static final AbstractLicenseResolverFactory factory = AbstractLicenseResolverFactory.getFactory();
  private static boolean warnedAboutMissingLicense = false;
  
  private static synchronized void init() {
    license = factory.resolveLicense();
    initialized = true;
    if (license != null) {
      LOG.info("Terracotta license loaded from " + factory.getLicenseLocation() + "\n" + license.toString());
    }
  }

  public static synchronized License getLicense() {
    if (!initialized) {
      init();
    }
    return license;
  }

  /**
   * check for null and expired license
   */
  public static void assertValidLicense() {
    boolean expiredOrMissing = false;

    if (getLicense() == null) {
      if(!warnedAboutMissingLicense) {
        LOG.info(
            "Terracotta license key is required for Enterprise capabilities. Please place " + LICENSE_KEY_FILENAME
                + " in the Terracotta installation directory or in the resource path. You can also specify it as a system property with -D"
                + PRODUCTKEY_PATH_PROPERTY + "=/path/to/key");
        warnedAboutMissingLicense = true;
      }
      expiredOrMissing = true;
    } else {
      Date expirationDate = getLicense().expirationDate();
      if (expirationDate != null && expirationDate.before(new Date())) {
        LOG.error("Your Terracotta license has expired on " + expirationDate);
        expiredOrMissing = true;
      }
    }

    if (expiredOrMissing) throw new LicenseException();
  }

  private static void verifyCapability(String capability) {
    assertValidLicense();
    if (!getLicense().isCapabilityEnabled(capability)) {
      //
      throw new LicenseException("Your license key doesn't allow usage of '" + capability + "' capability");
    }
  }

  public static boolean isCommercialLicense() {
    assertValidLicense();
    return getLicense().type().equals(LICENSE_TYPE_COMMERCIAL);
  }

  public static void verifyTMCCapability() {
    verifyCapability(CAPABILITY_TMC);
  }
}
