/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */

package com.terracotta.management.userprofile.dao.impl;

import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.userprofile.dao.UserProfileDao;
import com.terracotta.management.userprofile.resource.UserProfile;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;

/**
 * A DAO for {@link UserProfile} objects that uses an XML file as a datastore.
 *
 * @author brandony
 */
public final class XMLFileUserProfileDao implements UserProfileDao {
  private static final Logger LOG = LoggerFactory.getLogger(XMLFileUserProfileDao.class);
  /**
   * A constant for the default user profile settings file location.
   */
  private static final String DFLT_UPS_FILE_LOCATION = System
      .getProperty("user.home") + "/.tc/mgmt/usr_profile_settings.xml";

  /**
   * A constant for the jvm user profile settings file location property.
   */
  private static final String JVM_UPS_LOCATION_PROP = "com.tc.management.ups.xml";

  private UserProfileSettings profileSettings;

  private File settingsFile;

  public XMLFileUserProfileDao() throws IOException, JAXBException {
    settingsFile = new File(System.getProperty(JVM_UPS_LOCATION_PROP) == null ? DFLT_UPS_FILE_LOCATION : System
        .getProperty(JVM_UPS_LOCATION_PROP));

    if (settingsFile.exists()) {
      profileSettings = UserProfileSettings.load(new FileInputStream(settingsFile));
    } else {
      initSettingsFile();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public UserProfile getById(String userId) throws DataAccessException {
    return profileSettings.userProfileMap.get(userId);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void create(UserProfile userProfile) throws DataAccessException {
    UserProfile up = profileSettings.userProfileMap.putIfAbsent(userProfile.getId(), userProfile);

    if (up != null) throw new DataAccessException(
        String.format("Failed to create user profile '%s' because it already exists.", userProfile));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void createOrUpdate(UserProfile userProfile) throws DataAccessException {
    profileSettings.userProfileMap.replace(userProfile.getId(), userProfile);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void delete(UserProfile userProfile) throws DataAccessException {
    UserProfile up = profileSettings.userProfileMap.remove(userProfile.getId());

    if (up == null) throw new DataAccessException(
        String.format("Failed to delete user profile '%s' because it does not exist.", userProfile));

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void flush() {
    try {
      UserProfileSettings.write(profileSettings, settingsFile);
    } catch (IOException e) {
      throw new RuntimeException("Unable to write user profile settings!", e);
    } catch (JAXBException e) {
      throw new RuntimeException("Unable to serialize user profile settings!", e);
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void evict(UserProfile userProfile) {
    try {
      delete(userProfile);
    } catch (DataAccessException e) {
      LOG.info("Unable to find userProfile '{}' for eviction.", userProfile);
    }
  }

  private void initSettingsFile() throws IOException, JAXBException {
    File parent = settingsFile.getParentFile();
    if (parent != null) parent.mkdirs();
    File temp = File.createTempFile("tmp", UUID.randomUUID().toString(), parent);
    temp.renameTo(settingsFile);

    profileSettings = new UserProfileSettings();
    UserProfileSettings.write(profileSettings, settingsFile);
  }

  /**
   * An inner static class that serves as the container for all user profile detail and provides a mechanism to read and
   * write the underlying XML file.
   */
  @XmlRootElement(name = "UserProfileSettings")
  private final static class UserProfileSettings {
    //JAXBContext is thread-safe
    private final static JAXBContext JAXB_CTXT = initJAXBCtxt();

    private ConcurrentHashMap<String, UserProfile> userProfileMap = new ConcurrentHashMap<String, UserProfile>();

    @XmlElement(name = "userProfile")
    public List<UserProfile> getUserProfiles() {
      return new ArrayList<UserProfile>(userProfileMap.values());
    }

    public void setUserProfiles(List<UserProfile> userProfiles) {
      for (UserProfile up : userProfiles) {
        userProfileMap.put(up.getId(), up);
      }
    }

    private static JAXBContext initJAXBCtxt() {
      try {
        return JAXBContext.newInstance(UserProfileSettings.class);
      } catch (JAXBException e) {
        throw new RuntimeException("Unable to setup JAXB context!", e);
      }
    }

    private static UserProfileSettings load(InputStream is) throws JAXBException {
      Unmarshaller jaxbUnmarshaller = JAXB_CTXT.createUnmarshaller();

      synchronized (UserProfileSettings.class) {
        return UserProfileSettings.class.cast(jaxbUnmarshaller.unmarshal(is));
      }
    }

    private static void write(UserProfileSettings userProfileSettings,
                                          File configFile) throws JAXBException, IOException {
      Marshaller jaxbMarshaller = JAXB_CTXT.createMarshaller();

      jaxbMarshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);

      synchronized (UserProfileSettings.class) {
        jaxbMarshaller.marshal(userProfileSettings, configFile);
      }
    }
  }
}
