/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.web.shiro;

import java.io.IOException;
import java.util.Collection;
import java.util.Timer;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.xml.bind.JAXBException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.apache.shiro.mgt.RealmSecurityManager;
import org.apache.shiro.realm.Realm;
import org.apache.shiro.web.env.EnvironmentLoaderListener;
import org.apache.shiro.web.env.WebEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.license.LicenseException;
import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.services.Utils;
import org.terracotta.management.resource.services.validator.RequestValidator;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.sun.jersey.api.client.filter.ClientFilter;
import com.terracotta.license.LicenseManager;
import com.terracotta.management.UpdateChecker;
import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.resource.services.validator.impl.AggregateEhcacheRequestValidator;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.KeychainInitializationException;
import com.terracotta.management.security.RequestIdentityAsserter;
import com.terracotta.management.security.RequestTicketMonitor;
import com.terracotta.management.security.SSLContextFactory;
import com.terracotta.management.security.SecurityContextManager;
import com.terracotta.management.security.impl.DfltRequestTicketMonitor;
import com.terracotta.management.security.impl.DfltSSLContextFactory;
import com.terracotta.management.security.impl.NoSecurityContextAuthorizer;
import com.terracotta.management.security.impl.ObfuscatedSecretFileStoreKeyChainAccessor;
import com.terracotta.management.security.services.SecurityContextSetupService;
import com.terracotta.management.security.services.impl.IniFileSetupService;
import com.terracotta.management.security.shiro.ShiroAuthorizer;
import com.terracotta.management.security.shiro.ShiroSecurityContextManager;
import com.terracotta.management.security.shiro.realm.TCIniRealm;
import com.terracotta.management.security.web.impl.LicensedIdentityAsserter;
import com.terracotta.management.security.web.impl.NoOpIdentityAsserter;
import com.terracotta.management.security.web.jersey.TMSRequestClientFilter;
import com.terracotta.management.security.web.jersey.TMSRequestSecurityClientFilter;
import com.terracotta.management.services.ConfigService;
import com.terracotta.management.services.JerseyClientFactory;
import com.terracotta.management.services.ResourceServiceClientService;
import com.terracotta.management.services.impl.DfltJerseyClientFactory;
import com.terracotta.management.services.impl.FileConfigService;
import com.terracotta.management.services.impl.JerseyResourceServiceClientService;
import com.terracotta.management.user.UserInfoFactory;
import com.terracotta.management.user.dao.impl.IniFileUserInfoDao;
import com.terracotta.management.user.impl.DfltUserInfoFactory;
import com.terracotta.management.userprofile.dao.UserProfileDao;
import com.terracotta.management.userprofile.dao.impl.XMLFileUserProfileDao;

/**
 * @author brandony
 */
public final class TMSEnvironmentLoaderListener extends EnvironmentLoaderListener {
  private static final long EVERY_WEEK = 7 * 24 * 60 * 60 * 1000;
  private static final long DELAY_UPDATE_CHECK = 1000;

  public static Boolean HAS_LICENSE;

  public static Boolean LICENSE_IS_COMMERCIAL_LICENSE;

  private static final Logger LOG = LoggerFactory.getLogger(TMSEnvironmentLoaderListener.class);
  
  private Timer updateCheckTimer;

  @Override
  public void contextInitialized(ServletContextEvent servletContextEvent) {
    checkLicense();
    
    ConfigService configSvc = new FileConfigService();
    UserInfoFactory usrInfoFactory = new DfltUserInfoFactory();
    UserProfileDao usrProfileDao;
    try {
      usrProfileDao = new XMLFileUserProfileDao();
    } catch (IOException e) {
      throw new RuntimeException(
          "Failure instantiating TMS context because user profile datasource could not be found or initialized.", e);
    } catch (JAXBException e) {
      throw new RuntimeException("Failure instantiating TMS context due to invalid user profile data format.", e);
    }
    this.loadContextAccordingToLicenseAndAuth(servletContextEvent, configSvc, usrInfoFactory,usrProfileDao);
    
    scheduleUpdateCheckIfNeeded();
  }

  void loadContextAccordingToLicenseAndAuth(ServletContextEvent servletContextEvent, ConfigService configSvc, UserInfoFactory usrInfoFactory, UserProfileDao usrProfileDao ) {
    if (HAS_LICENSE && configSvc.isAuthenticationEnabled() != null && configSvc.isAuthenticationEnabled()) {
      initEnvironment(servletContextEvent.getServletContext());
      loadLicensedCtxt(configSvc, usrInfoFactory, usrProfileDao, servletContextEvent.getServletContext());
    } else {
      loadUnlicensedCtxt(configSvc, usrInfoFactory, usrProfileDao);
    }
  }

  @Override
  public void contextDestroyed(ServletContextEvent servletContextEvent) {
    if (HAS_LICENSE) {
      destroyEnvironment(servletContextEvent.getServletContext());
    }
    ServiceLocator.unload();
    
    if (updateCheckTimer != null) {
      updateCheckTimer.cancel();
    }
  }

  private void checkLicense() {
    if (HAS_LICENSE == null) {
      try {
        LicenseManager.verifyTMCCapability();
        HAS_LICENSE = true;
        LICENSE_IS_COMMERCIAL_LICENSE = LicenseManager.isCommercialLicense();
      } catch (LicenseException e) {
        HAS_LICENSE = false;
      }
    }
  }

  private void loadLicensedCtxt(ConfigService configSvc,
                                UserInfoFactory usrInfoFactory,
                                UserProfileDao usrProfileDao,
                                ServletContext sc) {
    Authorizer authorizer = new ShiroAuthorizer();

    RequestTicketMonitor monitor = new DfltRequestTicketMonitor();

    KeyChainAccessor keyChainAccessor;
    try {
      keyChainAccessor = new ObfuscatedSecretFileStoreKeyChainAccessor();
    } catch (KeychainInitializationException e) {
      throw new RuntimeException(
          "Failure instantiating a licensed TMS (security enabled) due to invalid keychain configuration.", e);
    }

    SSLContextFactory sslCtxtFactory = null;
    // if ssl is configured with a keystore and truststore, we assume they are stored in the tc mgmt dir
    if (isSSLConfiguredWithKeyStoreAndTrustStore()) {
      LOG.info("SSL configuration with keystore and truststore found, registering a custom sslContextFactory using them");
      sslCtxtFactory = new DfltSSLContextFactory(keyChainAccessor,
        System.getProperty("user.home") + "/.tc/mgmt/tms-keystore",
        System.getProperty("user.home") + "/.tc/mgmt/tms-truststore");
    } else {
      LOG.info("SSL configuration with keystore and truststore NOT found, if you want to use your keystore and trustore, use -Dtms.ia.useSSL=true");
    }

    ClientFilter clientFilter = new TMSRequestSecurityClientFilter(monitor, keyChainAccessor);
    JerseyClientFactory clientFactory = new DfltJerseyClientFactory(clientFilter, sslCtxtFactory);
    

    ResourceServiceClientService rsrcSvcClientSvc = new JerseyResourceServiceClientService(configSvc, clientFactory,
        authorizer);

    ShiroSecurityContextManager secCtxtMgr;
    try {
      secCtxtMgr = new ShiroSecurityContextManager(new IniFileUserInfoDao(), getInitializedTCIniRealm(sc));
    } catch (DataAccessException e) {
      throw new RuntimeException(
          "Failure instantiating TMS context because user info datasource could not be found or initialized.", e);
    }

    SecurityContextSetupService secCtxtSetupSvc = new IniFileSetupService(secCtxtMgr, usrInfoFactory);

    RequestIdentityAsserter idAsserter = new LicensedIdentityAsserter(authorizer, monitor, secCtxtMgr,
        keyChainAccessor);

    RequestValidator requestValidator = new AggregateEhcacheRequestValidator(authorizer);
    
    ServiceLocator locator = new ServiceLocator().loadService(RequestValidator.class, requestValidator)
                    .loadService(ConfigService.class, configSvc)
                    .loadService(ResourceServiceClientService.class, rsrcSvcClientSvc)
                    .loadService(UserProfileDao.class, usrProfileDao)
                    .loadService(Authorizer.class, authorizer)
                    .loadService(RequestIdentityAsserter.class, idAsserter)
                    .loadService(SecurityContextManager.class, secCtxtMgr)
                    .loadService(SecurityContextSetupService.class, secCtxtSetupSvc);
    ServiceLocator.load(locator);
  }

  private void loadUnlicensedCtxt(ConfigService configSvc,
                                  UserInfoFactory usrInfoFactory,
                                  UserProfileDao usrProfileDao) {
    Authorizer authorizer = new NoSecurityContextAuthorizer();

    JerseyClientFactory clientFactory = new DfltJerseyClientFactory(new TMSRequestClientFilter(), null);

    ResourceServiceClientService rsrcSvcClientSvc = new JerseyResourceServiceClientService(configSvc, clientFactory,
        authorizer);

    RequestIdentityAsserter idAsserter = new NoOpIdentityAsserter(usrInfoFactory);

    RequestValidator requestValidator = new AggregateEhcacheRequestValidator(authorizer);
    
    ServiceLocator locator = new ServiceLocator().loadService(RequestValidator.class, requestValidator)
        .loadService(ConfigService.class, configSvc)
        .loadService(ResourceServiceClientService.class, rsrcSvcClientSvc)
        .loadService(UserProfileDao.class, usrProfileDao)
        .loadService(Authorizer.class, authorizer)
        .loadService(RequestIdentityAsserter.class, idAsserter);
    ServiceLocator.load(locator);
  }

  private TCIniRealm getInitializedTCIniRealm(ServletContext sc) {
    TCIniRealm tcIniRealm = null;

    WebEnvironment env = (WebEnvironment) sc.getAttribute(ENVIRONMENT_ATTRIBUTE_KEY);

    Collection<Realm> realms = ((RealmSecurityManager) env.getWebSecurityManager()).getRealms();

    for (Realm r : realms) {
      if (r instanceof TCIniRealm) {
        tcIniRealm = (TCIniRealm) r;
        break;
      }
    }

    if (tcIniRealm == null) {
      throw new RuntimeException("Failure instantiating TMS context. Failure to find expected security realm.");
    } else {
      return tcIniRealm;
    }
  }
  
  /**
   * Check whether or not SSL is configured with both a keystore and a truststore
   * 
   * @return true if SSL is configured with both a keystore and a truststore
   */
  boolean isSSLConfiguredWithKeyStoreAndTrustStore() {

    String property = System.getProperty("tms.ia.useSSL");
    if (property != null) {
      if ("true".equals(property)) {
        return true;
      } else {
        return false;
      }
    }
    boolean keystoreIsThere = false;
    boolean trustStoreIsThere = false;

    try {
      DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
      DocumentBuilder builder = domFactory.newDocumentBuilder();
      // jetty.xml has to be in etc/ ...
      Document doc = builder.parse(System.getProperty("jetty.home") + "/etc/jetty.xml");
      XPath xpath = XPathFactory.newInstance().newXPath();
      XPathExpression expr = xpath
          .compile("/Configure/Call[contains(@name,'addConnector')]/Arg/New[contains(@class,'org.eclipse.jetty.server.ssl.SslSelectChannelConnector')]/Arg/New[contains(@class,'org.eclipse.jetty.http.ssl.SslContextFactory')]");

      
      Object sslContextFactoryNode = expr.evaluate(doc, XPathConstants.NODESET);
      NodeList nodes = (NodeList) sslContextFactoryNode;
      Node sslContextFactoryNodeContent = nodes.item(0);
      if (sslContextFactoryNodeContent == null) {
        // it's possible that the user uses another class (org.eclipse.jetty.util.ssl) to configure the
        // sslcontextfactory
        expr = xpath
            .compile("/Configure/Call[contains(@name,'addConnector')]/Arg/New[contains(@class,'org.eclipse.jetty.server.ssl.SslSelectChannelConnector')]/Arg/New[contains(@class,'org.eclipse.jetty.util.ssl.SslContextFactory')]");
        sslContextFactoryNode = expr.evaluate(doc, XPathConstants.NODESET);
        nodes = (NodeList) sslContextFactoryNode;
        sslContextFactoryNodeContent = nodes.item(0);
      }
      NodeList childNodes = sslContextFactoryNodeContent.getChildNodes();
      for (int i = 0; i < childNodes.getLength(); i++) {
        Node childNode = childNodes.item(i);
        if (Utils.trimToNull(childNode.getTextContent()) != null) {
          if(!keystoreIsThere){
            keystoreIsThere = childNode.getAttributes().item(0).getNodeValue().equals("keyStore") == true ? true : false;
          }
          if(!trustStoreIsThere){
            trustStoreIsThere = childNode.getAttributes().item(0).getNodeValue().equals("trustStore") == true ? true : false;
          }
        }
      }
    }

    catch (Exception e) {
      LOG.warn("Could not find out whether the Identity Assertion is using SSL or not, not using SSL then. "
               + e.getMessage());
    }

    return keystoreIsThere && trustStoreIsThere;
  }
  
  private void scheduleUpdateCheckIfNeeded() {
    if (!Boolean.getBoolean("com.terracotta.management.skipUpdateCheck")) {
      updateCheckTimer = new Timer(true);
      updateCheckTimer.scheduleAtFixedRate(new UpdateChecker(HAS_LICENSE.booleanValue()), DELAY_UPDATE_CHECK, EVERY_WEEK);
    }
  }

}
