/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.resource.services.impl;

import com.terracotta.management.AggregateCallback;
import com.terracotta.management.AggregateResourceEnvelope;
import com.terracotta.management.services.ResourceServiceClientService;
import org.terracotta.management.ServiceLocator;
import org.terracotta.management.resource.VersionedEntity;
import org.terracotta.management.resource.services.validator.RequestValidator;

import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import java.util.List;
import java.util.Map;

/**
 * <p>
 * A support class for aggregate resource services that envelope their payloads to encapsulate shared logic.
 * </p>
 *
 * @author brandony
 */
public abstract class AggregateEnvelopingResourceServiceSupport<ENTITY extends VersionedEntity> {

  private final ResourceServiceClientService svcClientSvc;

  private final RequestValidator validator;

  public AggregateEnvelopingResourceServiceSupport() {
    this.svcClientSvc = ServiceLocator.locate(ResourceServiceClientService.class);
    this.validator = ServiceLocator.locate(RequestValidator.class);
  }

  /**
   * <p>
   * Support method for enveloped aggregate collection GETs.
   * </p>
   *
   * @param info     - the request uri information
   * @param callback - the {@link com.terracotta.management.AggregateCollectionCallback} implementing the aggregation strategy
   * @return the aggregate collection
   */
  protected AggregateResourceEnvelope doGet(UriInfo info,
                                            AggregateCallback<ENTITY> callback,
                                            Class<ENTITY> entity) {
    String[] agentIds = ResourceServiceUtils.buildAgentIds(info);

    UriBuilder builder = ResourceServiceUtils.cleanURI(info);

    for (Map.Entry<String, List<String>> entry : info.getQueryParameters().entrySet()) {
      for (String value : entry.getValue()) {
        builder.queryParam(entry.getKey(), value);
      }
    }

    getSvcClientSvc().proxyGet(callback, builder.build(), agentIds, entity);

    return callback.calculateResult();
  }

  /**
   * @return the {@link ResourceServiceClientService} to subclasses
   */
  protected ResourceServiceClientService getSvcClientSvc() {
    return svcClientSvc;
  }

  /**
   * Accessor to the {@link RequestValidator} for subclasses
   *
   * @return the validator
   */
  protected RequestValidator getValidator() {
    return validator;
  }
}
