/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.servlet;

import com.terracotta.license.LicenseManager;
import com.terracotta.management.security.SecurityContextManager;
import com.terracotta.management.services.TMSServiceLocator;
import org.terracotta.license.LicenseException;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.RequestDispatcher;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

import static com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener.HAS_LICENSE;
import static com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener.LICENSE_IS_TRIAL_LICENSE;

/**
 * A {@code Filter} that is aware of TMC setup requirements and redirects requests to certain pages when no valid
 * security context is available or when the TMC is running without a license.
 *
 * @author brandony
 */
public final class SetupAwareFilter implements Filter {
  private static final String TMC_HOME = "/index.jsp";

  private static final String TMC_LOGOUT_URL = "/logout.jsp";

  private SecurityContextManager securityCtxtMgr;

  public SetupAwareFilter() {
    SecurityContextManager.Locator securityCtxtMgrLocator = TMSServiceLocator.locator();
    this.securityCtxtMgr = securityCtxtMgrLocator.locateSecurityContextManager();
  }

  @Override
  public void init(FilterConfig filterConfig) throws ServletException {
    //no-op
  }

  @Override
  public void doFilter(ServletRequest servletRequest,
                       ServletResponse servletResponse,
                       FilterChain filterChain) throws IOException, ServletException {
    HttpServletRequest req = (HttpServletRequest) servletRequest;
    HttpServletResponse resp = (HttpServletResponse) servletResponse;

    if (HAS_LICENSE == null) {
      try {
        LicenseManager.verifyTMCCapability();
        HAS_LICENSE = true;
        LICENSE_IS_TRIAL_LICENSE = LicenseManager.isTrialLicense();
      } catch (LicenseException e) {
        HAS_LICENSE = false;
      }
    }

    if (HAS_LICENSE) {
      boolean ctxtValid = securityCtxtMgr.hasValidSecurityContext();
      if (!ctxtValid && req.getRequestURI()
          .startsWith(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL)) {
        RequestDispatcher rd = req.getSession().getServletContext()
            .getRequestDispatcher(AccountInitializationServlet.TMC_SETUP_URL);
        rd.forward(req, resp);
      } else if (ctxtValid && req.getRequestURI().startsWith(req.getContextPath() + "/setup")) {
        resp.sendRedirect(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL);
      } else {
        filterChain.doFilter(req, resp);
      }
    } else {
      if (req.getRequestURI().equals(req.getContextPath() + AccountInitializationServlet.TMC_LOGIN_URL) || req
          .getRequestURI().equals(req.getContextPath() + AccountInitializationServlet.TMC_SETUP_URL) || req
          .getRequestURI().equals(req.getContextPath() + TMC_LOGOUT_URL)) {
        resp.sendRedirect(req.getContextPath() + TMC_HOME);
      } else {
        filterChain.doFilter(req, resp);
      }
    }
  }

  @Override
  public void destroy() {
    //no-op
  }
}
