/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.services.impl;

import com.sun.jersey.api.client.AsyncWebResource;
import com.sun.jersey.api.client.Client;
import com.sun.jersey.api.client.GenericType;
import com.sun.jersey.api.client.UniformInterfaceException;
import com.sun.jersey.api.client.WebResource;
import com.sun.jersey.api.client.config.ClientConfig;
import com.sun.jersey.api.client.config.DefaultClientConfig;
import com.sun.jersey.api.json.JSONConfiguration;
import com.terracotta.management.AggregateCallback;
import com.terracotta.management.AggregateCollectionCallback;
import com.terracotta.management.config.Agent;
import com.terracotta.management.jaxrs.AggregateObjectMapperProvider;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.security.IACredentials;
import com.terracotta.management.security.web.shiro.TMSEnvironmentLoaderListener;
import com.terracotta.management.services.ConfigService;
import com.terracotta.management.services.JerseyClientFactory;
import com.terracotta.management.services.ResourceServiceClientService;
import net.sf.ehcache.management.resource.CacheEntity;
import net.sf.ehcache.management.resource.CacheManagerEntity;
import net.sf.ehcache.management.resource.CacheStatisticSampleEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.embedded.StandaloneServer;
import org.terracotta.management.resource.AgentEntity;
import org.terracotta.management.resource.AgentMetadataEntity;
import org.terracotta.management.resource.Representable;
import org.terracotta.management.resource.VersionedEntity;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import static com.terracotta.management.services.JerseyClientFactory.CLIENT_BASE_URI;
import static com.terracotta.management.services.JerseyClientFactory.CLIENT_CERT_AUTH_ENABLED;
import static com.terracotta.management.services.JerseyClientFactory.SECURITY_ENABLED;

/**
 * <p>
 * An implementation of the {@link ResourceServiceClientService} using Jersey Client.
 * </p>
 *
 * @author brandony
 */
public final class JerseyResourceServiceClientService implements ResourceServiceClientService, Observer {
  private static final String HTTPS_SCHEME = "https://";

  private static final String URI_CTXT_SEP = "/";

  private static final Logger LOG = LoggerFactory.getLogger(JerseyResourceServiceClientService.class);

  private final ConfigService configSvc;

  private final JerseyClientFactory clientFactory;

  private final Authorizer authorizer;

  private final ConcurrentHashMap<String, Client> clientsById;

  public JerseyResourceServiceClientService(ConfigService configSvc,
                                            JerseyClientFactory clientFactory,
                                            Authorizer authorizer) {
    this.configSvc = configSvc;
    this.configSvc.registerObserver(this);

    this.clientFactory = clientFactory;

    this.authorizer = authorizer;

    Collection<Agent> knownAgents = configSvc.getAgents();

    ConcurrentHashMap<String, Client> clientsById = new ConcurrentHashMap<String, Client>(knownAgents.size());

    for (Agent a : knownAgents) {
      clientsById.put(a.getId(), prepareClient(a));
    }

    this.clientsById = clientsById;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<String> getKnownAgentIds() {
    return Collections.unmodifiableSet(clientsById.keySet());
  }

  /**
   * {@inheritDoc}
   */
  //TODO: consolidate the common code between the two proxyGet methods.
  @Override
  public <RESOURCE extends VersionedEntity> void proxyGet(AggregateCallback<RESOURCE> callback,
                                                          URI request,
                                                          String[] agentIds,
                                                          Class<RESOURCE> entityClass) {
    Map<String, Future<RESOURCE>> futures = new HashMap<String, Future<RESOURCE>>(clientsById.size());

    Collection<String> ids = agentIds == null ? clientsById.keySet() : Arrays.asList(agentIds);

    for (String id : ids) {
      Client client = clientsById.get(id);
      if (client != null) {
        URI location = buildURI(client, request);
        AsyncWebResource resource = client.asyncResource(location);

        Future<RESOURCE> asyncCalc;
        try {
          asyncCalc = resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId()).get(entityClass);
        } catch (UniformInterfaceException e) {
          LOG.warn("Failure setting up async computation.", e);
          continue;
        }

        futures.put(id, asyncCalc);
      }
    }

    // Using the future as a latch for processing the request to the embedded resource.
    for (Entry<String, Future<RESOURCE>> entry : futures.entrySet()) {

      try {
        RESOURCE entity = entry.getValue().get();
        entity.setAgentId(entry.getKey());

        callback.addResponse(entity);
      } catch (InterruptedException e) {
        LOG.warn("Future interrupted.", e);
      } catch (ExecutionException e) {
        logAgentExecutionException(e);
      }
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <RESOURCE extends Representable> void proxyGet(AggregateCollectionCallback<RESOURCE> callback,
                                                        URI request,
                                                        String[] agentIds,
                                                        Class<RESOURCE> entityClass) {
    Map<String, Future<Collection<RESOURCE>>> futures = new HashMap<String, Future<Collection<RESOURCE>>>(
        clientsById.size());

    Collection<String> ids = agentIds == null ? clientsById.keySet() : Arrays.asList(agentIds);

    for (String id : ids) {
      Client client = clientsById.get(id);
      if (client != null) {
        URI location = buildURI(client, request);
        AsyncWebResource resource = client.asyncResource(location);

        Future<Collection<RESOURCE>> asyncCalc;
        try {
          asyncCalc = resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
              .get(buildGenericType(entityClass));
        } catch (UniformInterfaceException e) {
          LOG.warn("Failure setting up async computation.", e);
          continue;
        }

        futures.put(id, asyncCalc);
      }
    }

    // Using the future as a latch for processing the request to the embedded resource.
    for (Entry<String, Future<Collection<RESOURCE>>> entry : futures.entrySet()) {

      try {
        Collection<RESOURCE> entities = entry.getValue().get();

        for (RESOURCE entity : entities)
          entity.setAgentId(entry.getKey());

        callback.addResponses(entities);
      } catch (InterruptedException e) {
        LOG.warn("Future interrupted.", e);
      } catch (ExecutionException e) {
        logAgentExecutionException(e);
      }
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void proxyDelete(URI request,
                          String agentId) {
    Client client = clientsById.get(agentId);

    if (client != null) {
      WebResource resource = client.resource(buildURI(client, request));
      resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId()).delete();
    } else throw new WebApplicationException(Response.Status.GONE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <RESOURCE> void proxyPut(URI request,
                                  RESOURCE entity,
                                  String agentId) {
    Client client = clientsById.get(agentId);

    if (client != null) {
      WebResource resource = client.resource(buildURI(client, request));
      resource.type(MediaType.APPLICATION_JSON_TYPE).header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
          .put(entity);
    } else throw new WebApplicationException(Response.Status.GONE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public <RESOURCE> Response proxyPost(URI request,
                                       RESOURCE entity,
                                       String agentId) {
    Client client = clientsById.get(agentId);

    if (client != null) {
      URI location = buildURI(client, request);
      WebResource resource = client.resource(location);
      resource.header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId()).post(entity);
      return Response.created(location).build();
    } else throw new WebApplicationException(Response.Status.GONE);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void update(Observable o,
                     Object id) {
    String agentId = String.class.cast(id);
    Agent agent = configSvc.getAgent(agentId);

    if (agent == null) clientsById.remove(agentId);
    else {
      Client current = clientsById.get(agentId);

      if(TMSEnvironmentLoaderListener.HAS_LICENSE) agent.setSecured(true);

      Client newClient = prepareClient(agent);

      URI checkClientURI;

      try {
        checkClientURI = new URI("agents/info");
      } catch (URISyntaxException e) {
        throw new RuntimeException("BUG Alert! Invalid agent info URI.", e);
      }

      URI checkLocation = buildURI(newClient, checkClientURI);
      WebResource resource = newClient.resource(checkLocation);

      boolean available = true;
      Collection<AgentMetadataEntity> ames;
      try{
        ames = (Collection<AgentMetadataEntity>) resource
          .header(IACredentials.TC_ID_TOKEN, authorizer.getSessionId())
          .get(buildGenericType(AgentMetadataEntity.class));
        if (!ames.isEmpty()) {
          AgentMetadataEntity ame = ames.iterator().next();

          if(agent.update(ame)) {
            newClient = prepareClient(agent);
          }
        }
      } catch (RuntimeException e) {
        available = false;
      }

      if (current != null) {
        clientsById.replace(agentId, current, newClient);
      } else {
        clientsById.putIfAbsent(agentId, newClient);
      }

      if(!available) {
        throw new AgentNotAvailableException("REST service not currently available or there is a problem with the configuration.");
      }
    }
  }

  private void logAgentExecutionException(ExecutionException e) {
    if (LOG.isDebugEnabled()) {
      LOG.warn("Failed to execute request against monitorable entity.", e);
    } else {
      LOG.warn("Failed to execute request against monitorable entity. {}", e.getMessage());
    }
  }

  private static URI buildURI(Client client,
                              URI request) {
    URI baseUri = (URI) client.getProperties().get(CLIENT_BASE_URI);

    if (baseUri == null) {
      throw new RuntimeException("Bug Alert! Invalid aggregator configuration: missing client base URI.");
    }

    return baseUri.resolve(request);
  }

  private Client prepareClient(Agent a) {
    ClientConfig clientConfig = new DefaultClientConfig();
    clientConfig.getFeatures().put(JSONConfiguration.FEATURE_POJO_MAPPING, Boolean.TRUE);
    clientConfig.getClasses().add(AggregateObjectMapperProvider.class);

    Integer cto = a.getConnectionTimeoutMillis();
    if (cto != null) clientConfig.getProperties().put(ClientConfig.PROPERTY_CONNECT_TIMEOUT, cto);

    Integer rto = a.getReadTimeoutMillis();
    if (rto != null) clientConfig.getProperties().put(ClientConfig.PROPERTY_READ_TIMEOUT, rto);

    URI clientBase;
    try {
      clientBase = extractBaseURI(a);
    } catch (URISyntaxException e) {
      throw new RuntimeException("BUG Alert! Failed to extract valid URI from agent while setting up client. ");
    }

    clientConfig.getProperties().put(CLIENT_BASE_URI, clientBase);

    clientConfig.getProperties()
        .put(CLIENT_CERT_AUTH_ENABLED, clientBase.toString().startsWith(HTTPS_SCHEME) ? a.isClientAuthEnabled() : false);
    clientConfig.getProperties().put(SECURITY_ENABLED, a.isSecured());

    return clientFactory.createClient(clientConfig);
  }

  private static URI extractBaseURI(Agent agent) throws URISyntaxException {
    URL agentURL;
    try {
      agentURL = new URL(agent.getAgentLocation());
    } catch (MalformedURLException e) {
      throw new RuntimeException("BUG Alert! Failed to retrieve agent location URL while setting up client.");
    }

    String path = agentURL.getPath();

    if (!path.matches(".*" + StandaloneServer.EMBEDDED_CTXT + "/$")) {
      path += StandaloneServer.EMBEDDED_CTXT + URI_CTXT_SEP;
    } else if (!path.endsWith(URI_CTXT_SEP)) {
      path += URI_CTXT_SEP;
    }

    return new URI(agentURL.getProtocol() + "://" + agentURL.getAuthority() + path);
  }

  private static GenericType buildGenericType(Class<? extends Representable> clazz) {
    if (clazz == AgentEntity.class) return new GenericType<Collection<AgentEntity>>() {
    };
    if (clazz == AgentMetadataEntity.class) return new GenericType<Collection<AgentMetadataEntity>>() {
    };
    if (clazz == CacheManagerEntity.class) return new GenericType<Collection<CacheManagerEntity>>() {
    };
    if (clazz == CacheEntity.class) return new GenericType<Collection<CacheEntity>>() {
    };
    if (clazz == CacheStatisticSampleEntity.class) return new GenericType<Collection<CacheStatisticSampleEntity>>() {
    };
    else throw new UnsupportedOperationException(
        String.format("Unknown entity class \"%s\" for GenericType construction.", clazz.getCanonicalName()));
  }
}