/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package com.terracotta.management.services.impl;

import com.terracotta.management.config.Agent;
import com.terracotta.management.config.Config;
import com.terracotta.management.services.ConfigService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;

import javax.xml.bind.JAXBException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Observer;
import java.util.Set;

/**
 * @author brandony
 */
public final class FileConfigService implements ConfigService {
  private static final Logger LOG = LoggerFactory.getLogger(ConfigService.class);

  
  /**
   * A constant for the default user profile settings file location.
   */
  private static final File DFLT_CONFIG_FILE = new File(System.getProperty("user.home") + "/.tc/mgmt/config.xml");

  private final File configFile;

  private final Config config;

  public FileConfigService() {
    String configProp = System.getProperty("com.tc.management.config");
    configFile = configProp == null ? DFLT_CONFIG_FILE : new File(configProp);

    String templateConfigFileName = System.getProperty("com.tc.management.config.tmplt", "templateConfig.xml");

    createConfigFromTemplateIfNotExists(templateConfigFileName, configFile);
    Config lConfig = null;
    try {
      lConfig = loadConfigFromFile(configFile);
    } catch (IOException e) {
      LOG.warn("Config file not found.  Creating empty one.");
    }

    // use the loaded config, or fall back to empty config
    config = lConfig == null ? new Config() : lConfig;
  }
  
  void createConfigFromTemplateIfNotExists(String tmpltConfigFileName, File cfgFile) {
    // build the config from template if it does not exist
    if (!cfgFile.exists()) {
      LOG.info("Config file does not exists.  Creating one from template: " + tmpltConfigFileName);
      try {
        copyConfigFileFromClassPathResource(tmpltConfigFileName, cfgFile);
      }
      catch(IOException e) {
        // only log, because we'll try more tactics below.
        LOG.warn("Failed to copy template config.", e);
      }
    }
  }

  @Override
  public Agent getAgent(String agentId) {
    return config.getAgent(agentId);
  }

  @Override
  public Collection<Agent> getAgents() {
    List<Agent> agents = new ArrayList<Agent>();

    Set<String> groups = config.getGroups();
    for (String group : groups) {
      Collection<Agent> agentsByGroup = getAgentsByGroup(group);
      agents.addAll(agentsByGroup);
    }

    return agents;
  }

  @Override
  public Collection<Agent> getAgentsByGroup(String groupId) {
    return config.getAgentsByGroup(groupId);
  }

  @Override
  public Collection<String> getGroups() {
    return config.getGroups();
  }

  @Override
  public boolean addGroup(String groupId) {
    return config.addGroup(groupId);
  }

  @Override
  public boolean deleteGroup(String groupId) {
    return config.deleteGroup(groupId);
  }

  @Override
  public boolean renameGroup(String oldGroupId,
                             String newGroupId) {
    return config.renameGroup(oldGroupId, newGroupId);
  }

  @Override
  public void deleteAgent(String agentId) {
    config.deleteAgent(agentId);
  }

  @Override
  public boolean addAgent(Agent agent) {
    return config.addAgent(agent);
  }

  @Override
  public boolean updateAgent(String agentId,
                             Agent agent) {
    return config.updateAgent(agentId, agent);
  }

  @Override
  public boolean saveConfig() throws ServiceExecutionException {
    boolean success = false;

    try {
      storeConfigInFile(config, configFile);
    } catch (IOException e) {
      throw new ServiceExecutionException("Failed to save config file.", e);
    } catch (JAXBException e) {
      throw new ServiceExecutionException("Failed to serialize config.", e);
    }

    return success;
  }

  @Override
  public void registerObserver(Observer observer) {
    config.addObserver(observer);
  }
  
  void storeConfigInFile(Config cfg, File cfgFile) throws IOException, JAXBException {
    
    boolean existingConfig = cfgFile.exists();
    
    File backFile = new File(cfgFile.getAbsolutePath() + "_tmp" + System.currentTimeMillis());
    if(existingConfig) {
      LOG.debug("Backing up config to: '" + backFile.getAbsolutePath() + "'");
      copyConfigFile(cfgFile, backFile);
      if (!cfgFile.delete()) {
        throw new IOException("Unable to delete existing configFile '" + cfgFile + "'");
      }
    }
    else {
      if (!cfgFile.getParentFile().exists() && !cfgFile.getParentFile().mkdirs()) {
        throw new IOException("Unable to create path '" + cfgFile.getParentFile() + "'");
      }
    }

    LOG.info("Writing config to: '" + cfgFile.getAbsolutePath() + "'");
    Config.write(cfg, cfgFile);

    if(existingConfig) {
      LOG.debug("Removing back up config: '" + backFile.getAbsolutePath() + "'");
      if (!backFile.delete()) {
        throw new IOException("Unable to delete backup file '" + backFile + "'");
      }
    }
  }

  Config loadConfigFromFile(File src) throws IOException {
    if (src.exists()) {
      FileInputStream fis;

      try {
        fis = new FileInputStream(src);
      } catch (FileNotFoundException e) {
        throw new IOException("Failure loading configuration file: " + src.getAbsolutePath(), e);
      }
      
      return loadConfigFromInputStream(fis);
    }
    return null; 
  }

  Config loadConfigFromInputStream(InputStream src) throws IOException {
    try {
      return Config.load(src);
    } catch (JAXBException e) {
      throw new IOException("Failure loading configuration input stream.", e);
    } finally {
      try {
        src.close();
      } catch (IOException e1) {
        LOG.warn("Failed to close config file input stream.", e1);
      }
    }
  }

  private void copyConfigFile(File sourceFile, File destFile) throws IOException {
    if(!sourceFile.exists()) {
      throw new IOException("Source file for copy " + sourceFile.getAbsolutePath() + " does not exist.");
    }
    try {
      Config tmpltConfig = loadConfigFromFile(sourceFile);
      storeConfigInFile(tmpltConfig, destFile);
    } catch(JAXBException e) {
      throw new IOException("Error serializing template config", e);
    }
  }

  private void copyConfigFileFromClassPathResource(String resourceName, File destFile) throws IOException {
    InputStream src = FileConfigService.class.getClassLoader().getResourceAsStream(resourceName);
    try {
      Config tmpltConfig = loadConfigFromInputStream(src);
      storeConfigInFile(tmpltConfig, destFile);
    } catch(JAXBException e) {
      throw new IOException("Error serializing template config", e);
    }
  }
}
