/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security.web.shiro;

import java.io.IOException;
import java.util.Collection;

import javax.servlet.ServletContextEvent;
import javax.xml.bind.JAXBException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.apache.shiro.mgt.RealmSecurityManager;
import org.apache.shiro.realm.Realm;
import org.apache.shiro.web.env.EnvironmentLoaderListener;
import org.apache.shiro.web.env.WebEnvironment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.license.LicenseException;
import org.terracotta.management.ServiceLocator;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import com.sun.jersey.api.client.filter.ClientFilter;
import com.terracotta.license.LicenseManager;
import com.terracotta.management.dao.DataAccessException;
import com.terracotta.management.resource.services.validator.impl.AggregateEhcacheRequestValidator;
import com.terracotta.management.security.Authorizer;
import com.terracotta.management.security.KeyChainAccessor;
import com.terracotta.management.security.KeychainInitializationException;
import com.terracotta.management.security.RequestIdentityAsserter;
import com.terracotta.management.security.RequestTicketMonitor;
import com.terracotta.management.security.SSLContextFactory;
import com.terracotta.management.security.impl.DfltRequestTicketMonitor;
import com.terracotta.management.security.impl.DfltSSLContextFactory;
import com.terracotta.management.security.impl.NoSecurityContextAuthorizer;
import com.terracotta.management.security.impl.ObfuscatedSecretFileStoreKeyChainAccessor;
import com.terracotta.management.security.services.SecurityContextSetupService;
import com.terracotta.management.security.services.impl.IniFileSetupService;
import com.terracotta.management.security.shiro.ShiroAuthorizer;
import com.terracotta.management.security.shiro.ShiroSecurityContextManager;
import com.terracotta.management.security.shiro.realm.TCIniRealm;
import com.terracotta.management.security.web.impl.LicensedIdentityAsserter;
import com.terracotta.management.security.web.impl.NoOpIdentityAsserter;
import com.terracotta.management.security.web.jersey.TMSRequestClientFilter;
import com.terracotta.management.security.web.jersey.TMSRequestSecurityClientFilter;
import com.terracotta.management.services.ConfigService;
import com.terracotta.management.services.JerseyClientFactory;
import com.terracotta.management.services.ResourceServiceClientService;
import com.terracotta.management.services.TMSServiceLocator;
import com.terracotta.management.services.impl.DfltJerseyClientFactory;
import com.terracotta.management.services.impl.FileConfigService;
import com.terracotta.management.services.impl.JerseyResourceServiceClientService;
import com.terracotta.management.user.UserInfoFactory;
import com.terracotta.management.user.dao.impl.IniFileUserInfoDao;
import com.terracotta.management.user.impl.DfltUserInfoFactory;
import com.terracotta.management.userprofile.dao.UserProfileDao;
import com.terracotta.management.userprofile.dao.impl.XMLFileUserProfileDao;

/**
 * @author brandony
 */
public final class TMSEnvironmentLoaderListener extends EnvironmentLoaderListener {

  public static Boolean HAS_LICENSE;

  public static Boolean LICENSE_IS_TRIAL_LICENSE;

  private static final Logger LOG = LoggerFactory.getLogger(TMSEnvironmentLoaderListener.class);

  @Override
  public void contextInitialized(ServletContextEvent servletContextEvent) {
    checkLicense();

    ConfigService configSvc = new FileConfigService();
    UserInfoFactory usrInfoFactory = new DfltUserInfoFactory();

    UserProfileDao usrProfileDao;
    try {
      usrProfileDao = new XMLFileUserProfileDao();
    } catch (IOException e) {
      throw new RuntimeException(
          "Failure instantiating TMS context because user profile datasource could not be found or initialized.", e);
    } catch (JAXBException e) {
      throw new RuntimeException("Failure instantiating TMS context due to invalid user profile data format.", e);
    }

    if (HAS_LICENSE) {
      initEnvironment(servletContextEvent.getServletContext());
      loadLicensedCtxt(configSvc, usrInfoFactory, usrProfileDao, servletContextEvent);
    } else {
      loadUnlicensedCtxt(configSvc, usrInfoFactory, usrProfileDao);
    }
  }

  @Override
  public void contextDestroyed(ServletContextEvent servletContextEvent) {
    if (HAS_LICENSE) {
      destroyEnvironment(servletContextEvent.getServletContext());
    }
  }

  private void checkLicense() {
    if (HAS_LICENSE == null) {
      try {
        LicenseManager.verifyTMCCapability();
        HAS_LICENSE = true;
        LICENSE_IS_TRIAL_LICENSE = LicenseManager.isTrialLicense();
      } catch (LicenseException e) {
        HAS_LICENSE = false;
      }
    }
  }

  private void loadLicensedCtxt(ConfigService configSvc,
                                UserInfoFactory usrInfoFactory,
                                UserProfileDao usrProfileDao,
                                ServletContextEvent sce) {
    Authorizer authorizer = new ShiroAuthorizer();

    RequestTicketMonitor monitor = new DfltRequestTicketMonitor();

    KeyChainAccessor keyChainAccessor;
    try {
      keyChainAccessor = new ObfuscatedSecretFileStoreKeyChainAccessor();
    } catch (KeychainInitializationException e) {
      throw new RuntimeException(
          "Failure instantiating a licensed TMS (security enabled) due to invalid keychain configuration.", e);
    }

    SSLContextFactory sslCtxtFactory = null;
    if (isSSLEnabledForIA()) {
      LOG.info("IdentityAssertion will use SSL");
      sslCtxtFactory = new DfltSSLContextFactory(keyChainAccessor,
        System.getProperty("user.home") + "/.tc/mgmt/tms-keystore",
        System.getProperty("user.home") + "/.tc/mgmt/tms-truststore");
    } 

    ClientFilter clientFilter = new TMSRequestSecurityClientFilter(monitor, keyChainAccessor);
    JerseyClientFactory clientFactory = new DfltJerseyClientFactory(clientFilter, sslCtxtFactory);
    

    ResourceServiceClientService rsrcSvcClientSvc = new JerseyResourceServiceClientService(configSvc, clientFactory,
        authorizer);

    ShiroSecurityContextManager secCtxtMgr;
    try {
      secCtxtMgr = new ShiroSecurityContextManager(new IniFileUserInfoDao(), getInitializedTCIniRealm(sce));
    } catch (DataAccessException e) {
      throw new RuntimeException(
          "Failure instantiating TMS context because user info datasource could not be found or initialized.", e);
    }

    SecurityContextSetupService secCtxtSetupSvc = new IniFileSetupService(secCtxtMgr, usrInfoFactory);

    RequestIdentityAsserter idAsserter = new LicensedIdentityAsserter(authorizer, monitor, secCtxtMgr,
        keyChainAccessor);


    ServiceLocator.load(
        new TMSServiceLocator(new AggregateEhcacheRequestValidator(authorizer), configSvc, rsrcSvcClientSvc,
            usrProfileDao, authorizer, idAsserter, secCtxtMgr, secCtxtSetupSvc));
  }

  private void loadUnlicensedCtxt(ConfigService configSvc,
                                  UserInfoFactory usrInfoFactory,
                                  UserProfileDao usrProfileDao) {
    Authorizer authorizer = new NoSecurityContextAuthorizer();

    JerseyClientFactory clientFactory = new DfltJerseyClientFactory(new TMSRequestClientFilter(), null);

    ResourceServiceClientService rsrcSvcClientSvc = new JerseyResourceServiceClientService(configSvc, clientFactory,
        authorizer);

    RequestIdentityAsserter idAsserter = new NoOpIdentityAsserter(usrInfoFactory);

    ServiceLocator.load(
        new TMSServiceLocator(new AggregateEhcacheRequestValidator(authorizer), configSvc, rsrcSvcClientSvc,
            usrProfileDao, authorizer, idAsserter, null, null));
  }

  private TCIniRealm getInitializedTCIniRealm(ServletContextEvent sce) {
    TCIniRealm tcIniRealm = null;

    WebEnvironment env = (WebEnvironment) sce.getServletContext().getAttribute(ENVIRONMENT_ATTRIBUTE_KEY);

    Collection<Realm> realms = ((RealmSecurityManager) env.getWebSecurityManager()).getRealms();

    for (Realm r : realms) {
      if (r instanceof TCIniRealm) {
        tcIniRealm = (TCIniRealm) r;
        break;
      }
    }

    if (tcIniRealm == null) {
      throw new RuntimeException("Failure instantiating TMS context. Failure to find expected security realm.");
    } else {
      return tcIniRealm;
    }
  }
  
  /**
   * Check whether or not SSL will be used for Identity Assertion
   * 
   * @return true if SSL will be used for IA
   */
  public boolean isSSLEnabledForIA() {

    String property = System.getProperty("tms.ia.useSSL");
    if (property != null) {
      if ("true".equals(property)) {
        return true;
      } else {
        return false;
      }
    }

    try {
      DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
      DocumentBuilder builder = domFactory.newDocumentBuilder();
      // jetty.xml has to be in etc/ ...
      Document doc = builder.parse("etc/jetty.xml");
      XPath xpath = XPathFactory.newInstance().newXPath();
      XPathExpression expr = xpath
          .compile("/Configure/Call[contains(@name,'addConnector')]/Arg/New[contains(@class,'org.eclipse.jetty.server.ssl.SslSelectChannelConnector')]/Arg/New/Set[contains(@name,'needClientAuth')]");

      Object needClientAuthTags = expr.evaluate(doc, XPathConstants.NODESET);
      NodeList nodes = (NodeList) needClientAuthTags;
      for (int i = 0; i < nodes.getLength(); i++) {
        if ("true".equals(nodes.item(i).getTextContent())) { return true; }
      }
    }

    catch (Exception e) {
      LOG.warn("Could not find out whether the Identity Assertion is using SSL or not, not using SSL then. "
               + e.getMessage());
    }

    return false;
  }

}
