/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */

package com.terracotta.management.security;

import com.terracotta.management.user.UserRole;

import java.util.Set;

/**
 * An interface for objects that can verify the authenticated Subject's authorization.
 *
 * @author brandony
 */
public interface Authorizer {
  interface Locator {
    Authorizer locateAuthorizer();
  }

  /**
   * Check to see if the authenticated Subject has the specified {@link UserRole}.
   * 
   * @param role to be checked
   * @return true if Subject has the submitted role, false otherwise
   */
  boolean isUserInRole(UserRole role);

  /**
   * Check to see if the authenticated Subject has all the specified {@link UserRole}s.
   * 
   * @param roles to be checked
   * @return true if Subject has all the submitted roles, false otherwise
   */
  boolean isUserInAllRoles(Set<UserRole> roles);

  /**
   * Check to see if the authenticated Subject has any of the specified {@link UserRole}s.
   *
   * @param roles to be checked
   * @return true if Subject has any of the submitted roles, false otherwise
   */
  boolean isUserInAnyRole(Set<UserRole> roles);

  /**
   * Gets the authenticated Subject's principal.
   * 
   * @return the username for the authenticated Subject
   */
  String getPrincipal();

  /**
   * Gets the principal associated with the submitted session id. If the {@code isAuthenticated} boolean parameter is
   * set to {@code true} the principal name will only be returned if the subject that is associated with it is
   * authenticated.
   * </p>
   * If the {@code isAuthenticated} boolean parameter is {@code false} the principal will be returned for any valid
   * session that has that id.
   *
   * @param sessionId the session
   * @param isAuthenticated
   * @return
   */
  String getPrincipalBySessionId(String sessionId, boolean isAuthenticated);

  String getSessionId();


}
