/*
 * All content copyright (c) 2003-2012 Terracotta, Inc., except as may otherwise be noted in a separate copyright notice.  All rights reserved.
 */
package com.terracotta.management.config.resource.services;

import com.terracotta.management.config.Agent;
import com.terracotta.management.services.ConfigService;
import com.terracotta.management.services.TMSServiceLocator;
import com.terracotta.management.services.impl.AgentNotAvailableException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.management.ServiceExecutionException;
import org.terracotta.management.resource.services.Utils;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * <p>A resource service for configuration of the aggregator management web service proxy.</p>
 *
 * @author brandony
 */
@Path("/config")
public final class ConfigResourceService {
  private static final Logger LOG = LoggerFactory.getLogger(ConfigResourceService.class);

  @Context
  private UriInfo info;

  private final ConfigService configSvc;

  public ConfigResourceService() {
    ConfigService.Locator configSvcLocator = TMSServiceLocator.locator();
    this.configSvc = configSvcLocator.locateConfigService();
  }

  @POST
  public void save() {
    try {
      configSvc.saveConfig();
    } catch (ServiceExecutionException e) {
      LOG.error(e.getMessage(), e.getCause());
      throw new WebApplicationException(
          Response.status(Response.Status.INTERNAL_SERVER_ERROR).entity(e.getMessage()).build());
    }
  }

  /**
   * Get all the {@link Agent}s of a group.
   *
   * @param groupId the group ID
   * @return a collection of all the agents of the specified group
   */
  @GET
  @Path("/groups/{groupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Collection<Agent> getAgents(@PathParam("groupId") String groupId) {
    return configSvc.getAgentsByGroup(groupId);
  }

  /**
   * <p>Get all agent group names.</p>
   *
   * @return a collection of all agent group names
   */
  @GET
  @Path("/groups")
  @Produces(MediaType.APPLICATION_JSON)
  public Collection<String> getGroups() {
    return configSvc.getGroups();
  }

  @PUT
  @Path("/groups/{groupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Response addGroup(@PathParam("groupId") String groupId) {
    boolean added = configSvc.addGroup(groupId);
    if (added) {
      return Response.status(Response.Status.OK).entity("Group created.").build();
    } else {
      return Response.status(Response.Status.CONFLICT).entity("Group already exists.").build();
    }
  }

  @PUT
  @Path("/groups/{oldGroupId}/{newGroupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Response addGroup(@PathParam("oldGroupId") String oldGroupId,
                           @PathParam("newGroupId") String newGroupId) {
    boolean renamed = configSvc.renameGroup(oldGroupId, newGroupId);
    if (renamed) {
      return Response.status(Response.Status.OK).entity("Group renamed.").build();
    } else {
      return Response.status(Response.Status.CONFLICT)
          .entity("Group source group does not exist or destination group already exists.").build();
    }
  }

  @DELETE
  @Path("/groups/{groupId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Response deleteGroup(@PathParam("groupId") String groupId) {
    boolean deleted = configSvc.deleteGroup(groupId);
    if (deleted) {
      return Response.status(Response.Status.OK).entity("Group deleted.").build();
    } else {
      return Response.status(Response.Status.NOT_FOUND).entity("Group does not exist or is not empty.").build();
    }
  }

  /**
   * <p>Get the current {@link Agent} configuration resource objects known by the TMS.</p>
   *
   * @return current config
   */
  @GET
  @Path("/agents")
  @Produces(MediaType.APPLICATION_JSON)
  public Collection<Agent> getAgents() {
    List<Agent> agents = new ArrayList<Agent>();

    Collection<String> groups = configSvc.getGroups();
    for (String groupId : groups) {
      Collection<Agent> agentsByGroup = configSvc.getAgentsByGroup(groupId);
      agents.addAll(agentsByGroup);
    }

    return agents;
  }

  /**
   * <p> </p>
   *
   * @param agent
   * @return
   */
  @POST
  @Path("/agents")
  @Consumes(MediaType.APPLICATION_JSON)
  public Response addAgent(Agent agent) {
    Response.ResponseBuilder responseBuilder;

    try {
      new URL(agent.getAgentLocation());
    } catch (MalformedURLException e) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity(String.format("Invalid connection URL, %s", e.getMessage()));
    }

    if (Utils.trimToNull(agent.getName()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing name.");
    } else if (Utils.trimToNull(agent.getGroupId()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing groupId.");
    } else if (agent.getAgentLocation() == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing agentLocation.");
    } else if (!isValidURL(agent.getAgentLocation())) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Invalid agentLocation.");
    } else {
      try{
        if (configSvc.addAgent(agent)) {
          UriBuilder builder = info.getRequestUriBuilder();
          builder.segment(agent.getId());
          responseBuilder = Response.status(Response.Status.CREATED).header("LOCATION", builder.build());
        } else {
          responseBuilder = Response.status(Response.Status.CONFLICT);
        }
      } catch (AgentNotAvailableException e) {
        responseBuilder = Response.status(Response.Status.GONE);
      }
    }

    return responseBuilder.build();
  }

  @GET
  @Path("/agents/{agentId}")
  @Produces(MediaType.APPLICATION_JSON)
  public Agent getAgent(@PathParam("agentId") String agentId) {
    return configSvc.getAgent(agentId);
  }

  @DELETE
  @Path("/agents/{agentId}")
  public void deleteAgent(@PathParam("agentId") String agentId) {
    configSvc.deleteAgent(agentId);
  }


  @PUT
  @Path("/agents/{agentId}")
  @Consumes(MediaType.APPLICATION_JSON)
  public Response updateAgent(@PathParam("agentId") String agentId,
                              Agent agent) {
    Response.ResponseBuilder responseBuilder;

    if (Utils.trimToNull(agent.getName()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing name.");
    } else if (Utils.trimToNull(agent.getGroupId()) == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing groupId.");
    } else if (agent.getAgentLocation() == null) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Missing agentLocation.");
    } else if (!isValidURL(agent.getAgentLocation())) {
      responseBuilder = Response.status(Response.Status.BAD_REQUEST)
          .entity("Invalid agent representation: Invalid agentLocation.");
    } else {
      try {
        if (configSvc.updateAgent(agentId, agent)) {
          responseBuilder = Response.ok();
        } else {
          responseBuilder = Response.status(Response.Status.NOT_MODIFIED);
        }
        } catch (AgentNotAvailableException e) {
          responseBuilder = Response.status(Response.Status.GONE);
      }
    }

    return responseBuilder.build();
  }

  private boolean isValidURL(String agentLocation) {
    boolean valid = true;

    try {
      new URL(agentLocation);
    } catch (MalformedURLException e) {
      valid = false;
    }

    return valid;
  }
}
