package com.bigmemory.samples.search;

import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import net.sf.ehcache.config.CacheConfiguration;
import net.sf.ehcache.config.Configuration;
import net.sf.ehcache.config.MemoryUnit;
import net.sf.ehcache.config.SearchAttribute;
import net.sf.ehcache.config.Searchable;
import net.sf.ehcache.search.Attribute;
import net.sf.ehcache.search.Direction;
import net.sf.ehcache.search.Query;
import net.sf.ehcache.search.Result;
import net.sf.ehcache.search.Results;
import net.sf.ehcache.search.aggregator.Aggregators;
import net.sf.ehcache.search.attribute.AttributeExtractor;
import net.sf.ehcache.search.attribute.AttributeExtractorException;

import com.bigmemory.commons.model.Person;
import com.bigmemory.commons.model.Person.Gender;

import java.io.IOException;

import static com.bigmemory.commons.util.ReadUtil.waitForInput;

/**
 * <p>Sample app briefly showing some of the api's one can use to search in
 * BigMemory.
 * <p/>
 * <p>Released to the public domain, as explained at  http://creativecommons.org/publicdomain/zero/1.0/
 * </p>
 *
 * @author steve
 * @author Mathilde Lemee
 */
public class BigMemorySearch {

  public static void main(String[] args) throws IOException {


    // Create Cache
    Configuration managerConfig = new Configuration()
        .cache(new CacheConfiguration().name("bm-sample")
            .eternal(true)
            .maxBytesLocalHeap(128, MemoryUnit.MEGABYTES)
            .maxBytesLocalOffHeap(1, MemoryUnit.GIGABYTES)
            .searchable(new Searchable()
                .searchAttribute(new SearchAttribute().name("age"))
                .searchAttribute(new SearchAttribute().name("gender").expression("value.getGender()"))
                .searchAttribute(new SearchAttribute().name("state").expression("value.getAddress().getState()"))
                .searchAttribute(new SearchAttribute().name("name").className(NameAttributeExtractor.class.getName()))
            )
        );

    CacheManager manager = CacheManager.create(managerConfig);
    Ehcache bm = manager.getEhcache("bm-sample");

    try {
      loadCache(bm);


      Attribute<Integer> age = bm.getSearchAttribute("age");
      Attribute<Gender> gender = bm.getSearchAttribute("gender");
      Attribute<String> name = bm.getSearchAttribute("name");
      Attribute<String> state = bm.getSearchAttribute("state");

      Query query = bm.createQuery();
      query.includeKeys();
      query.includeValues();
      query.addCriteria(name.ilike("Jul*").and(gender.eq(Gender.FEMALE)))
          .addOrderBy(age, Direction.ASCENDING).maxResults(10);

      System.out
          .println("**** Searching for all Person's who's name start with Jul and are Female: ****");

      Results results = query.execute();
      System.out.println(" Size: " + results.size());
      System.out.println("----Results-----\n");
      for (Result result : results.all()) {
        System.out.println("Got: Key[" + result.getKey()
                           + "] Value class [" + result.getValue().getClass()
                           + "] Value [" + result.getValue() + "]");
      }

      waitForInput();

      System.out.println("**** Adding another female Julie ****");

      bm.put(new Element(1, new Person("Julie Smith", 36, Gender.FEMALE,
          "eck street", "San Mateo", "CA")));

      System.out
          .println("**** Again Searching for all Person's who's name start with Jul and are Female: ****");
      results = query.execute();
      System.out.println(" Size: " + results.size());

      waitForInput();

      System.out
          .println("**** Find the average age of all the entries ****");

      Query averageAgeQuery = bm.createQuery();
      averageAgeQuery.includeAggregator(Aggregators.average(age));
      System.out.println("Average age: "
                         + averageAgeQuery.execute().all().iterator().next()
          .getAggregatorResults());

      waitForInput();

      System.out
          .println("**** Find the average age of all people between 30 and 40 ****");
      Query agesBetween = bm.createQuery();
      agesBetween.addCriteria(age.between(30, 40));
      agesBetween.includeAggregator(Aggregators.average(age));
      System.out.println("Average age between 30 and 40: "
                         + agesBetween.execute().all().iterator().next()
          .getAggregatorResults());

      waitForInput();

      System.out.println("**** Find the count of people from NJ ****");

      Query newJerseyCountQuery = bm.createQuery().addCriteria(
          state.eq("NJ"));
      newJerseyCountQuery.includeAggregator(Aggregators.count());
      System.out.println("Count of people from NJ: "
                         + newJerseyCountQuery.execute().all().iterator().next()
          .getAggregatorResults());


    } finally {
      if (manager != null) manager.shutdown();
    }
  }

  private static void loadCache(Ehcache bm) {
    bm.put(new Element(1, new Person("Jane Doe", 35, Gender.FEMALE,
        "eck street", "San Mateo", "CA")));
    bm.put(new Element(2, new Person("Marie Antoinette", 23, Gender.FEMALE,
        "berry st", "Parsippany", "LA")));
    bm.put(new Element(3, new Person("John Smith", 25, Gender.MALE,
        "big wig", "Beverly Hills", "NJ")));
    bm.put(new Element(4, new Person("Paul Dupont", 45, Gender.MALE,
        "cool agent", "Madison", "WI")));
    bm.put(new Element(5, new Person("Juliet Capulet", 30, Gender.FEMALE,
        "dah man", "Bangladesh", "MN")));
    for (int i = 6; i < 1000; i++) {
      bm.put(new Element(i, new Person("Juliet Capulet" + i, 30,
          Person.Gender.MALE, "dah man", "Bangladesh", "NJ")));
    }
  }


  public static class NameAttributeExtractor implements AttributeExtractor {

    /**
     * Implementing the AttributeExtractor Interface and passing it in
     * allows you to create very efficient and specific attribute extraction
     * for performance sensitive code
     */

    public Object attributeFor(Element element) {
      return ((Person)element.getObjectValue()).getName();
    }

    @Override
    public Object attributeFor(Element element, String arg1)
        throws AttributeExtractorException {
      return attributeFor(element);
    }
  }
}
