package com.tenqube.visual_scraper.market

import android.content.Context
import android.view.View
import android.webkit.CookieManager
import com.google.gson.Gson
import com.jakewharton.retrofit2.adapter.kotlin.coroutines.CoroutineCallAdapterFactory
import com.tenqube.commerce.domain.vo.CommerceId
import com.tenqube.visual_scraper.MallViewHandler
import com.tenqube.visual_scraper.api.ApiService
import com.tenqube.visual_scraper.api.ApiServiceImpl
import com.tenqube.visual_scraper.market.dto.MarketResult
import com.tenqube.visual_scraper.market.dto.OrderItem
import com.tenqube.visual_scraper.market.dto.coupang.CoupangOrderDto
import com.tenqube.visual_scraper.market.dto.coupang.CoupangProduct
import com.tenqube.visual_scraper.market.dto.coupang.CoupangRule
import com.tenqube.visual_scraper.market.util.Utils
import com.tenqube.visual_scraper.market.dto.ScrapingCode
import com.tenqube.visual_scraper.shared.util.getValue
import com.tenqube.visual_scraper.webviewhtmlloader.OnPageLoadedCallback
import com.tenqube.visual_scraper.webviewhtmlloader.WebViewLoader
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import com.tenqube.visual_scraper.webviewhtmlloader.model.WebViewResponse
import timber.log.Timber
import java.text.SimpleDateFormat
import java.util.*
import kotlinx.coroutines.*
import okhttp3.OkHttpClient
import okhttp3.logging.HttpLoggingInterceptor
import org.jsoup.Jsoup
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import kotlin.collections.HashMap


class Coupang(
        private val webViewHtmlLoader: WebViewLoader, override val key: MarketKey, override val context: Context, override val mallViewHandler: MallViewHandler?) :
        Market {

    private var rule: CoupangRule

    private var scrapingResult = MarketResult(ScrapingCode.UnknownError)

    private var userId: String = ""
    private var userPwd: String = ""

    private val apiService: ApiServiceImpl


    init {
        apiService = createApiService()
        val marketRule = Utils.getJsonDataFromAsset(context, "coupang_rules.json")?:  ""
        rule = Gson().fromJson(marketRule, CoupangRule::class.java)
    }

    private fun createApiService(): ApiServiceImpl {
        return ApiServiceImpl(Retrofit.Builder()
                .client(OkHttpClient.Builder().apply {
                    addInterceptor(HttpLoggingInterceptor().setLevel(HttpLoggingInterceptor.Level.BODY))
                }.build())
                .baseUrl("https://commerce.tenqube.kr/")
                .addConverterFactory(GsonConverterFactory.create())
                .addCallAdapterFactory(CoroutineCallAdapterFactory())
                .build().create(ApiService::class.java))
    }

    /**
     * login 부터 스크리팽 까지 한큐에 진행하는 함수입니다.
     */
    override suspend fun load(
            id: String,
            password: String
    ): MarketResult = withContext(Dispatchers.Main) {

        Timber.i("load")
//
        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {
            scrapingResult = login(id, password).run {
                order()
            }.run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("load 응답값: $it")
        }
    }

    /**
     * 사용자 로그인을 진행합니다.
     */
    override suspend fun signIn(id: String, password: String): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {
            login(id, password)
        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("signIn 응답값: $it")
        }
    }

    /**
     * 스크래핑을 진행합니다.
     * 기본적으로 로그인후 세션이 유지된상태에서 진행합니다.
     */
    override suspend fun scrapOrder(): MarketResult = withContext(Dispatchers.Main) {
        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = order().run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d("load 응답값: $it")
        }
    }

    /**
     * 필요한 룰을 가지고 로그인을 진행합니다.
     */
    private suspend fun login(id: String, password: String) {
        // html load
        userId = id
        userPwd = password

        val loginResult = webViewHtmlLoader
                .noCache()
                .go(rule.loginUrl)
                .go(getLoginUrl(id, password), true, rule.loginSuccessUrl)
                .load()
                .getValue()

        val result = checkLoginHtml(loginResult)
        when(result.code == ScrapingCode.Success) {
            true -> {
                scrapingResult = MarketResult(ScrapingCode.Success)
            }
            false -> {
                scrapingResult = MarketResult(result.code, result.msg)
                throw Exception()
            }
        }
    }

    /**
     * 쿠팡 주문을 위해 url 로드 -> js 실행 -> 백키 실행 -> json 파일 얻기
     * 1. url 얻기
     * 2. url 에 인덱스 붙혀서 html 가져오기
     * 3. 각각 html 파싱하기
     */
    private suspend fun order(): String {

        val orderResult = webViewHtmlLoader
                .noCache()
                .go(rule.orderUrl)
                .go(rule.orderJsForAction)
                .delayTime(500)
                .back(true)
                .delayTime(500)
                .go("list")
                .load()
                .getValue()
        // 결과 응답
        return orderResult.url

    }

    private fun loadParsedData(request: CoupangOrderDto): List<CoupangProduct>  {
        return try {

            val orderList = request.pageProps.domains.desktopOrder.orderList

            val orderResults = mutableListOf<CoupangProduct>()

            orderList.forEach { order ->

                order.deliveryGroupList.forEach { deliveryGroup ->

                    deliveryGroup.productList.forEach { product ->
                        orderResults.add(
                            CoupangProduct(key = key,
                                title = order.title,
                                productId = product.productId,
                                productName = product.productName,
                                orderDate = order.orderedAt,
                                unitPrice = product.unitPrice,
                                orderNum = order.orderId.toString(),
                                isDoneOrderState = deliveryGroup.tracking?.done,
                                orderPrice = product.originalPaymentPrice,
                                imgUrl = product.imagePath,
                                quantity = product.quantity,
                                vendorItemId = product.vendorItemId
                        )
                        )
                    }
                }
            }

            Timber.d("WEBTEST parseResults $orderResults")

            Timber.i("WEBTEST breaparseResultsk%s", orderResults.size)

            orderResults
        } catch (e: Exception) {
            Timber.d("parseResults $e")

            emptyList()
        }
    }

    private fun filter(products: List<CoupangProduct>) : List<OrderItem>{
        return try {
            products.map { product ->
                OrderItem(
                        key = key,
                        productId = product.productId.toString(),
                        title = product.productName,
                        orderDate = product.orderDate.toDate(),
                        unitPrice = product.unitPrice.toDouble(),
                        orderNum = product.orderNum,
                        orderState = orderStateFilter(product.isDoneOrderState),
                        orderPrice = product.orderPrice.toDouble(),
                        imgUrl = product.imgUrl,
                        quantity = product.quantity,
                        option = optionFilter(product.title),
                        orderLink = makeOrderLink(product.orderNum),
                        productLink = makeProductLink(product.productId.toString(), product.vendorItemId.toString())

                )
            }.run {

                Timber.i("filter $this")
                this
            }

        } catch (e: Exception) {
            e.printStackTrace()

            emptyList()
        }
    }

    /**
     * 로그인 결과에 대해 실패를 체크합니다.
     */
    private fun checkLoginHtml(response: WebViewResponse): MarketResult {
        Timber.d("checkLoginHtml rule ${response.isSuccess}")

        Timber.d("checkLoginHtml url ${response.url}")

        Timber.d("checkLoginHtml url ${response.callbackUrl}")


        response.html?.let {
            val document = Jsoup.parse(response.html)

            val captcha = document.select(rule.captcha).attr("src")
            Timber.d("WEBTEST captcha ${captcha}")
            if(captcha != null && captcha.isNotEmpty()) {
                //캡챠가 뜬 경우 사용자에게 웹뷰 보여주고 메인 페이지로 이동되면 웹뷰를 끄고 스크래핑을 시작한다.

                // captcha
                onCaptchaResult()
                return MarketResult(ScrapingCode.CaptchaError, captcha)
            } else {
                Timber.d("WEBTEST captcha  false")
            }

            Timber.d("WEBTEST captcha2 ${rule.captchaSecond}")

            val captcha2 = document.select(".attempt-limitation-modal__body").text()
            Timber.d("WEBTEST captcha2 ${captcha2}")

            if(captcha2 != null && captcha2.isNotEmpty()) {
                //캡챠가 뜬 경우 사용자에게 웹뷰 보여주고 메인 페이지로 이동되면 웹뷰를 끄고 스크래핑을 시작한다.

                // captcha
                onCaptchaResult()
                return MarketResult(ScrapingCode.CaptchaError, captcha, webView = webViewHtmlLoader.getWebView())
            } else {
                Timber.d("WEBTEST captcha2  false")

            }


            val errorMsg = document.select(rule.loginFailMessage).text()
            Timber.d("WEBTEST errorMsg $errorMsg")

            if(errorMsg != null && errorMsg.isNotEmpty()) {
                return MarketResult(ScrapingCode.IdOrPwdError, errorMsg)
            }

            val idErrorMsg = document.select(rule.idError).text()
            Timber.d("WEBTEST idErrorMsg $idErrorMsg")

            if(idErrorMsg != null && idErrorMsg.isNotEmpty()) {
                return MarketResult(ScrapingCode.IdOrPwdError, idErrorMsg)
            }

            if(response.html.length < 50) {
                return MarketResult(ScrapingCode.IdOrPwdError, idErrorMsg)
            }
        }

        if(response.callbackUrl.contains("https://login.coupang.com/login/pincode/select")) {
            onCaptchaResult()
            return MarketResult(ScrapingCode.CaptchaError, "")
        }

        return if(response.isSuccess) {
            MarketResult(
                    code = ScrapingCode.Success,
                    msg = "",
                    results = emptyList()
            )
        } else {

            Timber.d("checkLoginHtml callbackUrl ${response.callbackUrl}")
            Timber.d("checkLoginHtml loginSuccessUrl ${rule.loginSuccessUrl}")


            MarketResult(
                    code = if (response.code != StatusCode.Success && response.callbackUrl.contains(rule.loginSuccessUrl))
                        ScrapingCode.Success
                    else
                        ScrapingCode.NetworkError,
                    msg = response.msg ?: "",
                    results = emptyList()
            )
        }
    }

    private fun onCaptchaResult() {

        mallViewHandler?.setWebView(CommerceId.Coupang.onlineId!!, webViewHtmlLoader.getWebView(), View.VISIBLE)
        webViewHtmlLoader.onPageCallback(rule.captchaCallbackJs,
                object : OnPageLoadedCallback {
                    override fun onPageLoaded(url: String) {

                        Timber.d("WEBTEST onPageLoaded $url")
                        Timber.d("WEBTEST rule.loginSuccessUrl $rule.loginSuccessUrl")

                        Timber.d("WEBTEST rule.userId $userId")
                        Timber.d("WEBTEST rule.userPwd $userPwd")


                        if (url.contains(rule.loginSuccessUrl)) {
//                            webViewHtmlLoader.getWebView()?.hide()
                            mallViewHandler?.setWebView(CommerceId.Coupang.onlineId!!, webViewHtmlLoader.getWebView(), View.GONE)

                            mallViewHandler?.onCaptchaLoginSuccess(CommerceId.Coupang.onlineId!!, userId, userPwd)

                            webViewHtmlLoader.onPageCallback() // 콘솔 제거해줍니다.
                        }
                    }

                    override fun onUserIdChanged(id: String) {
                        userId = id
                    }

                    override fun onUserPwdChanged(pwd: String) {
                        userPwd = pwd
                    }
                })
    }

    /**
     * json을 전달 받아 우리가 원하는 객체로 파싱합니다.
     * json ->(gson) CoupangResult -> (filter) List<OrderItem>
     */
    private suspend fun parsing(url: String?): List<CoupangProduct> = withContext(Dispatchers.IO) {
        return@withContext try {
            if(url == null) throw Exception()

            val urls = (0..2).map {
                "$url?orderType=ALL&pageIndex=$it"
            }
            Timber.i("WEBTEST urls%s", urls)


            val asyncItems = urls.map {
                async {

//                    webViewHtmlLoader.getWebView()?.loadUrl(it)
                    val cookieManager = CookieManager.getInstance()
                    val cookie = cookieManager.getCookie(it)
                    call(it, cookie)
                }
            }

            val results = asyncItems.awaitAll()


            results.mapNotNull {
                it?.let {
                    loadParsedData(it)
                }
            }.flatten()

        } catch (e: Exception) {
            e.printStackTrace()

            emptyList<CoupangProduct>()
        }
    }


    private suspend fun call(url: String, cookies: String?): CoupangOrderDto? = withContext(Dispatchers.IO) {

        if(cookies == null) {
            return@withContext null
        }
        val headers = HashMap<String, String>()
        headers["Cookie"] = cookies.trim()

        val result = apiService.get(url, headers)
        Timber.i("cookie result" + result)

        return@withContext result
    }

    /**
     * 결과값을 랩핑합니다.
      */
    private fun toResult(orderItems: List<OrderItem>): MarketResult {
        return MarketResult(
                ScrapingCode.Success, "success",
                results = orderItems).also {
            Timber.d("results ${it.results}")
        }
    }

    override suspend fun logOut() {

        Timber.i("LOGOUT")
        Timber.i("LOGOUT" + rule.loginSuccessUrl)
        Timber.i("LOGOUT" + rule.logout)


        val loginResult = webViewHtmlLoader
            .go(rule.loginSuccessUrl)
            .go(rule.logout, true)
            .load()
            .getValue()

    }

    private fun getLoginUrl(id: String, password: String): String = String.format(rule.loginJs, id, password)


    private fun Long.toDate(): String {
        return SimpleDateFormat("yyyy-MM-dd hh:mm:ss", Locale.KOREA).format(Date(this))
    }

    private fun orderStateFilter(isDoneOrderState: Boolean?): String {
        return isDoneOrderState?.let {
            if (it)
                "배송 완료"
            else
                "배송중"

        } ?: run{
            "취소 완료"
        }
    }

    private fun makeOrderLink(orderId: String): String {
        return String.format(rule.orderLink, orderId)
    }

    private fun makeProductLink(
            productId: String,
            vendorItemId: String
    ): String {
        return String.format(rule.productLink, productId, vendorItemId)
    }

    private fun optionFilter(title: String) : String {
        return title.substring(title.indexOf(",") + 1)
    }
}