package com.tenqube.visual_scraper.market

import android.content.Context
import android.view.View
import com.google.gson.Gson
import com.google.gson.JsonArray
import com.google.gson.JsonObject
import com.tenqube.commerce.domain.vo.CommerceId
import com.tenqube.visual_scraper.MallViewHandler
import com.tenqube.visual_scraper.market.dto.MarketKurlyRule
import com.tenqube.visual_scraper.market.dto.OrderItem
import com.tenqube.visual_scraper.market.dto.MarketResult
import com.tenqube.visual_scraper.shared.util.Result
import com.tenqube.visual_scraper.shared.util.getValue

import com.tenqube.visual_scraper.market.util.Utils
import com.tenqube.visual_scraper.market.util.toPrice
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.jsoup.Jsoup
import java.text.SimpleDateFormat
import java.util.*
import timber.log.Timber
import com.tenqube.visual_scraper.webviewhtmlloader.OnPageLoadedCallback
import com.tenqube.visual_scraper.webviewhtmlloader.WebViewLoader
import com.tenqube.visual_scraper.webviewhtmlloader.model.StatusCode
import com.tenqube.visual_scraper.webviewhtmlloader.model.WebViewResponse
import com.tenqube.visual_scraper.webviewhtmlloader.util.hide
import com.tenqube.visual_scraper.market.dto.ScrapingCode
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.launch


class MarketKurly(
        private val webViewHtmlLoader: WebViewLoader, override val key: MarketKey,
        override val context: Context, override val mallViewHandler: MallViewHandler?
) : Market {

    private var rule: MarketKurlyRule

    private var userId: String = ""
    private var userPwd: String = ""


    init {
        val marketRule = Utils.getJsonDataFromAsset(context, "market_rules.json")?:  ""
        rule = Gson().fromJson(marketRule, MarketKurlyRule::class.java)
    }
    // 서버에서 받아올 값


    private var scrapingResult = MarketResult(ScrapingCode.UnknownError)

    // 콜백 방식으로 데이터 리턴
    override suspend fun load(
            id: String,
            password: String
    ): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = login(id, password).run {
                order()
            }.run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            Timber.i(e)
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d( "load 응답값: $it")
        }
    }

    override suspend fun signIn(id: String, password: String): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {
            login(id, password)
        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d( "signIn 응답값: $it")
        }
    }

    override suspend fun scrapOrder(): MarketResult = withContext(Dispatchers.Main) {

        scrapingResult = MarketResult(ScrapingCode.UnknownError)

        try {

            scrapingResult = order().run {
                parsing(this)
            }.run {
                filter(this)
            }.run {
                toResult(this)
            }

        } catch (e: Exception) {
            e.toString()
        }
        return@withContext scrapingResult.also {
            Timber.d( "load 응답값: $it")
        }
    }

    override suspend fun logOut() {

        Timber.i("LOGOUT")
        Timber.i("LOGOUT" + rule.baseUrl)
        Timber.i("LOGOUT" + rule.logout)



        val loginResult = webViewHtmlLoader
                .go(rule.baseUrl)
                .go(rule.logout, true)
                .load()
                .getValue()
    }

    private suspend fun login(id: String, password: String) {

        userId = id
        userPwd = password

        // html load
        val loginResult =  webViewHtmlLoader
                .noCache()
                .go(rule.loginUrl)
                .go(getLoginUrl(id, password), true)
                .load()
                .getValue()

        val result = checkLoginHtml(loginResult)
        when(result.code == ScrapingCode.Success) {
            true ->  scrapingResult = MarketResult(ScrapingCode.Success)

            false -> {
                scrapingResult = MarketResult(result.code, result.msg)
                throw Exception()
            }
        }
    }


    @Throws(Exception::class)
    private suspend fun order(): String {
        val orderResult = webViewHtmlLoader
                .go(rule.orderUrl, true)
                .load()
                .getValue()
                .also {
                    Timber.d( "order ${it.isSuccess}")
                }

        when(orderResult.isSuccess) {
            true -> {
                when (orderResult.html == null) {
                    true -> {
                        Timber.d( "order: html is null")

                        scrapingResult = MarketResult(ScrapingCode.UnknownError, "html is empty")
                        throw Exception()
                    }
                    false -> {
                        Timber.d( "order: html exist")

                        return orderResult.html!!
                    }
                }
            }
            false -> {
                scrapingResult = MarketResult(ScrapingCode.NetworkError)
                throw Exception()
            }
        }
    }

    private fun parsing(html: String): JsonArray {
        return loadJsoupHtmlParser(html).getValue().also {
            Timber.d( "parsing: $it")
        }
    }

    private fun filter(jsonArray: JsonArray): List<OrderItem> {

        Timber.d( "filter: $jsonArray")

        return loadParsedDataFilter(jsonArray).also {
            Timber.d( "filter: $it")
        }
    }

    private fun toResult(orderItems: List<OrderItem>): MarketResult {
        return MarketResult(
                ScrapingCode.Success, "success",
                results = orderItems).also {
            Timber.d( "results ${it.results}")
        }
    }

    private fun checkLoginHtml(response: WebViewResponse): MarketResult {
        Timber.d( "checkLoginHtml rule ${response.isSuccess}")

//
        return when (response.isSuccess) {

            true -> {

                response.alertJS?.let {
                    Timber.d( "checkLoginHtml alertJS ${response.alertJS}")

                    Timber.d( "checkLoginHtml alertJS ${rule.loginErrorUrl}")
                    Timber.d( "checkLoginHtml alertJS ${it.url}")

                    Timber.d( "checkLoginHtml alertJS ${rule.loginErrorUrl == it.url}")

                    if (rule.loginErrorUrl == it.url) {
                        return MarketResult(ScrapingCode.IdOrPwdError, msg = it.msg ?: "")
                    }
                }

                MarketResult(
                        code = ScrapingCode.Success,
                        msg = "",
                        results = emptyList()
                )
            }
            false -> {

                response.html?.let {

                    Timber.d( "checkLoginHtml captcha rule ${rule.captcha}")
                    Timber.d( "checkLoginHtml captcha ${it.contains(rule.captcha)}")

                    if(it.contains(rule.captcha)) {
                        onCaptchaResult()
                        return MarketResult(ScrapingCode.CaptchaError, webView = webViewHtmlLoader.getWebView())
                    }
                }

                response.alertJS?.let {
                    if (rule.loginErrorUrl == it.url) {
                        return MarketResult(ScrapingCode.IdOrPwdError, msg = it.msg ?: "")
                    }
                }

                MarketResult(
                    code = if (response.code == StatusCode.TimeOut && response.callbackUrl.contains(rule.loginSuccessUrl))
                        ScrapingCode.Success
                    else
                        ScrapingCode.NetworkError,
                    msg = response.msg ?: "",
                    results = emptyList()
                )
            }
        }
    }

    private fun onCaptchaResult() {

        mallViewHandler?.setWebView(CommerceId.MarketKurly.onlineId!!, webViewHtmlLoader.getWebView(), View.VISIBLE)

        webViewHtmlLoader.onPageCallback(rule.captchaCallbackJs,
                object : OnPageLoadedCallback {
                    override fun onPageLoaded(url: String) {

                        Timber.d("WEBTEST onPageLoaded $url")
                        Timber.d("WEBTEST rule.loginSuccessUrl $rule.loginSuccessUrl")

                        Timber.d("WEBTEST rule.userId $userId")
                        Timber.d("WEBTEST rule.userPwd $userPwd")


                        if (url.contains(rule.baseUrl)) {

                            mallViewHandler?.setWebView(CommerceId.MarketKurly.onlineId!!, webViewHtmlLoader.getWebView(), View.GONE)

                            mallViewHandler?.retry(CommerceId.MarketKurly.onlineId!!, userId, userPwd)


                            webViewHtmlLoader.onPageCallback() // 콘솔 제거해줍니다.
                        }
                    }

                    override fun onUserIdChanged(id: String) {
                        userId = id
                    }

                    override fun onUserPwdChanged(pwd: String) {
                        userPwd = pwd
                    }
                })
    }

    private fun loadJsoupHtmlParser(html: String): Result<JsonArray> {

        return try {

            val body = Jsoup.parse(html).body()
            val jsonArray = JsonArray()
            val items = body.select(rule.container)

            items.forEach {

                val jsonObject = JsonObject()

                val orderDate = it.select(rule.orderDate).text()
                val orderNum = it.select(rule.orderNum).text()
                val orderPrice = it.select(rule.orderPrice).text()
                val orderState = it.select(rule.orderState).text()

                val productName = it.select(rule.productName).text()
                val productImgUrl = it.select(rule.productImgUrl).attr("src")

                jsonObject.addProperty("orderDate", orderDate)
                jsonObject.addProperty("orderNum", orderNum)
                jsonObject.addProperty("orderPrice", orderPrice)
                jsonObject.addProperty("orderState", orderState)
                jsonObject.addProperty("orderLink", String.format(rule.orderLink, orderNum))

                jsonObject.addProperty("productName", productName)
                jsonObject.addProperty("productImgUrl", productImgUrl)

                jsonArray.add((jsonObject))
            }


            Result.Success(jsonArray)

        } catch (e: Exception) {
            Result.Error(e)
        }
    }

    private fun loadParsedDataFilter(jsonArray: JsonArray): List<OrderItem>  {
        return jsonArray.mapNotNull {

            val jsonObject = it.asJsonObject

            val title = jsonObject["productName"].asString //[Sea to Table] 손질 생새우살 200g(냉동)
            val orderDate = jsonObject["orderDate"].asString.toDate() //2020.07.14 (14시 38분)
            val price = jsonObject["orderPrice"].asString.toPrice()// "7,055원"
            val orderNum = jsonObject["orderNum"].asString// "1594704964752"
            val orderState = jsonObject["orderState"].asString //취소 완료
            val orderLink = jsonObject["orderLink"].asString //https://img-cf.kurly.com/shop/data/goods/1536546970410s0.jpg

            val imgUrl = jsonObject["productImgUrl"].asString //https://img-cf.kurly.com/shop/data/goods/1536546970410s0.jpg

            Timber.d( "loadParsedDataFilter: $orderDate")

            if(orderDate != null) {
                OrderItem(key = key,
                        title = title,
                        orderDate = orderDate,
                        unitPrice = price,
                        orderNum = orderNum,
                        orderState = orderState,
                        imgUrl = imgUrl,
                        option = "",
                        quantity = 1,
                        productLink = "",
                        orderLink = orderLink,
                        orderPrice = price)
            } else {
                null
            }
        }
    }

    private fun String.toDate(): String? {
        //2020.07.17 (9시 36분)
        // 2020.07.14 (14시 38분) -> 2020-07-14 14:38:00
        val kurlyFormat =
                SimpleDateFormat(rule.dateFormat, Locale.KOREA)

        Timber.i(rule.dateFormat)
        val baseFormat =
                SimpleDateFormat("yyyy-MM-dd HH:mm:ss", Locale.KOREA)

        val date = kurlyFormat.parse(this)

        return date?.let {
            baseFormat.format(it)
        }
    }

    private fun getLoginUrl(id: String, password: String): String =
            rule.loginJs.replace("\$id", id).replace("\$password", password)

}
