package com.tenqube.visual_scraper;

import android.content.Context;
import android.os.Handler;
import androidx.annotation.Nullable;
import android.text.TextUtils;
import android.webkit.WebView;
import android.widget.Toast;

import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.mall.FactoryMall;
import com.tenqube.visual_scraper.mall.Mall;
import com.tenqube.visual_scraper.mall.data.MallData;
import com.tenqube.visual_scraper.manager.WebViewManager;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.repository.ScrapRepository;
import com.tenqube.visual_scraper.thirdparty.market.MarketScrapper;
import com.tenqube.visual_scraper.thirdparty.mart.MartData;
import com.tenqube.visual_scraper.thirdparty.mart.MartScrapper;
import com.tenqube.visual_scraper.utils.Utils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import timber.log.Timber;

public class ScrapManager {

    private static ScrapManager sInstance;

    private final Context context;

    public static final String HANDLER_MSG_ERROR = "error";
    public static final String HANDLER_MSG = "msg";
    public static final String HANDLER_MSG_MALL_ID = "mallId";

    // Sets the initial threadpool size to 8
    private static final int CORE_POOL_SIZE = 2;

    // Sets the maximum threadpool size to 8
    private static final int MAXIMUM_POOL_SIZE = 4;


    // Sets the amount of time an idle thread will wait for a task before terminating
    private static final int KEEP_ALIVE_TIME = 10;

    // Sets the Time Unit to seconds
    private static final TimeUnit KEEP_ALIVE_TIME_UNIT = TimeUnit.SECONDS;


    private BlockingQueue<Runnable> mParserWorkQueue;

    private ThreadPoolExecutor mParserThreadPool;

    private final ScrapRepository repository;

    private final Handler handler;

    private ScrapService.OnResultCallback<Integer> callback;
    private MartScrapper martScrapper;

    private MarketScrapper marketScrapper;


    public static ScrapManager getInstance(final Context context, final ScrapRepository repository, final Handler handler) {
        if (sInstance == null) {
            synchronized (ScrapManager.class) {
                if (sInstance == null) {
                    sInstance = new ScrapManager(context, repository, handler);
                }
            }
        }
        return sInstance;
    }

    public void setCallback(ScrapService.OnResultCallback<Integer> callback) {
        this.callback = callback;
    }

    private ScrapManager(Context context, ScrapRepository repository, Handler handler) {

        this.context = context;
        this.repository = repository;
        this.handler = handler;

        mParserWorkQueue = new LinkedBlockingQueue<>();
        mParserThreadPool = new ThreadPoolExecutor(CORE_POOL_SIZE, MAXIMUM_POOL_SIZE,
                KEEP_ALIVE_TIME, KEEP_ALIVE_TIME_UNIT, mParserWorkQueue);

    }

    private void showScrapingToast(String mallName) {
        Handler mainHandler = new Handler(context.getMainLooper());
        Runnable myRunnable = () -> Toast.makeText(context, mallName + " 쇼핑 내역을 불러오고 있습니다", Toast.LENGTH_SHORT).show();
        mainHandler.post(myRunnable);
    }


    private void showFinishToast(String mallName) {
        Handler mainHandler = new Handler(context.getMainLooper());
        Runnable myRunnable = () -> Toast.makeText(context, mallName + " 조회 완료", Toast.LENGTH_SHORT).show();
        mainHandler.post(myRunnable);
    }

    private void sendMsg(@Nullable Handler handler, Constants.ScrapState state, Constants.ERROR error, String msg, int mallId) {

        if(callback != null) {
            if(state == Constants.ScrapState.SUCCESS) {
                callback.onDataLoaded(mallId);
            } else {
                callback.onFail(mallId, error, msg);
            }
        }

        Thread.interrupted();
    }

    void startScrap(MallWithUser mallWithUser, MartScrapper martScrapper, boolean isLoginSkip) {
        mParserThreadPool.execute(() -> {
            try {

                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }

                martScrapper.init(msg -> {

                    Timber.i("init : %s", msg);

                    if(!TextUtils.isEmpty(msg)) {
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NONE,  msg, mallWithUser.mall.getId());
                        return;
                    }

                    if(mallWithUser.mall.getId() != Constants.MALL.Homeplus.getMallId()) {

                        if(isLoginSkip) {
                            startParsing(mallWithUser, martScrapper, isLoginSkip);
                        } else {
                            martScrapper.addUser(mallWithUser.mall.getId(),
                                    mallWithUser.user.getUserId(),
                                    mallWithUser.user.getUserPwd().getValue(), userMsg -> {

                                        Timber.i("addUser : %s", userMsg);


                                        if(!TextUtils.isEmpty(userMsg)) {
                                            sendMsg(handler,Constants.ScrapState.FAIL,
                                                    Constants.ERROR.LOGIN_FAIL,
                                                    userMsg,
                                                    mallWithUser.mall.getId());
                                            return;
                                        }

                                        mallWithUser.user.setLogin(true);
                                        repository.insertUsers(mallWithUser.user, uId -> {

                                            if(!isLoginSkip) showScrapingToast(mallWithUser.mall.getDisplayName());
                                            mallWithUser.user.setId(uId);
                                            startParsing(mallWithUser, martScrapper, isLoginSkip);

                                        });
                                    });
                        }


                    } else { // 홈플러스의 경우 사용자 등록이 별도로 필요없게 user 정보만 db에 저장합니다.

                        if(isLoginSkip) {
                            startParsing(mallWithUser, martScrapper, isLoginSkip);
                        } else {
                            mallWithUser.user.setLogin(true);
                            repository.insertUsers(mallWithUser.user, uId -> {
                                mallWithUser.user.setId(uId);
                                startParsing(mallWithUser, martScrapper, isLoginSkip);

                            });
                        }
                    }
                });
                    //sdk에서 받아서

            }catch (Exception e) {
                sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NONE, e.toString(), mallWithUser.mall.getId());
            }
        });
    }

    void startScrap(MallWithUser mallWithUser, MarketScrapper marketScrapper, boolean isLoginSkip) {
        mParserThreadPool.execute(() -> {
            try {

                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }

                marketScrapper.startWebViewLogin(mallWithUser.user, new ScrapService.OnResult<List<OrderEntity>>() {
                    @Override
                    public void onSuccess(int mId, List<OrderEntity> items, String msg) {

                        mallWithUser.user.setLogin(true);
                        repository.insertUsers(mallWithUser.user, uId -> {

                            mallWithUser.user.setId(uId);
                            repository.insertOrders(items, (insertedList -> {
                                if(!isLoginSkip) showScrapingToast(mallWithUser.mall.getDisplayName());

                                long lastScrapAt = Utils.getLastScrapAt();
                                repository.updateUserLastScrapAt(lastScrapAt, mallWithUser.mall.getId(), mallWithUser.user.getUserId());

                                repository.callInsertOrderList();
                                if(!isLoginSkip) showFinishToast(mallWithUser.mall.getDisplayName());
                                sendMsg(handler, Constants.ScrapState.SUCCESS, Constants.ERROR.NONE,  "success", mallWithUser.mall.getId());

                            }));
                        });


                    }

                    @Override
                    public void onFail(int mId, Constants.ERROR error, String msg) {
                        sendMsg(handler, Constants.ScrapState.FAIL, error,  msg, mId);
                    }

                    @Override
                    public void onCaptcha(int mId, WebView webView) {
//                        mallViewHandler.setWebView(mId, webView, View.VISIBLE);
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.CAPTCHA,  "captcha", mId);
                    }
                });

                //sdk에서 받아서

            }catch (Exception e) {
                sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NONE, e.toString(), mallWithUser.mall.getId());
            }
        });
    }

    private void startParsing(MallWithUser mallWithUser, MartScrapper martScrapper, boolean isLoginSkip) {
        martScrapper.start(mallWithUser.mall.getId(), scrapMsg -> {

            Timber.i("start Scrap : %s", scrapMsg);

            if(!TextUtils.isEmpty(scrapMsg)) {
                sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.LOGIN_FAIL,  scrapMsg, mallWithUser.mall.getId());
                return;
            }


            martScrapper.getMartItems(mallWithUser.mall.getId(), martItems -> {

                if(!isLoginSkip) showScrapingToast(mallWithUser.mall.getDisplayName());

                //martItems 변경하기
                List<OrderEntity> results = new ArrayList<>();
                for(MartData mart : martItems) {
                    results.add(mart.toEntity(mallWithUser.mall.getId(), mallWithUser.user.getId()));
                }

                repository.insertOrders(results, (insertedList -> {

                    long lastScrapAt = Utils.getLastScrapAt();
                    repository.updateUserLastScrapAt(lastScrapAt, mallWithUser.mall.getId(), mallWithUser.user.getUserId());

                    repository.callInsertOrderList();


                    if(!isLoginSkip) showFinishToast(mallWithUser.mall.getDisplayName());
                    sendMsg(handler, Constants.ScrapState.SUCCESS, Constants.ERROR.NONE,  "success", mallWithUser.mall.getId());

                }));
            });
        });
    }

    void startScrap(MallWithUser mallWithUser, WebViewManager webViewManager, boolean shouldInsert) {
        mParserThreadPool.execute(() -> {
            try {

                if (Thread.interrupted()) {
                    throw new InterruptedException();
                }

                Timber.i("login" + "MallWithUser : "+mallWithUser);
                // login rule 가져오기

                repository.getLoginRule(mallWithUser.mall.getId(), (loginRule) -> {
                    Timber.i("login" + "loginRule : "+loginRule);

                    Mall m = FactoryMall.create(new MallData(mallWithUser.mall.getId(), mallWithUser.mall.getName(), mallWithUser.mall.getDisplayName()), repository, webViewManager);
                    Timber.i("login" + "Mall : "+m);
                    if (m == null) {
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.EMPTY_MALL,  "empty factory mall", mallWithUser.mall.getId());
                        return;
                    }

                    webViewManager.setMall(m);

                    if (loginRule != null && loginRule.loginWebRule != null) {
                        // web login
                        Timber.i("login" +"로그인 룰정보 존재, startWebViewLogin");
                        m.startWebViewLogin(loginRule.loginWebRule, mallWithUser.user, shouldInsert, new ScrapService.OnResult<List<OrderEntity>>() {
                            @Override
                            public void onSuccess(int mId, List<OrderEntity> items, String msg) {

                                repository.callInsertOrderList();
                                sendMsg(handler, Constants.ScrapState.SUCCESS, Constants.ERROR.NONE,  "success", mId);
                            }

                            @Override
                            public void onFail(int mId, Constants.ERROR error, String msg) {
                                sendMsg(handler, Constants.ScrapState.FAIL, error,  msg, mId);
                            }

                            @Override
                            public void onCaptcha(int mId, WebView webView) {

                            }
                        });

                    } else {
                        // login rule 정보 없음
                        Timber.i("login" +"로그인 룰정보 없음 ");
                        sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NOT_FOUND_LOGIN_RULE, "로그인 룰정보 없음", mallWithUser.mall.getId());
                    }
                });

            } catch (Exception e) {
                sendMsg(handler, Constants.ScrapState.FAIL, Constants.ERROR.NOT_FOUND_LOGIN_RULE, e.toString(), mallWithUser.mall.getId());
            }

        });
    }

    public void logout(int mId) {
        try {

            WebViewManager webViewManager = new WebViewManager(context, null);

            Mall m = FactoryMall.create(new MallData(mId, "", ""), repository,  webViewManager);
            webViewManager.setMall(m);
            if(m != null)
                m.logout();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


}
