package com.tenqube.visual_scraper.repository;

import android.content.SharedPreferences;
import android.text.TextUtils;
import android.util.Log;

import com.tenqube.visual_scraper.ScrapService;
import com.tenqube.visual_scraper.api.ScrapApiService;
import com.tenqube.visual_scraper.constants.Constants;
import com.tenqube.visual_scraper.db.ScraperDatabase;
import com.tenqube.visual_scraper.db.dao.LoginApiRuleDao;
import com.tenqube.visual_scraper.db.dao.LoginRuleDao;
import com.tenqube.visual_scraper.db.dao.LoginWebRuleDao;
import com.tenqube.visual_scraper.db.dao.MallDao;
import com.tenqube.visual_scraper.db.dao.MallWithUserDao;
import com.tenqube.visual_scraper.db.dao.OrderApiRuleDao;
import com.tenqube.visual_scraper.db.dao.OrderDao;
import com.tenqube.visual_scraper.db.dao.OrderDetailWebRuleDao;
import com.tenqube.visual_scraper.db.dao.OrderWebRuleDao;
import com.tenqube.visual_scraper.db.dao.UserDao;
import com.tenqube.visual_scraper.db.dao.UserWithOrdersDao;
import com.tenqube.visual_scraper.model.query.LoginRule;
import com.tenqube.visual_scraper.model.query.MallWithUser;
import com.tenqube.visual_scraper.db.entity.MallEntity;
import com.tenqube.visual_scraper.db.entity.OrderApiRuleJson;
import com.tenqube.visual_scraper.db.entity.OrderEntity;
import com.tenqube.visual_scraper.db.entity.OrderWebRuleJson;
import com.tenqube.visual_scraper.db.entity.UserEntity;
import com.tenqube.visual_scraper.model.jsonParsing.MallScrapInfo;
import com.tenqube.visual_scraper.model.newApi.request.OrderBody;
import com.tenqube.visual_scraper.model.newApi.request.SaveOrdersBody;
import com.tenqube.visual_scraper.model.newApi.request.SignUpBody;
import com.tenqube.visual_scraper.model.newApi.response.ResultBody;
import com.tenqube.visual_scraper.model.newApi.response.ResultSignUp;
import com.tenqube.visual_scraper.model.view.MallInfo;
import com.tenqube.visual_scraper.model.view.OrderInfo;
import com.tenqube.visual_scraper.model.view.OrderWebRule;
import com.tenqube.visual_scraper.utils.AppExecutors;
import com.tenqube.visual_scraper.utils.DateUtils;
import com.tenqube.visual_scraper.utils.Utils;


import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;

public class ScrapRepository {

    private static ScrapRepository sInstance;

    private AppExecutors appExecutors;
    private final ScraperDatabase mDatabase;
    private final UserDao userDao;
    private final OrderDao orderDao;
    private final UserWithOrdersDao userWithOrdersDao;
    private final MallWithUserDao mallWithUserDao;
    private final MallDao mallDao;
    private final LoginRuleDao loginRuleDao;

    private final LoginApiRuleDao loginApiRuleDao;
    private final LoginWebRuleDao loginWebRuleDao;
    private final OrderApiRuleDao orderApiRuleDao;
    private final OrderWebRuleDao orderWebRuleDao;
    private final OrderDetailWebRuleDao orderDetailWebRuleDao;

    private final ScrapApiService mApi;
    private SharedPreferences preferences;

    private ScrapRepository(final ScraperDatabase database,
                            final ScrapApiService scrapApi,
                            final AppExecutors appExecutors,
                            final SharedPreferences preferences) {

        // local
        mDatabase = database;
        userDao = database.userDao();
        orderDao = database.orderDao();
        userWithOrdersDao = database.userWithOrdersDao();
        mallWithUserDao = database.mallWithUserDao();
        mallDao = database.mallDao();
        loginRuleDao = database.loginRuleDao();
        loginApiRuleDao = database.loginApiRuleDao();
        loginWebRuleDao = database.loginWebRuleDao();
        orderApiRuleDao = database.orderApiRuleDao();
        orderWebRuleDao = database.orderWebRuleDao();
        orderDetailWebRuleDao = database.orderDetailWebRuleDao();
        this.preferences = preferences;

        // remote
        mApi = scrapApi;
        this.appExecutors = appExecutors;
    }

    public static ScrapRepository getInstance(final ScraperDatabase database, final ScrapApiService scrapApi, final AppExecutors appExecutors, final SharedPreferences preferences) {
        if (sInstance == null) {
            synchronized (ScrapRepository.class) {
                if (sInstance == null) {
                    sInstance = new ScrapRepository(database, scrapApi, appExecutors, preferences);
                }
            }
        }
        return sInstance;
    }

    public void saveKey(String apiKey) {
        if (TextUtils.isEmpty(apiKey)) apiKey = "sVPnrNW0Fg4fujRSLJrYn8FJWyO5BtS21hAbOYXc"; //TEST ***
        preferences.edit().putString(Constants.PREF_KEY.X_API_KEY, apiKey).apply();
    }

    public void saveLayer(String layer) {
        preferences.edit().putString(Constants.PREF_KEY.LAYER, layer).apply();
    }

    public void signUpUser(String appName, String uuid, ScrapService.Callback<Void> callback) {
        appExecutors.networkIO().execute(() -> {

            Response<ResultBody<ResultSignUp>> response;
            try {
                response = mApi.callSignUpUser(new SignUpBody(appName, uuid)).execute();

                if (response.isSuccessful()) {
                    Log.i("signUpUser","success : "+response.body());
                    preferences.edit().putString(Constants.PREF_KEY.UUID, uuid).apply();
                    ResultBody<ResultSignUp> result = response.body();
                    if (result != null) {
                        preferences.edit().putString(Constants.PREF_KEY.TOKEN, result.result.authorization.mall).apply();
                        appExecutors.mainThread().execute(() -> callback.onDataLoaded(null));
                    }

                } else {
                    Log.i("signUpUser","error :"+response.message());
                }

            } catch (Exception e) {
                e.printStackTrace();
            }
        });
    }

    public void syncRule(ScrapService.Callback<Void> callback) {

        appExecutors.networkIO().execute(() -> {

            try {
                Response<ResultBody<MallScrapInfo>> response = mApi.callGetRules().execute();

                if (response.isSuccessful()) {

                    Log.i("syncRule","success !!: "+response.body());

                    // json 파싱 후 저장
                    ResultBody<MallScrapInfo> results = response.body();

                    if (results != null) {
                        if (preferences.getInt(Constants.PREF_KEY.RULE_VERSION, -1) == results.result.getVersion()){
                            callback.onDataLoaded(null);
                            return;
                        }

                        // Save new version data
                        preferences.edit().putInt(Constants.PREF_KEY.RULE_VERSION, results.result.getVersion()).apply();

                        mallDao.insertList(results.result.getMalls());
                        if (results.result.getLoginJsonRules() != null) loginApiRuleDao.insert(results.result.getLoginJsonRules());
                        loginWebRuleDao.insertList(results.result.getLoginHtmlRules());
                        orderApiRuleDao.insert(results.result.getOrderJsonRules());
                        orderWebRuleDao.insertList(results.result.getOrderHtmlRules());
                        orderDetailWebRuleDao.insertList(results.result.getOrderDetailHtmlRules());


                        Log.i("getJsonMallScraps", "getMalls : "+results.result.getMalls());
                        Log.i("getJsonMallScraps", "getLoginJsonRules : "+results.result.getLoginJsonRules());
                        Log.i("getJsonMallScraps", "getLoginHtmlRules : "+results.result.getLoginHtmlRules());
                        Log.i("getJsonMallScraps", "getOrderJsonRules : "+results.result.getOrderJsonRules());
                        Log.i("getJsonMallScraps", "getOrderHtmlRules : "+results.result.getOrderHtmlRules());
                        Log.i("getJsonMallScraps", "getOrderHtmlRules : "+results.result.getOrderDetailHtmlRules());

                    }


                } else {
                    Log.i("syncRule",""+response.message());
                }

                appExecutors.mainThread().execute(()-> callback.onDataLoaded(null));

            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }


    public void callInsertOrderList(List<OrderEntity> orders) {

        appExecutors.networkIO().execute(() -> {

            List<OrderBody> results = new ArrayList<>();
            for (OrderEntity o: orders) {
                OrderBody orderBody = new OrderBody(o.mId,
                        o.title,
                        DateUtils.transformDateStr(o.orderDate),
                        o.orderHms == null ? "" : DateUtils.transformHmsStr(o.orderHms),
                        o.orderOption,
                        o.quantity,
                        o.price,
                        o.currency,
                        o.orderNum,
                        o.orderState,
                        o.imgUrl);

                results.add(orderBody);
                Log.i("saveOrderBody","info : "+orderBody);
            }

            try {
                Log.i("result","SaveOrdersBody : "+new SaveOrdersBody(results));
                Response<String> response = mApi.callSaveOrders(new SaveOrdersBody(results)).execute();
                if (response.isSuccessful()) {
                    String result = response.body();
                    Log.i("callInsertOrderList","result : "+result);
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }

    public void getOrderList(ScrapService.Callback<List<OrderEntity>> callback, int... mIds) {
        appExecutors.diskIO().execute(() -> {
            List<OrderEntity> list = orderDao.getOrders(mIds);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(list));
        });
    }

    // send api
    public void getOrderList(ScrapService.Callback<List<OrderEntity>> callback, int mId) {
        appExecutors.diskIO().execute(() -> {
            List<OrderEntity> list = orderDao.getOrders(mId);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(list));
        });
    }

    // send view
    public void getAllOrderList(ScrapService.Callback<List<OrderInfo>> callback) {
        appExecutors.diskIO().execute(() -> {
            List<OrderEntity> list = orderDao.getOrders();

            appExecutors.mainThread().execute(() -> callback.onDataLoaded(Utils.convertOrderInfo(list)));
        });

    }

    public void updateUserLastScrapAt(long lastScrapAt, int mallId, String userId){
        try {

            appExecutors.diskIO().execute(() -> userDao.updateUserLastScrapAt(lastScrapAt, mallId, userId));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void updateUserForSignOut(int mId, ScrapService.OnResultCallback<Void> callback) {
        try {

            appExecutors.diskIO().execute(() -> {
                long i = userDao.updateUserIsLogin(mId, false);
                Log.i("updateUserForSignOut","i : "+i);
                appExecutors.mainThread().execute(new Runnable() {
                    @Override
                    public void run() {
                        if (i > 0 ) {
                            callback.onDataLoaded(null);
                        } else {
                            callback.onFail(mId, Constants.ERROR.LOGIN_OUT_FAIL, "logout fail");
                        }
                    }
                });

            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public void insertOrders(List<OrderEntity> orderEntityList) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.insertList(orderEntityList));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void insertUsers(UserEntity userEntity) {
        try {
            appExecutors.diskIO().execute(() -> {
                userDao.insert(userEntity);
                List<UserEntity> l = userDao.getUsers();
                Log.i("user", "info : "+ l);
                for (UserEntity u : l) {
                    Log.i("user", "info : "+ u);
                }
            });
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteOrder(int orderId) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrder(orderId));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteOrder(int... orderIds) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrders(orderIds));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteForMallId(int mId) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrdersForMallId(mId));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteForMallIds(int... mIds) {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteOrderForMallIds(mIds));

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public void deleteAll() {
        try {
            appExecutors.diskIO().execute(() -> orderDao.deleteAll());

        } catch (Exception e) {
            e.printStackTrace();
        }
    }

//    public void getUser(String userId, String pwd, Constants.MALL mall, ScrapService.Callback<UserEntity> callback) {
//
//        Runnable runnable = () -> {
//
//            UserEntity userEntity = null;
//            try {
//                userEntity = userDao.getUser(userId, pwd, mall.name());
//
//                if(userEntity == null) {
//                    userEntity = new UserEntity(0, mall.name(), userId, pwd, "");
//                    userEntity.setId((int) userDao.insertList(userEntity));
//                }
//            } catch (Exception e) {
//                e.printStackTrace();
//            } finally {
//                UserEntity finalUserEntity = userEntity;
//                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalUserEntity));
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//
//    }

//    public void getUser(Constants.MALL mall, ScrapService.Callback<UserEntity> callback) {
//
//        Runnable runnable = () -> {
//
//            UserEntity userEntity = null;
//            try {
//                userEntity = userDao.getUser(mall.name());
//            } catch (Exception e) {
//                e.printStackTrace();
//            } finally {
//                UserEntity finalUserEntity = userEntity;
//                appExecutors.mainThread().execute(() -> callback.onDataLoaded(finalUserEntity));
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//
//    }

//    public void deleteUser(Constants.MALL mall) {
//        Runnable runnable = () -> {
//            try {
//                userDao.deleteUser(mall.name());
//            } catch (Exception e) {
//                e.printStackTrace();
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//    }
//
//    public void deleteUser(UserEntity userEntity) {
//        Runnable runnable = () -> {
//            try {
//                userDao.deleteUser(userEntity.getId());
//            } catch (Exception e) {
//                e.printStackTrace();
//            }
//        };
//
//        appExecutors.diskIO().execute(runnable);
//    }

    public void checkIsActiveMall(int mId, ScrapService.Callback<MallEntity> callback) {
        Runnable runnable = () -> {
            MallEntity mallEntity = mallDao.getMall(mId);
            callback.onDataLoaded(mallEntity);
        };
        appExecutors.diskIO().execute(runnable);
    }

    public void getMallList(ScrapService.Callback<List<MallInfo>> callback) {
        appExecutors.diskIO().execute(() -> {
            try {
                List<MallEntity> list = mallDao.getMalls();
                List<MallInfo> results = new ArrayList<>();

                for (MallEntity mallEntity : list) {
                    results.add(new MallInfo(mallEntity.getId(),
                            mallEntity.getDisplayName(),
                            mallEntity.getIconUrl()));
                }
                appExecutors.mainThread().execute(() -> {
                    callback.onDataLoaded(results);
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });
    }

    public void getMallWithUser(int mId, String uId, ScrapService.onResultInterface<MallWithUser> callback) {
        appExecutors.diskIO().execute(() -> {
            try {
                MallWithUser mallWithUser = mallWithUserDao.getInfo(mId, uId);
                Log.i("getMallWithUser","info : "+mallWithUser);
                if (mallWithUser != null) {
                    callback.onDataLoaded(mallWithUser);
                } else {
                    callback.onFail();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }

    public void getMallWithUser(ScrapService.Callback<List<MallWithUser>> callback) {

        appExecutors.diskIO().execute(() -> {
            try {
                List<MallWithUser> list = mallWithUserDao.getInfoForActiveMall();
//                List<UserEntity> l = userDao.getUsers();
//                Log.i("user", "info : "+ l);
//                for (UserEntity u : l) {
//                    Log.i("user", "info : "+ u);
//                }
                appExecutors.mainThread().execute(() -> {
                    callback.onDataLoaded(list);
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }

    public void getMallWithActiveUser(ScrapService.Callback<List<MallWithUser>> callback) {

        appExecutors.diskIO().execute(() -> {
            try {
                List<MallWithUser> list = mallWithUserDao.getInfoForActiveUser();
                Log.i("getMallWithActiveUser", "info : "+ list);
                for (MallWithUser u : list) {
                    Log.i("getMallWithActiveUser", "info : "+ u);
                }
                appExecutors.mainThread().execute(() -> {
                    callback.onDataLoaded(list);
                });

            } catch (Exception e) {
                e.printStackTrace();
            }
        });

    }


    public void getLoginRule(int mId, ScrapService.Callback<LoginRule> callback) {
        appExecutors.diskIO().execute(() -> {
            LoginRule loginRule = loginRuleDao.getInfo(mId);
            Log.i("login","loginRule : "+loginRule);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(loginRule));
        });
    }



    public void getOrderWebRule(int mId, ScrapService.Callback<OrderWebRule> callback) {
        appExecutors.diskIO().execute(() -> {
            OrderWebRuleJson orderWebRuleJson = orderWebRuleDao.getInfo(mId);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(new OrderWebRule(orderWebRuleJson)));
        });
    }

    public void getOrderApiRule(int mId, ScrapService.Callback<OrderApiRuleJson> callback) {
        appExecutors.diskIO().execute(() -> {
            OrderApiRuleJson orderApiRuleJson = orderApiRuleDao.getInfo(mId);
            appExecutors.mainThread().execute(() -> callback.onDataLoaded(orderApiRuleJson));
        });
    }

}
