package com.tencent.wecast.utils;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.os.Looper;
import android.text.TextUtils;

import com.tencent.avlab.sdk.Xcast;
import com.tencent.avlab.sdk.XcastDefine;
import com.tencent.avlab.sdk.XcastError;
import com.tencent.avlab.sdk.XcastVariant;
import com.tencent.wecast.WeCastConstant;
import com.tencent.wecast.WeCastEnv;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.UUID;

public class WeCastUtil {
    private static final String TAG = "WeCastUtil";

    private static int mCapturePermissionCode;
    private static Intent mCapturePermissionData;

    public static void setCapturePermission(int code, Intent data) {
        mCapturePermissionCode = code;
        mCapturePermissionData = data;
    }

    public static void setCaptureData() {
        XcastVariant caps = Xcast.getProperty("device.screen-capture");
        if (caps.size() >= 1) {
            String property = String.format(XcastDefine.XcastProperty.SCREEN_CAPTURE_SETTING, caps.strAt(0));

            XcastVariant settings = Xcast.getProperty(property);
            if (settings == null) {
                settings = new XcastVariant();
            }
            settings.set("auth_result_code", mCapturePermissionCode);
            settings.set("auth_result_data", mCapturePermissionData);

            int ret = Xcast.setProperty(property, settings);
            Logger.kp().d("XCast set capture property result: " + ret);
            if (ret != XcastError.XCAST_OK) {
                Logger.kp().e("XCast set capture property error message: " + Xcast.errMsg());
            }
        } else {
            Logger.kp().e("XCast capture size(%d) error!", caps.size());
        }
    }

    public static boolean hasCapturePermission() {
        return mCapturePermissionCode != 0 && mCapturePermissionData != null;
    }

    public static boolean isMainThread() {
        return Looper.getMainLooper() == Looper.myLooper();
    }

    public static int getFileSize(String filePath) {
        if (TextUtils.isEmpty(filePath)) {
            return 0;
        }
        FileInputStream fis = null;
        int size = 0;
        try {
            fis = new FileInputStream(filePath);
            size = fis.available();
        } catch (IOException e) {
            e.printStackTrace();
            Logger.t(TAG).e("getFileSize filePath is not exist path = " + filePath);
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        Logger.t(TAG).d("getFileSize  filePath = %s | size = %d", filePath, size);
        return size;

    }


    public static String getPackageName(Context context) {
        return context.getApplicationContext().getPackageName();
    }

    /**
     * 获取进程号对应的进程名
     *
     * @param pid 进程号
     * @return 进程名
     */
    public static String getProcessName(int pid) {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(new FileReader("/proc/" + pid + "/cmdline"));
            String processName = reader.readLine();
            if (!TextUtils.isEmpty(processName)) {
                processName = processName.trim();
            }
            return processName;
        } catch (Throwable throwable) {
            throwable.printStackTrace();
        } finally {
            try {
                if (reader != null) {
                    reader.close();
                }
            } catch (IOException exception) {
                exception.printStackTrace();
            }
        }
        return null;
    }

    public static String getVersionAndEnv() {
        String suffix = "";
        if (WeCastEnv.getCurrentEnv() == WeCastEnv.ENV_PRE_RELEASE)
            suffix = "Pre-release Env.";
        if (WeCastEnv.getCurrentEnv() == WeCastEnv.ENV_DEVELOP)
            suffix = "Dev Env.";
        return getWmpSdkVersion() + " " + suffix;
    }

    // 获取宿主app的versionName（非SDK的版本号）
    public static String getAppVersionName(Context context) {
        try {
            PackageInfo pkg = context.getPackageManager()
                    .getPackageInfo(context.getApplicationContext().getPackageName(), 0);
            return pkg.versionName;
        } catch (Exception e) {
            Logger.t(TAG).e("getAppVersionName e = " + e);
            return "0.0.0.0";
        }
    }

    // 获取SDK的版本号
    public static String getWmpSdkVersion() {
        return WeCastConstant.WECAST_VERSION;
    }

    public static String genUUID() {
        return UUID.randomUUID().toString().replace("-", "");
    }

    // FIXME: 用登录后的设备名
    public static String getDeviceName() {
        return DeviceInfoUtil.getDeviceName();
    }

    public static void copyXcastConfFromAssets2SDCard(Context appContext) {
        if (appContext == null) {
            return;
        }
        FileOutputStream fos = null;
        BufferedInputStream is = null;
        try {
            String xcastConfDirPath = StorageUtil.getXCastConfigDirPath(appContext);
            String xcastConfFilePath = StorageUtil.getXCastConfFilePath(appContext);
            File xcastConfFile = new File(xcastConfFilePath);
            // 已经存在，不再处理
            if (xcastConfFile.exists() && xcastConfFile.isFile()) {
                return;
            }

            // 不存在，先清理文件夹
            deleteAllFile(xcastConfDirPath);
            File xcastConfDir = new File(xcastConfDirPath);
            xcastConfDir.mkdirs();

            // 从Assets拷贝文件
            is = new BufferedInputStream(appContext.getAssets().open(StorageUtil.getAssetsXCastName()));
            fos = new FileOutputStream(xcastConfFilePath);
            byte[] buffer = new byte[1024];
            int byteCount = 0;
            while ((byteCount = is.read(buffer)) != -1) {
                fos.write(buffer, 0, byteCount);
            }
            fos.flush();
        } catch (Throwable t) {
            Logger.t(TAG).e("copyXcastConfFromAssets2SDCard t = " + t);
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (Throwable t) {
                    Logger.t(TAG).e("copyXcastConfFromAssets2SDCard t = " + t);
                }
            }
            if (fos != null) {
                try {
                    fos.close();
                } catch (Throwable t) {
                    Logger.t(TAG).e("copyXcastConfFromAssets2SDCard t = " + t);
                }
            }
        }
    }

    private static void deleteAllFile(String path) {
        try {
            File dir = new File(path);
            if (!dir.exists()) {
                return;
            }
            if (!dir.isDirectory()) {
                return;
            }
            File[] fileList = dir.listFiles();

            if (fileList != null) {
                for (File file : fileList) {
                    file.delete();
                }
            }
        } catch (Throwable t) {
            Logger.t(TAG).e("deleteAllFile t = " + t);
        }
    }

    public static void killWeCast(Context context) {
        try {
            String weCastPackageName = context.getPackageName();
            Logger.t(TAG).d("killWeCast weCastPackageName = " + weCastPackageName);
            ShellUtils.execCmd("am force-stop " + weCastPackageName, false, true);
        } catch (Exception e) {
            Logger.t(TAG).e("killWeCast e = " + e);
        }
    }

    public static void copyTracesToLogDir(final Context appContext) {
        new Thread(new Runnable() {
            @Override
            public void run() {
                if (appContext == null) {
                    return;
                }
                FileOutputStream fos = null;
                BufferedInputStream is = null;
                try {
                    File tracesFile = new File("/data/anr/traces.txt");
                    if (!tracesFile.exists()) {
                        Logger.t(TAG).e("copyTracesToLogDir tracesFile not exists");
                        return;
                    }

                    SimpleDateFormat formatter = new SimpleDateFormat("yyyyMMdd");
                    String dateString = formatter.format(Calendar.getInstance().getTime());

                    File tracesLogFile = new File(StorageUtil.getWmpLogDirPath(appContext)
                            + File.separator
                            + "wmp_traces_" + dateString + "_" + System.currentTimeMillis() + ".txt");


                    is = new BufferedInputStream(new FileInputStream(tracesFile));
                    fos = new FileOutputStream(tracesLogFile);
                    byte[] buffer = new byte[1024];
                    int byteCount = 0;
                    while ((byteCount = is.read(buffer)) != -1) {
                        fos.write(buffer, 0, byteCount);
                    }
                    fos.flush();
                } catch (Throwable t) {
                    Logger.t(TAG).e("copyTracesToLogDir t = " + t);
                } finally {
                    if (is != null) {
                        try {
                            is.close();
                        } catch (Throwable t) {
                            Logger.t(TAG).e("copyTracesToLogDir t = " + t);
                        }
                    }
                    if (fos != null) {
                        try {
                            fos.close();
                        } catch (Throwable t) {
                            Logger.t(TAG).e("copyTracesToLogDir t = " + t);
                        }
                    }
                }
            }
        }).start();
    }
}
