package com.tencent.wecast.utils;

import android.annotation.SuppressLint;
import android.content.Context;
import android.net.ConnectivityManager;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.telephony.TelephonyManager;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.HttpURLConnection;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InterfaceAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

public class NetworkUtil {
    private static final String TAG = "NetworkUtil";

    public enum NetworkType {
        NETWORK_ETHERNET,
        NETWORK_WIFI,
        NETWORK_4G,
        NETWORK_3G,
        NETWORK_2G,
        NETWORK_UNKNOWN
    }

    /**
     * 获取活动网络信息
     * <p>需添加权限
     * {@code <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" />}</p>
     *
     * @return NetworkInfo
     */
    @SuppressLint("MissingPermission")
    private static NetworkInfo getActiveNetworkInfo(Context context) {
        ConnectivityManager manager =
                (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (manager == null) return null;
        return manager.getActiveNetworkInfo();
    }

    /**
     * 针对API>28
     *
     * @param context
     * @return
     */
    private static NetworkCapabilities getActiveNetworkInfo2(Context context) {
        ConnectivityManager manager =
                (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (manager == null) return null;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            return manager.getNetworkCapabilities(manager.getActiveNetwork());
        }
        return null;
    }


    private static final int NETWORK_TYPE_GSM = 16;
    private static final int NETWORK_TYPE_TD_SCDMA = 17;
    private static final int NETWORK_TYPE_IWLAN = 18;

    /**
     * 获取当前网络类型
     * <p>需添加权限
     * {@code <uses-permission android:name="android.permission.ACCESS_NETWORK_STATE" />}</p>
     *
     * @return 网络类型
     * <ul>
     * <li>{@link NetworkUtil.NetworkType#NETWORK_WIFI   } </li>
     * <li>{@link NetworkUtil.NetworkType#NETWORK_4G     } </li>
     * <li>{@link NetworkUtil.NetworkType#NETWORK_3G     } </li>
     * <li>{@link NetworkUtil.NetworkType#NETWORK_2G     } </li>
     * <li>{@link NetworkUtil.NetworkType#NETWORK_UNKNOWN} </li>
     * </ul>
     */
    public static NetworkType getNetworkType(Context context) {
        NetworkType netType = NetworkType.NETWORK_UNKNOWN;
        NetworkInfo info = getActiveNetworkInfo(context);
        if (info != null && info.isAvailable()) {

            if (info.getType() == ConnectivityManager.TYPE_WIFI) {
                netType = NetworkType.NETWORK_WIFI;
            } else if (info.getType() == ConnectivityManager.TYPE_ETHERNET) {
                netType = NetworkType.NETWORK_ETHERNET;
            } else if (info.getType() == ConnectivityManager.TYPE_MOBILE) {
                switch (info.getSubtype()) {

                    case NETWORK_TYPE_GSM:
                    case TelephonyManager.NETWORK_TYPE_GPRS:
                    case TelephonyManager.NETWORK_TYPE_CDMA:
                    case TelephonyManager.NETWORK_TYPE_EDGE:
                    case TelephonyManager.NETWORK_TYPE_1xRTT:
                    case TelephonyManager.NETWORK_TYPE_IDEN:
                        netType = NetworkType.NETWORK_2G;
                        break;

                    case NETWORK_TYPE_TD_SCDMA:
                    case TelephonyManager.NETWORK_TYPE_EVDO_A:
                    case TelephonyManager.NETWORK_TYPE_UMTS:
                    case TelephonyManager.NETWORK_TYPE_EVDO_0:
                    case TelephonyManager.NETWORK_TYPE_HSDPA:
                    case TelephonyManager.NETWORK_TYPE_HSUPA:
                    case TelephonyManager.NETWORK_TYPE_HSPA:
                    case TelephonyManager.NETWORK_TYPE_EVDO_B:
                    case TelephonyManager.NETWORK_TYPE_EHRPD:
                    case TelephonyManager.NETWORK_TYPE_HSPAP:
                        netType = NetworkType.NETWORK_3G;
                        break;

                    case NETWORK_TYPE_IWLAN:
                    case TelephonyManager.NETWORK_TYPE_LTE:
                        netType = NetworkType.NETWORK_4G;
                        break;
                    default:
                        String subtypeName = info.getSubtypeName();
                        if (subtypeName.equalsIgnoreCase("TD-SCDMA")
                                || subtypeName.equalsIgnoreCase("WCDMA")
                                || subtypeName.equalsIgnoreCase("CDMA2000")) {
                            netType = NetworkType.NETWORK_3G;
                        } else {
                            netType = NetworkType.NETWORK_UNKNOWN;
                        }
                        break;
                }
            } else {
                netType = NetworkType.NETWORK_UNKNOWN;
            }
        }
        return netType;
    }

    public static boolean isUsingWifi(Context context) {
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                NetworkCapabilities nc = getActiveNetworkInfo2(context);
                if (nc != null) {
                    return nc.hasTransport(NetworkCapabilities.TRANSPORT_WIFI);
                }
            } else {
                String netType = getNetType(context);
                return netType.equalsIgnoreCase("wifi");
            }
        } catch (Exception e) {
            Logger.t(TAG).e("Exception: " + e.toString());
        }
        return true;
    }

    public static boolean isUsingCellular(Context context) {
        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                NetworkCapabilities nc = getActiveNetworkInfo2(context);
                if (nc != null) {
                    return nc.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR);
                }
            } else {
                String netType = getNetType(context);
                return netType.equalsIgnoreCase("2G") || netType.equalsIgnoreCase("3G")
                    || netType.equalsIgnoreCase("4G");
            }
        } catch (Exception e) {
            Logger.t(TAG).e("Exception: " + e.toString());
        }
        return false;
    }

    public static boolean isConnTheSsidWifi(Context context, String ssid) {
        String connectedWifiSSID = getWifiSSID(context);
        return connectedWifiSSID.equalsIgnoreCase(ssid);
    }

    @SuppressLint("MissingPermission")
    public static String getWifiSSID(Context context) {
        try {
            if (Build.VERSION.SDK_INT <= Build.VERSION_CODES.O || Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
                WifiManager mWifiManager = (WifiManager) context.getSystemService(Context.WIFI_SERVICE);
                assert mWifiManager != null;
                WifiInfo info = mWifiManager.getConnectionInfo();
                return info.getSSID().replace("\"", "");
            } else if (Build.VERSION.SDK_INT == Build.VERSION_CODES.O_MR1) {
                ConnectivityManager connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                assert connManager != null;
                NetworkInfo networkInfo = connManager.getActiveNetworkInfo();
                if (networkInfo.isConnected()) {
                    if (networkInfo.getExtraInfo() != null) {
                        return networkInfo.getExtraInfo().replace("\"", "");
                    }
                }
            }
        } catch (Exception ignore) {
        }
        return "Unknown";
    }

    @SuppressLint("MissingPermission")
    public static String getWifiMacAddress(Context context) {
        String add = "";
        if (context == null) {
            return add;
        }
        try {
            WifiManager wifi = (WifiManager) context.getApplicationContext()
                    .getSystemService(Context.WIFI_SERVICE);
            WifiInfo info = wifi.getConnectionInfo();
            add = info.getBSSID();
        } catch (Throwable ignore) {
        }
        return add;
    }

    public static String getNetType(Context context) {
        NetworkType networkType = getNetworkType(context);
        switch (networkType) {
            case NETWORK_2G:
                return "2G";
            case NETWORK_3G:
                return "3G";
            case NETWORK_4G:
                return "4G";
            case NETWORK_WIFI:
                return "wifi";
            case NETWORK_ETHERNET:
                return "wired";
            default:
                return "unknown";
        }
    }

    public static String getIp(Context context) {
        try {
            Enumeration<NetworkInterface> networkInterfaces = NetworkInterface.getNetworkInterfaces();
            if (null != networkInterfaces) {
                for (; networkInterfaces.hasMoreElements(); ) {
                    NetworkInterface networkInterface = networkInterfaces.nextElement();
                    for (Enumeration<InetAddress> enumIpAddress = networkInterface.getInetAddresses(); enumIpAddress.hasMoreElements(); ) {
                        InetAddress inetAddress = enumIpAddress.nextElement();
                        if (!inetAddress.isLoopbackAddress() && (inetAddress instanceof Inet4Address)) {
                            return inetAddress.getHostAddress();
                        }
                    }
                }
            } else {
                Logger.t(TAG).e("the networkInterfaces is null");
            }
        } catch (Exception e) {
            Logger.t(TAG).e("getIp e =" + e);
        }
        return "";
    }


    // 获取所有网卡的信息列表
    public static List<NetworkCardInfo> getNetCardInfoList() {
        List<NetworkCardInfo> networkCardInfoList = new ArrayList<>();
        try {
            // 获取本地设备的所有网络接口
            Enumeration<NetworkInterface> allNetInterfaces = NetworkInterface.getNetworkInterfaces();
            while (allNetInterfaces.hasMoreElements()) {
                NetworkInterface networkCard = allNetInterfaces.nextElement();
                if (!networkCard.isUp()) {
                    // 网卡未使用
                    continue;
                }

                String displayName = networkCard.getDisplayName();
                // 仅有线和wifi
                if (displayName.equals("wlan0") || displayName.equals("eth0")) {
                    List<InterfaceAddress> addressList = networkCard.getInterfaceAddresses();
                    Iterator<InterfaceAddress> addressIterator = addressList.iterator();
                    while (addressIterator.hasNext()) {
                        InterfaceAddress interfaceAddress = addressIterator.next();
                        InetAddress address = interfaceAddress.getAddress();
                        if (!address.isLoopbackAddress()) {
                            String hostAddress = address.getHostAddress();
                            if (!(hostAddress.indexOf(":") > 0)) {
                                String maskAddress = calcMaskByPrefixLength(interfaceAddress.getNetworkPrefixLength());
                                String gateway = calcSubnetAddress(hostAddress, maskAddress);

                                /*
                                String broadcastAddress = null;
                                InetAddress broadcast = interfaceAddress.getBroadcast();
                                if (broadcast != null) {
                                    broadcastAddress = broadcast.getHostAddress();
                                }
                                */
                                NetworkCardInfo cardInfo = new NetworkCardInfo();
                                cardInfo.gate = gateway;
                                cardInfo.ip = hostAddress;
                                cardInfo.mask = maskAddress;
                                networkCardInfoList.add(cardInfo);
                            }
                        }
                    }
                }
            }

        } catch (SocketException e) {
            e.printStackTrace();
        }
        return networkCardInfoList;
    }

    public static NetworkInterface getActiveNetworkInterface() {
        Enumeration<NetworkInterface> interfaces;
        try {
            interfaces = NetworkInterface.getNetworkInterfaces();
        } catch (SocketException e) {
            return null;
        }

        while (interfaces.hasMoreElements()) {
            NetworkInterface iface = interfaces.nextElement();
            // 非wifi或有线网直接跳过
            if (!iface.getName().contains("lan") && !iface.getName().contains("eth")) {
                continue;
            }

            Enumeration<InetAddress> inetAddresses = iface.getInetAddresses();

            /* Check if we have a non-local address. If so, this is the active
             * interface.
             *
             * This isn't a perfect heuristic: I have devices which this will
             * still detect the wrong interface on, but it will handle the
             * common cases of wifi-only and Ethernet-only.
             */
            while (inetAddresses.hasMoreElements()) {
                InetAddress addr = inetAddresses.nextElement();

                if (!(addr.isLoopbackAddress() || addr.isLinkLocalAddress())) {
                    return iface;
                }
            }
        }

        return null;
    }

    public static InetAddress getLocalV4Address(NetworkInterface netif) {
        Enumeration addrs = netif.getInetAddresses();
        while (addrs.hasMoreElements()) {
            InetAddress addr = (InetAddress) addrs.nextElement();
            if (addr instanceof Inet4Address && !addr.isLoopbackAddress())
                return addr;
        }
        return null;
    }

    public static InetAddress getLocalV6Address(NetworkInterface netif) {
        Enumeration addrs = netif.getInetAddresses();
        while (addrs.hasMoreElements()) {
            InetAddress addr = (InetAddress) addrs.nextElement();
            if (addr instanceof Inet6Address && !addr.isLoopbackAddress())
                return addr;
        }
        return null;
    }

    public static String getLocalV4Address() {
        NetworkInterface netInterface = NetworkUtil.getActiveNetworkInterface();
        if (null != netInterface) {
            InetAddress localV4Address = NetworkUtil.getLocalV4Address(netInterface);
            if (localV4Address != null) {
                return localV4Address.getHostAddress();
            }
        }
        return "";
    }

    public static String getLocalV6Address() {
        NetworkInterface netInterface = NetworkUtil.getActiveNetworkInterface();
        if (null != netInterface) {
            InetAddress localV6Address = NetworkUtil.getLocalV6Address(netInterface);
            if (localV6Address != null) {
                return localV6Address.getHostAddress();
            }
        }
        return "";
    }

    public static class NetworkCardInfo {
        public String ip;
        public String mask;
        public String gate;

        @Override
        public String toString() {
            return "{ip:'" + ip + "' mask:'" + mask + "' gate:'" + gate + "'}";
        }
    }

    private static final String defaultMac = "02:00:00:00:00:00";

    @SuppressLint("HardwareIds")
    public static String getMAC(Context context) {
        WifiManager wifiMan = (WifiManager) context.getApplicationContext()
                .getSystemService(Context.WIFI_SERVICE);
        if (wifiMan != null) {
            WifiInfo wifiInf = wifiMan.getConnectionInfo();

            if (wifiInf != null && defaultMac.equals(wifiInf.getMacAddress())) {
                String result = null;
                try {
                    result = getMacByInterface();
                    if (result != null) {
                        return result;
                    } else {
                        result = getMacByFile(wifiMan);
                        return result;
                    }
                } catch (Exception ignore) {
                }
            } else {
                if (wifiInf != null && wifiInf.getMacAddress() != null) {
                    return wifiInf.getMacAddress();
                } else {
                    return "";
                }
            }
        }
        return defaultMac;
    }


    private static String calcMaskByPrefixLength(int length) {
        int mask = 0xffffffff << (32 - length);
        int partsNum = 4;
        int bitsOfPart = 8;
        int maskParts[] = new int[partsNum];
        int selector = 0x000000ff;

        for (int i = 0; i < maskParts.length; i++) {
            int pos = maskParts.length - 1 - i;
            maskParts[pos] = (mask >> (i * bitsOfPart)) & selector;
        }

        String result = "";
        result = result + maskParts[0];
        for (int i = 1; i < maskParts.length; i++) {
            result = result + "." + maskParts[i];
        }
        return result;
    }

    private static String calcSubnetAddress(String ip, String mask) {
        String result = "";
        try {
            // calc sub-net IP
            InetAddress ipAddress = InetAddress.getByName(ip);
            InetAddress maskAddress = InetAddress.getByName(mask);

            byte[] ipRaw = ipAddress.getAddress();
            byte[] maskRaw = maskAddress.getAddress();

            int unsignedByteFilter = 0x000000ff;
            int[] resultRaw = new int[ipRaw.length];
            for (int i = 0; i < resultRaw.length; i++) {
                resultRaw[i] = (ipRaw[i] & maskRaw[i] & unsignedByteFilter);
            }

            // make result string
            result = result + resultRaw[0];
            for (int i = 1; i < resultRaw.length; i++) {
                result = result + "." + resultRaw[i];
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return result;
    }

    private static String getMacByInterface() {
        try {
            List<NetworkInterface> all = Collections.list(NetworkInterface.getNetworkInterfaces());
            for (NetworkInterface nif : all) {
                if (nif.getName().equalsIgnoreCase("wlan0")) {
                    byte[] macBytes = nif.getHardwareAddress();
                    if (macBytes == null) {
                        return "";
                    }

                    StringBuilder res1 = new StringBuilder();
                    for (byte b : macBytes) {
                        res1.append(String.format("%02X:", b));
                    }

                    if (res1.length() > 0) {
                        res1.deleteCharAt(res1.length() - 1);
                    }
                    return res1.toString();
                }
            }

        } catch (Exception ignore) {
        }
        return null;
    }

    @SuppressLint("MissingPermission")
    private static String getMacByFile(WifiManager wifiMan) throws Exception {
        String ret;
        int wifiState = wifiMan.getWifiState();
        wifiMan.setWifiEnabled(true);
        File fl = new File("/sys/class/net/wlan0/address");
        FileInputStream fin = new FileInputStream(fl);
        ret = getStringFromStream(fin);
        fin.close();
        boolean enabled = WifiManager.WIFI_STATE_ENABLED == wifiState;
        wifiMan.setWifiEnabled(enabled);
        return ret;
    }

    private static String getStringFromStream(InputStream inputStream) throws IOException {
        if (inputStream != null) {
            Writer writer = new StringWriter();
            char[] buffer = new char[2048];
            try {
                Reader reader = new BufferedReader(new InputStreamReader(inputStream, "UTF-8"));
                int counter;
                while ((counter = reader.read(buffer)) != -1) {
                    writer.write(buffer, 0, counter);
                }
            } finally {
                inputStream.close();
            }
            return writer.toString();
        } else {
            return "No Contents";
        }
    }

    /**
     * 判断是否有网络连接
     */
    public static boolean isNetworkConnected(Context context) {
        if (context != null) {
            // 获取手机所有连接管理对象(包括对wi-fi,net等连接的管理)
            ConnectivityManager manager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (manager != null) {
                NetworkInfo networkInfo = manager.getActiveNetworkInfo();
                if (networkInfo != null) {
                    return networkInfo.isAvailable();
                }
            }
        }
        return false;
    }

    public static boolean checkHttpGetRet(String url) {
        HttpURLConnection httpCoon = null;
        try {
            URL netUrl = new URL(url);
            httpCoon = (HttpURLConnection) netUrl.openConnection();
            httpCoon.setRequestMethod("GET");
            httpCoon.setConnectTimeout(2000);
            httpCoon.connect();
            int responseCode = httpCoon.getResponseCode();
            return responseCode == HttpURLConnection.HTTP_OK;
        } catch (Exception ignore) {
        } finally {
            try {
                if (httpCoon != null) httpCoon.disconnect();
            } catch (Exception ignore) {
            }
        }
        return false;
    }
}
