package com.tencent.wecast.sender.cloud.utils

import android.app.Activity
import android.app.Dialog
import android.content.Context
import android.content.res.Resources
import android.graphics.Color
import android.os.Build
import android.support.annotation.ColorInt
import android.support.annotation.IntRange
import android.support.annotation.NonNull
import android.support.v4.widget.DrawerLayout
import android.view.View
import android.view.ViewGroup
import android.view.ViewGroup.MarginLayoutParams
import android.view.Window
import android.view.WindowManager
import android.widget.LinearLayout


class StatusBarUtils private constructor() {

    init {
        throw UnsupportedOperationException("u can't instantiate me...")
    }

    companion object {

        ///////////////////////////////////////////////////////////////////////////
        // status bar
        ///////////////////////////////////////////////////////////////////////////

        private val DEFAULT_ALPHA = 112
        private val TAG_COLOR = "TAG_COLOR"
        private val TAG_ALPHA = "TAG_ALPHA"
        private val TAG_OFFSET = "TAG_OFFSET"
        private val KEY_OFFSET = -123

        /**
         * Return the status bar's height.
         *
         * @return the status bar's height
         */
        fun getStatusBarHeight(): Int {
            val resources = Resources.getSystem()
            val resourceId = resources.getIdentifier("status_bar_height", "dimen", "android")
            return resources.getDimensionPixelSize(resourceId)
        }

        /**
         * Set the status bar's visibility.
         *
         * @param activity  The activity.
         * @param isVisible True to set status bar visible, false otherwise.
         */
        fun setStatusBarVisibility(@NonNull activity: Activity,
                                   isVisible: Boolean) {
            setStatusBarVisibility(activity.window, isVisible)
        }

        /**
         * Set the status bar's visibility.
         *
         * @param window    The window.
         * @param isVisible True to set status bar visible, false otherwise.
         */
        fun setStatusBarVisibility(@NonNull window: Window,
                                   isVisible: Boolean) {
            if (isVisible) {
                window.clearFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN)
                showColorView(window)
                showAlphaView(window)
                addMarginTopEqualStatusBarHeight(window)
            } else {
                window.addFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN)
                hideColorView(window)
                hideAlphaView(window)
                subtractMarginTopEqualStatusBarHeight(window)
            }
        }

        /**
         * Return whether the status bar is visible.
         *
         * @param activity The activity.
         * @return `true`: yes<br></br>`false`: no
         */
        fun isStatusBarVisible(@NonNull activity: Activity): Boolean {
            val flags = activity.window.attributes.flags
            return flags and WindowManager.LayoutParams.FLAG_FULLSCREEN == 0
        }

        /**
         * Set the status bar's light mode.
         *
         * @param activity    The activity.
         * @param isLightMode True to set status bar light mode, false otherwise.
         */
        fun setStatusBarLightMode(@NonNull activity: Activity,
                                  isLightMode: Boolean) {
            setStatusBarLightMode(activity.window, isLightMode)
        }

        /**
         * Set the status bar's light mode.
         *
         * @param window      The window.
         * @param isLightMode True to set status bar light mode, false otherwise.
         */
        fun setStatusBarLightMode(@NonNull window: Window,
                                  isLightMode: Boolean) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
                val decorView: View? = window.decorView
                if (decorView != null) {
                    var vis = decorView.getSystemUiVisibility()
                    if (isLightMode) {
                        window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                        vis = vis or View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR
                    } else {
                        vis = vis and View.SYSTEM_UI_FLAG_LIGHT_STATUS_BAR.inv()
                    }
                    decorView.setSystemUiVisibility(vis)
                }
            }
        }

        /**
         * Add the top margin size equals status bar's height for view.
         *
         * @param view The view.
         */
        fun addMarginTopEqualStatusBarHeight(@NonNull view: View) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            view.tag = TAG_OFFSET
            val haveSetOffset = view.getTag(KEY_OFFSET)
            if (haveSetOffset != null && haveSetOffset as Boolean) return
            val layoutParams = view.layoutParams as MarginLayoutParams
            layoutParams.setMargins(layoutParams.leftMargin,
                    layoutParams.topMargin + getStatusBarHeight(),
                    layoutParams.rightMargin,
                    layoutParams.bottomMargin)
            view.setTag(KEY_OFFSET, true)
        }

        /**
         * Subtract the top margin size equals status bar's height for view.
         *
         * @param view The view.
         */
        fun subtractMarginTopEqualStatusBarHeight(@NonNull view: View) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            val haveSetOffset = view.getTag(KEY_OFFSET)
            if (haveSetOffset == null || !(haveSetOffset as Boolean)) return
            val layoutParams = view.layoutParams as MarginLayoutParams
            layoutParams.setMargins(layoutParams.leftMargin,
                    layoutParams.topMargin - getStatusBarHeight(),
                    layoutParams.rightMargin,
                    layoutParams.bottomMargin)
            view.setTag(KEY_OFFSET, false)
        }

        private fun addMarginTopEqualStatusBarHeight(window: Window) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            val withTag: View = window.decorView.findViewWithTag(TAG_OFFSET) ?: return
            addMarginTopEqualStatusBarHeight(withTag)
        }

        private fun subtractMarginTopEqualStatusBarHeight(window: Window) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            val withTag: View = window.decorView.findViewWithTag(TAG_OFFSET) ?: return
            subtractMarginTopEqualStatusBarHeight(withTag)
        }

        /**
         * Set the status bar's color.
         *
         * @param activity The activity.
         * @param color    The status bar's color.
         */
        fun setStatusBarColor(@NonNull activity: Activity,
                              @ColorInt color: Int) {
            setStatusBarColor(activity, color, DEFAULT_ALPHA, false)
        }

        /**
         * Set the status bar's color.
         *
         * @param activity The activity.
         * @param color    The status bar's color.
         * @param alpha    The status bar's alpha which isn't the same as alpha in the color.
         */
        fun setStatusBarColor(@NonNull activity: Activity,
                              @ColorInt color: Int,
                              @IntRange(from = 0, to = 255) alpha: Int) {
            setStatusBarColor(activity, color, alpha, false)
        }

        /**
         * Set the status bar's color.
         *
         * @param activity The activity.
         * @param color    The status bar's color.
         * @param alpha    The status bar's alpha which isn't the same as alpha in the color.
         * @param isDecor  True to add fake status bar in DecorView,
         * false to add fake status bar in ContentView.
         */
        fun setStatusBarColor(@NonNull activity: Activity,
                              @ColorInt color: Int,
                              @IntRange(from = 0, to = 255) alpha: Int,
                              isDecor: Boolean) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            hideAlphaView(activity)
            transparentStatusBar(activity)
            addStatusBarColor(activity, color, alpha, isDecor)
        }

        /**
         * Set the status bar's color.
         *
         * @param fakeStatusBar The fake status bar view.
         * @param color         The status bar's color.
         */
        fun setStatusBarColor(@NonNull fakeStatusBar: View,
                              @ColorInt color: Int) {
            setStatusBarColor(fakeStatusBar, color, DEFAULT_ALPHA)
        }

        /**
         * Set the status bar's color.
         *
         * @param fakeStatusBar The fake status bar view.
         * @param color         The status bar's color.
         * @param alpha         The status bar's alpha which isn't the same as alpha in the color.
         */
        fun setStatusBarColor(@NonNull fakeStatusBar: View,
                              @ColorInt color: Int,
                              @IntRange(from = 0, to = 255) alpha: Int) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            fakeStatusBar.visibility = View.VISIBLE
            transparentStatusBar(fakeStatusBar.context as Activity)
            val layoutParams = fakeStatusBar.layoutParams
            layoutParams.width = ViewGroup.LayoutParams.MATCH_PARENT
            layoutParams.height = getStatusBarHeight()
            fakeStatusBar.setBackgroundColor(getStatusBarColor(color, alpha))
        }

        /**
         * Set the status bar's alpha.
         *
         * @param activity The activity.
         */
        fun setStatusBarAlpha(@NonNull activity: Activity) {
            setStatusBarAlpha(activity, DEFAULT_ALPHA, false)
        }

        /**
         * Set the status bar's alpha.
         *
         * @param activity The activity.
         * @param alpha    The status bar's alpha.
         */
        fun setStatusBarAlpha(@NonNull activity: Activity,
                              @IntRange(from = 0, to = 255) alpha: Int) {
            setStatusBarAlpha(activity, alpha, false)
        }

        /**
         * Set the status bar's alpha.
         *
         * @param activity The activity.
         * @param alpha    The status bar's alpha.
         * @param isDecor  True to add fake status bar in DecorView,
         * false to add fake status bar in ContentView.
         */
        fun setStatusBarAlpha(@NonNull activity: Activity,
                              @IntRange(from = 0, to = 255) alpha: Int,
                              isDecor: Boolean) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            hideColorView(activity)
            transparentStatusBar(activity)
            addStatusBarAlpha(activity, alpha, isDecor)
        }

        /**
         * Set the status bar's alpha.
         *
         * @param fakeStatusBar The fake status bar view.
         */
        fun setStatusBarAlpha(@NonNull fakeStatusBar: View) {
            setStatusBarAlpha(fakeStatusBar, DEFAULT_ALPHA)
        }

        /**
         * Set the status bar's alpha.
         *
         * @param fakeStatusBar The fake status bar view.
         * @param alpha         The status bar's alpha.
         */
        fun setStatusBarAlpha(@NonNull fakeStatusBar: View,
                              @IntRange(from = 0, to = 255) alpha: Int) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            fakeStatusBar.visibility = View.VISIBLE
            transparentStatusBar(fakeStatusBar.context as Activity)
            val layoutParams = fakeStatusBar.layoutParams
            layoutParams.width = ViewGroup.LayoutParams.MATCH_PARENT
            layoutParams.height = getStatusBarHeight()
            fakeStatusBar.setBackgroundColor(Color.argb(alpha, 0, 0, 0))
        }

        /**
         * Set the custom status bar.
         *
         * @param fakeStatusBar The fake status bar view.
         */
        fun setStatusBarCustom(@NonNull fakeStatusBar: View) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            fakeStatusBar.visibility = View.VISIBLE
            transparentStatusBar(fakeStatusBar.context as Activity)
            val layoutParams = fakeStatusBar.layoutParams
            layoutParams.width = ViewGroup.LayoutParams.MATCH_PARENT
            layoutParams.height = getStatusBarHeight()
        }

        /**
         * Set the status bar's color for DrawerLayout.
         *
         * DrawLayout must add `android:fitsSystemWindows="true"`
         *
         * @param activity      The activity.
         * @param drawer        The DrawLayout.
         * @param fakeStatusBar The fake status bar view.
         * @param color         The status bar's color.
         * @param isTop         True to set DrawerLayout at the top layer, false otherwise.
         */
        fun setStatusBarColor4Drawer(@NonNull activity: Activity,
                                     @NonNull drawer: DrawerLayout,
                                     @NonNull fakeStatusBar: View,
                                     @ColorInt color: Int,
                                     isTop: Boolean) {
            setStatusBarColor4Drawer(activity, drawer, fakeStatusBar, color, DEFAULT_ALPHA, isTop)
        }

        /**
         * Set the status bar's color for DrawerLayout.
         *
         * DrawLayout must add `android:fitsSystemWindows="true"`
         *
         * @param activity      The activity.
         * @param drawer        The DrawLayout.
         * @param fakeStatusBar The fake status bar view.
         * @param color         The status bar's color.
         * @param alpha         The status bar's alpha which isn't the same as alpha in the color.
         * @param isTop         True to set DrawerLayout at the top layer, false otherwise.
         */
        fun setStatusBarColor4Drawer(@NonNull activity: Activity,
                                     @NonNull drawer: DrawerLayout,
                                     @NonNull fakeStatusBar: View,
                                     @ColorInt color: Int,
                                     @IntRange(from = 0, to = 255) alpha: Int,
                                     isTop: Boolean) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            drawer.fitsSystemWindows = false
            transparentStatusBar(activity)
            setStatusBarColor(fakeStatusBar, color, if (isTop) alpha else 0)
            var i = 0
            val len = drawer.childCount
            while (i < len) {
                drawer.getChildAt(i).fitsSystemWindows = false
                i++
            }
            if (isTop) {
                hideAlphaView(activity)
            } else {
                addStatusBarAlpha(activity, alpha, false)
            }
        }

        /**
         * Set the status bar's alpha for DrawerLayout.
         *
         * DrawLayout must add `android:fitsSystemWindows="true"`
         *
         * @param activity      The activity.
         * @param drawer        drawerLayout
         * @param fakeStatusBar The fake status bar view.
         * @param isTop         True to set DrawerLayout at the top layer, false otherwise.
         */
        fun setStatusBarAlpha4Drawer(@NonNull activity: Activity,
                                     @NonNull drawer: DrawerLayout,
                                     @NonNull fakeStatusBar: View,
                                     isTop: Boolean) {
            setStatusBarAlpha4Drawer(activity, drawer, fakeStatusBar, DEFAULT_ALPHA, isTop)
        }

        /**
         * Set the status bar's alpha for DrawerLayout.
         *
         * DrawLayout must add `android:fitsSystemWindows="true"`
         *
         * @param activity      The activity.
         * @param drawer        drawerLayout
         * @param fakeStatusBar The fake status bar view.
         * @param alpha         The status bar's alpha.
         * @param isTop         True to set DrawerLayout at the top layer, false otherwise.
         */
        fun setStatusBarAlpha4Drawer(@NonNull activity: Activity,
                                     @NonNull drawer: DrawerLayout,
                                     @NonNull fakeStatusBar: View,
                                     @IntRange(from = 0, to = 255) alpha: Int,
                                     isTop: Boolean) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            drawer.fitsSystemWindows = false
            transparentStatusBar(activity)
            setStatusBarAlpha(fakeStatusBar, if (isTop) alpha else 0)
            var i = 0
            val len = drawer.childCount
            while (i < len) {
                drawer.getChildAt(i).fitsSystemWindows = false
                i++
            }
            if (isTop) {
                hideAlphaView(activity)
            } else {
                addStatusBarAlpha(activity, alpha, false)
            }
        }

        private fun addStatusBarColor(activity: Activity,
                                      color: Int,
                                      alpha: Int,
                                      isDecor: Boolean) {
            val parent = if (isDecor)
                activity.window.decorView as ViewGroup
            else
                activity.findViewById<View>(android.R.id.content) as ViewGroup
            val fakeStatusBarView: View? = parent.findViewWithTag(TAG_COLOR)
            if (fakeStatusBarView != null) {
                if (fakeStatusBarView.visibility == View.GONE) {
                    fakeStatusBarView.visibility = View.VISIBLE
                }
                fakeStatusBarView.setBackgroundColor(getStatusBarColor(color, alpha))
            } else {
                parent.addView(createColorStatusBarView(activity, color, alpha))
            }
        }

        private fun addStatusBarAlpha(activity: Activity,
                                      alpha: Int,
                                      isDecor: Boolean) {
            val parent = if (isDecor)
                activity.window.decorView as ViewGroup
            else
                activity.findViewById<View>(android.R.id.content) as ViewGroup
            val fakeStatusBarView: View? = parent.findViewWithTag(TAG_ALPHA)
            if (fakeStatusBarView != null) {
                if (fakeStatusBarView.visibility == View.GONE) {
                    fakeStatusBarView.visibility = View.VISIBLE
                }
                fakeStatusBarView.setBackgroundColor(Color.argb(alpha, 0, 0, 0))
            } else {
                parent.addView(createAlphaStatusBarView(activity, alpha))
            }
        }

        private fun hideColorView(activity: Activity) {
            hideColorView(activity.window)
        }

        private fun hideAlphaView(activity: Activity) {
            hideAlphaView(activity.window)
        }

        private fun hideColorView(window: Window) {
            val decorView = window.decorView as ViewGroup
            val fakeStatusBarView: View? = decorView.findViewWithTag(TAG_COLOR) ?: return
            fakeStatusBarView?.visibility = View.GONE
        }

        private fun hideAlphaView(window: Window) {
            val decorView = window.decorView as ViewGroup
            val fakeStatusBarView: View? = decorView.findViewWithTag(TAG_ALPHA) ?: return
            fakeStatusBarView?.visibility = View.GONE
        }

        private fun showColorView(window: Window) {
            val decorView = window.decorView as ViewGroup
            val fakeStatusBarView: View? = decorView.findViewWithTag(TAG_COLOR) ?: return
            fakeStatusBarView?.visibility = View.VISIBLE
        }

        private fun showAlphaView(window: Window) {
            val decorView = window.decorView as ViewGroup
            val fakeStatusBarView: View? = decorView.findViewWithTag(TAG_ALPHA) ?: return
            fakeStatusBarView?.visibility = View.VISIBLE
        }

        private fun getStatusBarColor(color: Int, alpha: Int): Int {
            if (alpha == 0) return color
            val a = 1 - alpha / 255f
            var red = color shr 16 and 0xff
            var green = color shr 8 and 0xff
            var blue = color and 0xff
            red = (red * a + 0.5).toInt()
            green = (green * a + 0.5).toInt()
            blue = (blue * a + 0.5).toInt()
            return Color.argb(255, red, green, blue)
        }

        private fun createColorStatusBarView(context: Context,
                                             color: Int,
                                             alpha: Int): View {
            val statusBarView = View(context)
            statusBarView.layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT, getStatusBarHeight())
            statusBarView.setBackgroundColor(getStatusBarColor(color, alpha))
            statusBarView.tag = TAG_COLOR
            return statusBarView
        }

        private fun createAlphaStatusBarView(context: Context, alpha: Int): View {
            val statusBarView = View(context)
            statusBarView.layoutParams = LinearLayout.LayoutParams(
                    ViewGroup.LayoutParams.MATCH_PARENT, getStatusBarHeight())
            statusBarView.setBackgroundColor(Color.argb(alpha, 0, 0, 0))
            statusBarView.tag = TAG_ALPHA
            return statusBarView
        }

        private fun transparentStatusBar(activity: Activity) {
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.KITKAT) return
            val window = activity.window
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                val option = View.SYSTEM_UI_FLAG_LAYOUT_STABLE or View.SYSTEM_UI_FLAG_LAYOUT_FULLSCREEN
                window.decorView.systemUiVisibility = option
                window.statusBarColor = Color.TRANSPARENT
            } else {
                window.addFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS)
            }
        }


        fun setWindowStatusBarColor(activity: Activity, color: Int) {
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    val window = activity.window
                    window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                    window.statusBarColor = color
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }

        }

        fun setWindowStatusBarColor(dialog: Dialog, colorResId: Int) {
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    val window = dialog.window
                    window!!.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS)
                    window.statusBarColor = dialog.context.resources.getColor(colorResId)
                }
            } catch (e: Exception) {
                e.printStackTrace()
            }

        }

    }
}